/*
 *  Ray++ - Object-oriented ray tracing library
 *  Copyright (C) 1998-2001 Martin Reinecke and others.
 *  See the AUTHORS file for more information.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  See the README file for more information.
 */

#ifndef RAYPP_GAS_H
#define RAYPP_GAS_H

#include "kernel/kernel.h"

namespace RAYPP {

/** \class GAS volumes/gas.h volumes/gas.h
   Class for rendering inhomogeneous volumes.
   \author Neeharika Adabala
*/
class GAS: public VOLUME
  {
  public:
    enum phase_func {LAMBERT,ANISOTROPIC,RAYLEIGH};

    class DENSITY_FUNCTOR
      {
      public:
        virtual ~DENSITY_FUNCTOR () {}

        virtual float8 Get_Dens (const VECTOR &Loc) const = 0;
      };

  private:
    STRANSFORM Trans;
    COLOUR Opacity;
    phase_func phase;
    HANDLE <DENSITY_FUNCTOR> Densfunc;
    float4 min_step;

    /*!
       Helper function for Integrate_Brightness().

       \a term1 accounts for the attenuation of light due to traveling through 
       the gaseous volume from point of entry into volume to the location of
       interest.

       \a term2 accounts for the attenuation of light from the light sources due
       to travelling throught the gaseous volume to the location of interest.
       It is computed for each light source visible to the location.

       \a term3 causes the brigthness to be a directly proportional to the
       density of the gaseous volume at the location of interest.
    */
    COLOUR Get_Contrib (const VECTOR &Entry, const VECTOR &Interest,
      const VECTOR &dir) const;

    /*!
       Returns the integral over the density between the points
       Lower and Higher (computed by the trapezoid rule).
    */
    float8 Integrate_Density_Function(const VECTOR &Lower,
      const VECTOR &Higher) const;

    /*!
       This is the most important function for rendering gaseous volumes.
       It returns the brigthness/pixel value that should exist at a location
       within a gaseous volume.

       Integration in performed along the direction of the ray.
    */
    COLOUR Integrate_Brightness(const VECTOR &Lower,
      const VECTOR &Higher, const VECTOR &dir) const;

    /*!
       This function takes care of the way light is reflected of the particles
       that constitute the gaseous volume.
    */ 
    float4 Phase_Function(const float4 cosangle) const;

  public:
    GAS ();
    GAS (const COLOUR &opac, phase_func ph,
         const HANDLE<DENSITY_FUNCTOR> &Dens);

    virtual void Init ();

    virtual void Transform (const TRANSFORM &trans);

    virtual float8 Refractive_Index (const SHADING_INFO &)  const
      { return 1; }

    /*!
       Computes the attentuation of ray intensity due to passing through the
       gaseous volume.
    */
    virtual COLOUR Calc_new_Importance (const RAY &Ray) const;

    /*!
       Gives the modification of colour of light traveling through the gaseous
       volume.
    */
    virtual COLOUR Calc_Modified_Colour
      (const RAY &Ray, const COLOUR &Ingoing) const;

    /*!
       Gives the attentuation of light due to travelling through the
       gaseous volume
    */
    virtual COLOUR Get_Attenuated_Light
      (const RAY &Ray, const COLOUR &Ingoing) const;

    /*!
       Set the opacity of the gas.
    */
    void Set_Opacity (const COLOUR &new_opacity);
    /*!
       Set the phase function.
    */
    void Set_Phase (phase_func pf);
    /*!
       Set the density function.
    */
    void Set_Density (const HANDLE<DENSITY_FUNCTOR> &Dens);
  };

} // namespace RAYPP;

#endif
