
#====================================================================#
# svg/core.rb
#====================================================================#

#====================================================================#
# SVG Class
class SVG

  #==================================================================#
  # Include Module

  include SVG::ArrayMixin

  #==================================================================#
  # Initialize Method

  def initialize(width, height, view_box = nil)
    @width    = width
    @height   = height
    @x        = nil
    @y        = nil
    @view_box = view_box
    @title    = nil
    @desc     = nil

    @elements = []
    @styles   = []
  end

  #==================================================================#
  # Accessor

  attr_reader   :elements, :styles
  attr_accessor :width, :height, :x, :y, :view_box, :title, :desc

  #==================================================================#
  # Instance Methods

  def array
    return @elements
  end
  private :array

  def define_style(class_name, style)
    @styles << DefineStyle.new(class_name, style)
  end

  def to_s
    text  = %|<?xml version="1.0" standalone="no"?>\n|
    text << %|<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 20000629//EN" "http://www.w3.org/TR/2000/WD-SVG-20000629/DTD/svg-20000629.dtd">\n|
    text << %|<svg width="#{@width}" height="#{@height}"|
    text << %| viewBox="#{@view_box}"| if @view_box
    text << %|>\n|

    unless @styles.empty?
      text << %|<defs>\n|
      text << %|<style type="text/css"><![CDATA[\n|
      text << @styles.collect { |define| define.to_s + "\n" }.join
      text << %|]]></style>\n|
      text << %|</defs>\n|
    end

    text << %|<title>#{@title}</title>\n| if @title
    text << %|<desc>#{@desc}</desc>\n|    if @desc
    text << @elements.collect { |element| element.to_s + "\n" }.join
    text << %|</svg>\n|
    return text
  end

  def compress
    return Deflate.deflate(self.to_s, Deflate::BEST_COMPRESSION)
  end

  def mime_type
    return 'image/svg-xml'
  end

  #==================================================================#
  # DefineStyle Class
  class DefineStyle

    #================================================================#
    # Initialize Method

    def initialize(class_name, style)
      @class_name = class_name
      @style      = style
    end

    #================================================================#
    # Accessor

    attr_accessor :class_name, :style

    #================================================================#
    # Instance Methods

    def to_s
      return "#{@class_name} { #{@style} }"
    end

  end

end

#====================================================================#
#====================================================================#
