''' 
threeDS.py - Python Module for loading and displaying 3DStudio format modules
Copyright (C) 2000-2001 Jason Petrone <jp@demonseed.net>

The 3DS format is very robust, and much of it is not implemented in this
module.  Currently, 3DS geometry, materials(including textures), and undirected
lights are supported.  Keyframing, directed lighting, and everything else is
currently unimplemented.

'''

__version__ = '0.3.1'
__author__ = 'Jason Petrone <jp@demonseed.net>'

DEBUG = 0

from struct import *
import os, cStringIO, math, copy

from OpenGL.GL import *
from Numeric import *

dispListCnt = 0

# generic chunks used throughout
RGB1           = 0x0010  # 3 floats of RGB 
RGB2           = 0x0011  # 3 bytes of RGB 
IPERCENT       = 0x0030  # integer percentage
FPERCENT       = 0x0031  # float percentage

OBJMESH        = 0x3D3D
SMOOTH_GROUP   = 0x4150

# geometry chunks
OBJBLOCK       = 0x4000
TRIMESH        = 0x4100
VERTLIST       = 0x4110
FACELIST       = 0x4120
FACEMAT        = 0x4130
MAPLIST        = 0x4140
TRMATRIX       = 0x4160
LIGHT          = 0x4600
SPOTLIGHT      = 0x4610
CAMERA         = 0x4700
MAIN           = 0x4D4D

# material chunks
AMBIENT        = 0xA010
DIFFUSE        = 0xA020
SPECULAR       = 0xA030
SHININESS      = 0xA040
TRANSPARENCY   = 0xA050
DOUBLESIDED    = 0xA081
TEXTURE        = 0xA200
MATNAME        = 0xA000
MAT_MAPNAME    = 0xA300
MATMAPTILING   = 0xA351
TEXBLUR        = 0xA353
MAT_MAP_USCALE = 0xA354
MAT_MAP_VSCALE = 0xA356
MATERIAL       = 0xAFFF

KEYFRAMER      = 0xB000

# global textures(dont want to load them more than once!)
tex_data = {}

def readStr(f):
  txt = ""
  [c] = unpack('c', f.read(1) )
  while c != '\0':
    txt = txt + c
    [c] = unpack('c', f.read(1) )
  return txt

def readHdr(f):
  [hdr] = unpack( '<H', f.read(2) )
  [size] = unpack( '<I', f.read(4) )
  return (hdr, size)

class TexImage:
  data = None
  sizex = 0
  sizey = 0

class Texture:
  def __init__(self, f, myLen):
    self.imgfile = None
    self.texId = -1
    while f.tell() < myLen:
      [hdr, size] = readHdr(f)
      if hdr == MAT_MAPNAME:
        global tex_data
        self.imgfile = readStr(f)
        if not tex_data.has_key(self.imgfile):
          try:
            from PIL import Image
            img = Image.open(self.imgfile)
          except IOError, e:
            print 'Cannot load '+self.imgfile+': '+ str(e)
            self.imgfile = None
            continue
              
          img = img.convert('RGBA', colors=32)
          # texture size must be aligned by 2^n
          w = 0; h = 0
          [w, h] = img.size
          power = 2
          while w > power:
            power = power * 2
          w = power
          power = 2
          while h > power:
            power = power * 2
          h = power
          img = img.resize((w, h))

          # now create storage class
          teximg = TexImage()
          teximg.sizex, teximg.sizey = w, h
          teximg.data = img.tostring('raw', 'RGBA', 0, -1)
          # outfile = self.imgfile + '.png'
          # img.save(outfile, "PNG")
          tex_data[self.imgfile] = teximg

      elif hdr == IPERCENT:
        [self.percent] = unpack( '<h', f.read(2) )
      elif hdr == FPERCENT:
        [self.percent] = unpack( '<f', f.read(4) )
      elif hdr == MATMAPTILING:
        # not sure what to do with this
        [self.tiling] = unpack( '<h', f.read(2) )
      elif hdr == TEXBLUR:
        [self.blur] = unpack( '<f', f.read(4) )
      else:
        if DEBUG: print 'tex hdr: 0x'+hex(hdr)[2:].upper()
        f.read(size - 6)

    return None

  def load(self):
    global tex_data
    self.texId = glGenTextures(1)
    glBindTexture(GL_TEXTURE_2D, self.texId)
    if self.imgfile is None: return
    teximg = tex_data[self.imgfile]
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1)
    # glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT)
    # glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT)
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP)
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP)
    # glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT)
    # glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT)
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST)
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST)
    # glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL)
    # glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_BLEND)
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE)
    # glTexImage2D(GL_TEXTURE_2D, 0, 4, teximg.sizex, teximg.sizey, 0,
    #                GL_RGBA, GL_UNSIGNED_BYTE, teximg.data)
    # glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR)
    # glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR)

    glTexImage2D(GL_TEXTURE_2D, 0, 3, teximg.sizex, teximg.sizey, 0,
                   GL_RGBA, GL_UNSIGNED_BYTE, teximg.data)

  def set(self):
    if DEBUG: print 'setting texture ' + self.imgfile
    if self.texId == -1: self.load()
    glBindTexture(GL_TEXTURE_2D, self.texId)
    
class Material:
  "All shading values are stored as floats"
  def __init__(self, f = None, myLen = None):

    self.texture = None
    self.doublesided = None
    if f == None:
      # set defaults
      self.ambient = [1.0, 1.0, 1.0, 1.0]
      self.diffuse = [1.0, 1.0, 1.0, 1.0]
      self.specular = [1.0, 1.0, 1.0, 1.0]
      return

    self.ambient = []
    self.diffuse = []
    self.specular = []
    self.shininess = 0
    self.transparency = 0
    self.name = ""
    while f.tell() < myLen:
      [hdr, size] = readHdr(f)
      if hdr == MATNAME:
        self.name = readStr(f)
      elif hdr == AMBIENT:
        self.ambient = self.readShading(f, f.tell() + size - 6)
        self.ambient.append(0.6)
      elif hdr == DIFFUSE:
        self.diffuse = self.readShading(f, f.tell() + size - 6)
        self.diffuse.append(0.8)
      elif hdr == SPECULAR:
        self.specular = self.readShading(f, f.tell() + size - 6)
        self.specular.append(0.5)
      elif hdr == SHININESS:
        [hdr, size] = readHdr(f)
        [self.shininess] = unpack('<H', f.read(2))
      elif hdr == TRANSPARENCY:
        [hdr, size] = readHdr(f)
        [self.transparency] = unpack('<H', f.read(2))
      elif hdr == TEXTURE:
        self.texture = Texture(f, f.tell() + size - 6)
        if DEBUG: print self.name + ': ' + self.texture.imgfile
      elif hdr == DOUBLESIDED:
        if DEBUG: print 'doublesided material: '+self.name
        self.doublesided = 1
      else:
        # print 'mat hdr: 0x'+hex(hdr)[2:].upper()
        f.read(size - 6)
       
  def readShading(self, f, myLen):
    while f.tell() < myLen:
      [hdr, size] = readHdr(f)
      if hdr == RGB1:
        return list(unpack('<3f', f.read(12)))
      elif hdr == RGB2:
        [r, g, b] = unpack( '3B', f.read(3) )
        return [ float(r)/256.0, float(g)/256.0, float(b)/256.0 ]
      else:
        f.read(size - 6)

  def set(self):
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, self.ambient)
    glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, self.diffuse)
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, self.specular)
    # glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, self.shininess)
    # glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, (self.shininess/100.0)*128)
    if self.texture: self.texture.set()


class Light:
  color = [0, 0, 0]
  pos = [0, 0, 0]
  target = None

  def __init__(self, name, f, myLen):
    self.pos = unpack('<3f', f.read(12))
    while f.tell() < myLen:
      [hdr, size] = readHdr(f)
      if hdr == RGB1:
        self.color = unpack( '<3f', f.read(12) )
      elif hdr == RGB2:
        [r, g, b] = unpack( '3B', f.read(3) )
        self.color = [ float(r)/256.0, float(g)/256.0, float(b)/256.0 ]
      elif hdr == SPOTLIGHT:
        self.target = [0, 0, 0]
        self.target = unpack('<3f', f.read(12))
        self.hotspot = unpack('<1f', f.read(4))
        self.falloff = unpack('<1f', f.read(4))
      else:
        f.read(size - 6)

class TriObj:
  def __init__(self, name, f, myLen):
    self.name = name
    self.vertices = []
    self.normals = []
    self.maps = []
    self.traMatrix = []
    self.matName = ""
    self.faceMats = []
    self.faces = []
    self.displayList = None
    self.material = None
    self.materialList = None
    self.load(f, myLen)

  def load(self, f, end):
        while f.tell() < end:
          [hdr, size] = readHdr(f)
          if hdr == VERTLIST:
            [num] = unpack( '<h', f.read(2) )
            c = 0
            for v in range(0, num):
              self.vertices.append( unpack('<3f', f.read(12)) ) 
              c = c + 1
          elif hdr == FACELIST:
            [num] = unpack( '<h', f.read(2) )
            for v in range(0, num):
              self.faces.append( unpack('<4h', f.read(8)) )
          elif hdr == MAPLIST:
            [num] = unpack( '<h', f.read(2) )
            for v in range(0, num):
              self.maps.append( unpack('<ff0f', f.read(8)) )
          elif hdr == TRMATRIX:
            self.traMatrix = unpack( '<12f0f', f.read(48) )
          elif hdr == FACEMAT:
            self.matName = readStr(f)                
            [num] = unpack( '<h', f.read(2) )
            for v in range(0, num):
              [mat] = unpack( '<h', f.read(2) )
              self.faceMats.append(mat)
          else:
            # print 'hdr: '+hex(hdr)
            f.read(size - 6)


  def draw(self):
    glMatrixMode(GL_MODELVIEW)
    glPushMatrix()
    #glRotatef(145, 0, 1, 0)
    if self.material: self.material.set()
    glBegin(GL_TRIANGLES)
    for i in range(len(self.faces)):
      face = self.faces[i][:3]
      if len(face) != 3: 
        if DEBUG: print "BAD FACE " + str(len(face))
        continue
      for j in range(len(face)):
        vertex = face[j]
        try: v = self.vertices[vertex]
        except: 
          if DEBUG:
            print 'bad vertex: ' + str(vertex) + '/' + str(len(self.vertices))
          continue
        n = self.normals[vertex]
        glNormal3f(n[0], n[1], n[2])
        if self.material.texture:
          m = self.maps[vertex]
          glTexCoord2f(m[0], m[1])
        glVertex3f(v[0], v[2], v[1])
    glEnd()
    glPopMatrix()

  def fastDraw(self):
    if self.displayList is None:
      if DEBUG: print 'creating list: ' + self.name
      self.displayList = glGenLists(1)
      glNewList(self.displayList, GL_COMPILE)
      self.draw()
      glEndList()
      if DEBUG: print 'done'
    glCallList(self.displayList)

  def calcNormals(self):
    self.normals = []
    for i in xrange(len(self.vertices)): self.normals.append([])
    for i in xrange(len(self.faces)):
      face = self.faces[i][:3]
      v0 = self.vertices[ face[0] ]
      v1 = self.vertices[ face[1] ]
      v2 = self.vertices[ face[2] ]
      a = [ v0[0] - v1[0], v0[1] - v1[1], v0[2] - v1[2] ]
      b = [ v1[0] - v2[0], v1[1] - v2[1], v1[2] - v2[2] ]
      c = [ a[1]*b[2] - b[1]*a[2], 
            b[0]*a[2] - a[0]*b[2],
            a[0]*b[1] - b[0]*b[1] ]

      l = math.sqrt(c[0]*c[0] + c[1]*c[1] + c[2]*c[2])
      if l == 0:
        c = [0, 0, 0]
      #   self.normals.append([0, 0, 0])
      #   continue
      # l = l * .95
      else:
        c[0] = c[0] / l
        c[1] = c[1] / l
        c[2] = c[2] / l
        # if c[2] < 0:
        #   c[0] *= -1
        #   c[1] *= -1
        #   c[2] *= -1
      # self.normals.append(c)
      # for vertex in self.faces[i]:
      for vertex in face:
        self.normals[vertex].append(c)
    if DEBUG: print 'squashing '+str(len(self.normals))+' normals'
    for i in xrange(len(self.normals)):
      size = len(self.normals[i])
      if size == 0: continue
      # print 'size: ' + str(size)
      norm = sum(self.normals[i])
      l = sqrt(sum(multiply(norm, norm)))
      if l == 0: norm = [0, 0, 0]
      else: divide(norm, l)
      self.normals[i] = list(norm)

class Scene:
  materials = {}
  objects = []
  lights = []
  lists = []

  def __init__(self, filename):
    (path, filename) = os.path.split(filename)
    if path != '':
      os.chdir(path)
    f = open(filename, "rb")
    if DEBUG: print 'read ' + filename + '.'
    if f is None:
      print "error loading " + str(filename) + "."
      f.close()
      return

    # read file header
    [hdr, totLen] = readHdr(f)
    if hdr != MAIN:
      print str(filename) + ' is not a 3ds file'
      f.close()
      return
    
    end = f.tell() + totLen
    while f.tell() < totLen:
      [hdr, size] = readHdr(f)
      if hdr == OBJMESH:
        self.readMesh(f, f.tell() + size - 6)  
      # elif hdr == KEYFRAMER:
      #   print 'KEY FRAME ENCOUNTERED!'
      #   f.read(size - 6)   # skip to next chunk
      else:
        # print 'UNKNOWN STRUCTURE ENCOUNTERED!'
        f.read(size - 6)   # skip to next chunk
      if DEBUG: print 'read mesh.'
    # print str(len(f.read())) + ' bytes unread!'
    f.close()
    self.calcNormals()
    if DEBUG: print 'setting materials...'
    self.setMaterials()

  def readMesh(self, f, myLen):
    while f.tell() < myLen:
      [hdr, size] = readHdr(f)
      end = f.tell() + size - 6
      if hdr == MATERIAL:
        m = Material(f, end)
        self.materials[m.name] = m  # new material
      elif hdr == OBJBLOCK:
        self.readObj(f, end)
      else:
        f.read(size - 6)                         # skip to next block

  def readObj(self, f, myLen):
    while f.tell() < myLen:
      name = readStr(f)
      [hdr, size] = readHdr(f)
      end = f.tell() + size - 6
      if hdr == TRIMESH:
        self.objects.append(TriObj(name, f, end))      # new object
      elif hdr == LIGHT:
        self.lights.append(Light(name, f, end))
      else: 
        # print 'OBJ CHUNK: ' + str(hex(hdr))
        f.read(size - 6)

  def calcNormals(self):
    for m in self.objects:
      if DEBUG: print 'calculating normals for '+m.name+'...'
      m.calcNormals()

  def setMaterials(self):
    for obj in self.objects:
      try:
        obj.material = self.materials[obj.matName]
        if DEBUG: print obj.name + ' -> ' + obj.matName
      except Exception, e:
        # use default material
        if DEBUG: print ' using default material!!!'
        obj.material = Material()
      
  def objNames(self):
    objs = []
    for obj in self.objects:
      objs.append(obj.name)
    return objs

  def getIndex(self, objName):
    for i in range(0, len(self.objects)):
      obj = self.objects[i]
      if obj.name == objName:
        return i

  def draw(self, index = None):
    if index is None:
      if self.lists == []:
        for i in range(0, len(self.objects)): 
          if DEBUG: print 'building display lists' + str(i)
          self.objects[i].fastDraw()
          # self.objects[i].draw()
          self.lists.append(self.objects[i].displayList)
      glCallLists(self.lists)
    else:
      self.objects[index].fastDraw()

  def lighting(self):
    if len(self.lights) == 0: return None
    glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE)
    cnt = 0
    for light in self.lights[:8]:
      ambient = [light.color[0] * .1, light.color[1] * .1, light.color[1] * .1]
      diffuse = [light.color[0] * .6, light.color[1] * .6, light.color[1] * .6]
      specular = [light.color[0] * .1, light.color[1] * .1, light.color[1] * .1]
      glLightf(GL_LIGHT0 + cnt , GL_AMBIENT, ambient)
      glLightf(GL_LIGHT0 + cnt , GL_DIFFUSE, diffuse)
      glLightf(GL_LIGHT0 + cnt , GL_SPECULAR, specular)
      glLightf(GL_LIGHT0, GL_POSITION, light.pos)
      cnt = cnt + 1 
    return 'true'
