/*
 *  A libESMTP Example Application.
 *  Copyright (C) 2001  Brian Stafford <brian@stafford.uklinux.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published
 *  by the Free Software Foundation; either version 2 of the License,
 *  or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* This program accepts a single file argument followed by a list of
   recipients.  The file is mailed to each of the recipients.

   Error checking is minimal to non-existent, this is just a quick
   and dirty program to give a feel for using libESMTP.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>

#include <libesmtp.h>

struct option longopts[] =
  {
    { "help", no_argument, NULL, '?', }, 
    { "host", required_argument, NULL, 'h', }, 
    { "monitor", no_argument, NULL, 'm', }, 
    { "notify", required_argument, NULL, 'n', }, 
    { "mdn", no_argument, NULL, 'd', }, 
    { "subject", required_argument, NULL, 's', }, 
    { "reverse-path", required_argument, NULL, 'f', }, 
    { NULL, },
  };

char *readfp_cb (char **buf, int *len, void *arg);
void monitor_cb (const char *buf, int buflen, int writing, void *arg);
void print_recipient_status (smtp_recipient_t recipient,
			     const char *mailbox, void *arg);
void usage (void);

int
main (int argc, char **argv)
{
  smtp_session_t session;
  smtp_message_t message;
  smtp_recipient_t recipient;
  const smtp_status_t *status;
  char *host = NULL;
  char *from = NULL;
  char *subject = NULL;
  char *file;
  FILE *fp;
  int c;
  enum notify_flags notify = Notify_NOTSET;

  /* This program sends only one message at a time.  Create an SMTP
     session and add a message to it. */
  session = smtp_create_session ();
  message = smtp_add_message (session);

  while ((c = getopt_long (argc, argv, "dmh:f:s:n:",
  			   longopts, NULL)) != EOF)
    switch (c)
      {
      case 'h':
        host = optarg;
        break;

      case 'f':
        from = optarg;
        break;

      case 's':
        subject = optarg;
        break;

      case 'm':
        smtp_set_monitorcb (session, monitor_cb, stdout, 1);
        break;

      case 'n':
        if (strcmp (optarg, "success") == 0)
          notify |= Notify_SUCCESS;
        else if (strcmp (optarg, "failure") == 0)
          notify |= Notify_FAILURE;
        else if (strcmp (optarg, "delay") == 0)
          notify |= Notify_DELAY;
        else if (strcmp (optarg, "never") == 0)
          notify = Notify_NEVER;
        break;

      case 'd':
        /* Request MDN sent to the same address as the reverse path */
        smtp_set_header (message, "Disposition-Notification-To", NULL, NULL);
        break;

      default:
        usage ();
        exit (2);
      }

  /* At least two more arguments are needed.
   */
  if (optind > argc - 2)
    {
      usage ();
      exit (2);
    }

  /* Set the host running the SMTP server.  LibESMTP has a default port
     number of 587, however this is not widely deployed so the port
     is specified as 25 along with the default MTA host. */
  smtp_set_server (session, host ? host : "localhost:25");

  /* Set the reverse path for the mail envelope.  (NULL is ok)
   */
  smtp_set_reverse_path (message, from);

  /* RFC 822 doesn't actually require a message to have a Message-Id:
     header.  libESMTP can generate one.  This is how to request the
     library to do this */
  smtp_set_header (message, "Message-Id", NULL);

  /* Set the Subject: header.  For no reason, we want the supplied subject
     to override any subject line in the message headers. */
  if (subject != NULL)
    {
      smtp_set_header (message, "Subject", subject);
      smtp_set_header_option (message, "Subject", Hdr_OVERRIDE, 1);
    }
  
  /* Open the message file and set the callback to read it.
   */
  file = argv[optind++];
  if (strcmp (file, "-") == 0)
    fp = stdin;
  else if ((fp = fopen (file, "r")) == NULL)
    {
      fprintf (stderr, "can't open %s: %s\n", file, strerror (errno));
      exit (1);
    }
  smtp_set_messagecb (message, readfp_cb, fp);

  /* Add remaining program arguments as message recipients.
   */
  while (optind < argc)
    {
      recipient = smtp_add_recipient (message, argv[optind++]);

      /* Recipient options set here */
      if (notify != Notify_NOTSET)
        smtp_dsn_set_notify (recipient, notify);
    }

  /* Initiate a connection to the SMTP server and transfer the
     message. */
  smtp_start_session (session);

  /* Report on the success or otherwise of the mail transfer.
   */
  status = smtp_message_transfer_status (message);
  printf ("%d %s", status->code, status->text);
  smtp_enumerate_recipients (message, print_recipient_status, NULL);

  /* Free resources consumed by the program.
   */
  smtp_destroy_session (session);
  fclose (fp);
  exit (0);
}

/* Callback to prnt the recipient status */
void
print_recipient_status (smtp_recipient_t recipient,
			const char *mailbox, void *arg)
{
  const smtp_status_t *status;

  status = smtp_recipient_status (recipient);
  printf ("%s: %d %s", mailbox, status->code, status->text);
}

/* Callback function to read the message from a file.  The file MUST be
   formatted according to RFC 822 and lines MUST be terminated with the
   canonical CRLF sequence.

   To make this example program into a useful one, a more sophisticated
   callback capable of translating the Un*x \n tro CRLF would be needed.

   Until libESMTP implements full header processing, the file must contain
   all the required message headers.
 */
#define BUFLEN	8192

char *
readfp_cb (char **buf, int *len, void *arg)
{
  if (*buf == NULL)
    *buf = malloc (BUFLEN);

  if (len == NULL)
    {
      rewind ((FILE *) arg);
      return NULL;
    }

  *len = fread (*buf, 1, BUFLEN, (FILE *) arg);
  return *buf;
}

void
monitor_cb (const char *buf, int buflen, int writing, void *arg)
{
  FILE *fp = arg;

  if (writing == SMTP_CB_HEADERS)
    {
      fputs ("H: ", fp);
      fwrite (buf, 1, buflen, fp);
      return;
    }

 fputs (writing ? "C >>>>\n" : "S <<<<\n", fp);
 fwrite (buf, 1, buflen, fp);
 if (buf[buflen - 1] != '\n')
   putc ('\n', fp);
}

void
usage (void)
{
  fputs ("usage: mail-file [options] file mailbox [mailbox ...]\n"
         "\t-h,--host=hostname[:port] -- set SMTP server\n"
         "\t-f,--reverse-path=mailbox -- set reverse path\n"
         "\t-s,--subject=text -- set subject of the message\n"
         "\t-n,--notify=success|failure|delay|never -- request DSN\n"
         "\t-d,--mdn -- request MDN\n"
         "\t-m,--monitor -- watch the protocol session with the server\n"
         "\t--help -- this message\n",
         stderr);
}

