
/*
 * libEtPan! -- a mail stuff library
 *
 * chash - Implements generic hash tables.
 *
 * Copyright (c) 1999-2000, Gal Roualland <gael.roualland@iname.com>
 * interface changes - 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef CHASH_H
#define CHASH_H

#include "mail.h"

typedef struct {
  char * data;
  int len;
} chashdatum;

struct chash {
  int size, count, copyvalue, copykey;
  struct chashcell ** cells; 
};

typedef struct chash chash;

struct chashcell {
  unsigned int func;
  chashdatum key;
  chashdatum value;
  struct chashcell * next;
};

typedef struct chashcell chashiter;

#define CHASH_COPYNONE    0
#define CHASH_COPYKEY     1
#define CHASH_COPYVALUE   2
#define CHASH_COPYALL     (CHASH_COPYKEY | CHASH_COPYVALUE)

#define CHASH_DEFAULTSIZE 13
  
/* Allocates a new (empty) hash using this initial size and the given flags,
   specifying which data should be copied in the hash.
    CHASH_COPYNONE  : Keys/Values are not copied.
    CHASH_COPYKEY   : Keys are dupped and freed as needed in the hash.
    CHASH_COPYVALUE : Values are dupped and freed as needed in the hash.
    CHASH_COPYALL   : Both keys and values are dupped in the hash.
 */
chash * chash_new(int size, int flags);

/* Frees a hash */
void chash_free(chash * hash);

/* Adds an entry in the hash table.
   Length can be 0 if key/value are strings.
   If an entry already exists for this key, it is replaced, and its value
   is returned. Otherwise, the data pointer will be NULL and the length
   field be set to TRUE or FALSe to indicate success or failure. */
int chash_set(chash * hash,
	      chashdatum * key,
	      chashdatum * value,
	      chashdatum * oldvalue);

/* Retrieves the data associated to the key if it is found in the hash table.
   The data pointer and the length will be NULL if not found*/
int chash_get(chash * hash,
	      chashdatum * key, chashdatum * result);

/* Removes the entry associated to this key if it is found in the hash table,
   and returns its contents if not dupped (otherwise, pointer will be NULL
   and len TRUE). If entry is not found both pointer and len will be NULL. */
int chash_delete(chash * hash,
		 chashdatum * key,
		 chashdatum * oldvalue);

/* Resizes the hash table to the passed size. */
int chash_resize(chash * hash, int size);

/* Returns an iterator to the first non-empty entry of the hash table */
chashiter * chash_begin(chash * hash);

/* Returns the next non-empty entry of the hash table */
chashiter * chash_next(chash * hash, chashiter * iter);

/* Some of the following routines can be implemented as macros to
   be faster. If you don't want it, define NO_MACROS */
#ifdef NO_MACROS
/* Returns the size of the hash table */
int          chash_size(chash * hash);

/* Returns the number of entries in the hash table */
int          chash_count(chash * hash);

/* Returns the key part of the entry pointed by the iterator */
void chash_key(chashiter * iter, chashdatum * result);

/* Returns the value part of the entry pointed by the iterator */
void chash_value(chashiter iter, chashdatum * result);

#else
#define      chash_size(hash)        ((hash)->size)
#define      chash_count(hash)       ((hash)->count)
#define      chash_key(iter, result)         ((* (result)) = (iter)->key)
#define      chash_value(iter, result)       ((* (result)) (iter)->value)
#endif

#if 0
/* String hash encapsulation
   The following defines are designed for use with strings only.
   No length fields are necessary/used. */

typedef chash cstrhash;
typedef chashiter cstrhashiter;

/* Creates a suitable fully copied string to string hash.
   If you need another memory model, use chash_new with your choices. */
#define      cstrhash_new()               (chash_new(CHASH_DEFAULTSIZE, CHASH_COPYALL))

/* Simple mappings */
#define      cstrhash_free(h)             (chash_free(h))
#define      cstrhash_resize(h, s)        (chash_resize(h, s))
#define      cstrhash_begin(i)            (chash_begin(i))
#define      cstrhash_next(i, j)          (chash_next(i,j))
#define      cstrhash_size(h)             (chash_size(h))
#define      cstrhash_count(h)            (chash_count(h))

/* Only get the data part. Length can be (re)computed with strlen. */
#define      cstrhash_key(i)              ((chash_key(i)).data)
#define      cstrhash_value(i)            ((chash_value(i)).data)
#define      cstrhash_set(h, k, v)        ((chash_set(h, k, 0, v, 0)).data)
#define      cstrhash_get(h, k)           ((chash_get(h, k, 0)).data)
#define      cstrhash_delete(h, k)        ((chash_delete(h, k, 0)).data)
#endif

#endif
