/*
 * libEtPan! -- a mail stuff library
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: mailimap_types.h,v 1.12 2002/06/26 21:15:10 hoa Exp $
 */


/*
  IMAP4rev1 grammar

   address         = "(" addr-name SP addr-adl SP addr-mailbox SP
                     addr-host ")"

   addr-adl        = nstring
                       ; Holds route from [RFC-822] route-addr if
                       ; non-NIL

   addr-host       = nstring
                       ; NIL indicates [RFC-822] group syntax.
                       ; Otherwise, holds [RFC-822] domain name

   addr-mailbox    = nstring
                       ; NIL indicates end of [RFC-822] group; if
                       ; non-NIL and addr-host is NIL, holds
                       ; [RFC-822] group name.
                       ; Otherwise, holds [RFC-822] local-part
                       ; after removing [RFC-822] quoting



   addr-name       = nstring
                       ; If non-NIL, holds phrase from [RFC-822]
                       ; mailbox after removing [RFC-822] quoting

   append          = "APPEND" SP mailbox [SP flag-list] [SP date-time] SP
                     literal

   astring         = 1*ASTRING-CHAR / string

   ASTRING-CHAR   = ATOM-CHAR / resp-specials

   atom            = 1*ATOM-CHAR

   ATOM-CHAR       = <any CHAR except atom-specials>

   atom-specials   = "(" / ")" / "{" / SP / CTL / list-wildcards /
                     quoted-specials / resp-specials

   authenticate    = "AUTHENTICATE" SP auth-type *(CRLF base64)

   auth-type       = atom
                       ; Defined by [SASL]

   base64          = *(4base64-char) [base64-terminal]

   base64-char     = ALPHA / DIGIT / "+" / "/"
                       ; Case-sensitive

   base64-terminal = (2base64-char "==") / (3base64-char "=")

   body            = "(" (body-type-1part / body-type-mpart) ")"

   body-extension  = nstring / number /
                      "(" body-extension *(SP body-extension) ")"
                       ; Future expansion.  Client implementations
                       ; MUST accept body-extension fields.  Server
                       ; implementations MUST NOT generate
                       ; body-extension fields except as defined by
                       ; future standard or standards-track
                       ; revisions of this specification.

   body-ext-1part  = body-fld-md5 [SP body-fld-dsp [SP body-fld-lang
                     *(SP body-extension)]]
                       ; MUST NOT be returned on non-extensible
                       ; "BODY" fetch


   body-ext-mpart  = body-fld-param [SP body-fld-dsp [SP body-fld-lang
                     *(SP body-extension)]]
                       ; MUST NOT be returned on non-extensible
                       ; "BODY" fetch

   body-fields     = body-fld-param SP body-fld-id SP body-fld-desc SP
                     body-fld-enc SP body-fld-octets

   body-fld-desc   = nstring

   body-fld-dsp    = "(" string SP body-fld-param ")" / nil

   body-fld-enc    = (DQUOTE ("7BIT" / "8BIT" / "BINARY" / "BASE64"/
                     "QUOTED-PRINTABLE") DQUOTE) / string

   body-fld-id     = nstring

   body-fld-lang   = nstring / "(" string *(SP string) ")"

   body-fld-lines  = number

   body-fld-md5    = nstring

   body-fld-octets = number

   body-fld-param  = "(" string SP string *(SP string SP string) ")" / nil

   body-type-1part = (body-type-basic / body-type-msg / body-type-text)
                     [SP body-ext-1part]

   body-type-basic = media-basic SP body-fields
                       ; MESSAGE subtype MUST NOT be "RFC822"

   body-type-mpart = 1*body SP media-subtype
                     [SP body-ext-mpart]

   body-type-msg   = media-message SP body-fields SP envelope
                     SP body SP body-fld-lines

   body-type-text  = media-text SP body-fields SP body-fld-lines

   capability      = ("AUTH=" auth-type) / atom
                       ; New capabilities MUST begin with "X" or be
                       ; registered with IANA as standard or
                       ; standards-track


   capability-data = "CAPABILITY" *(SP capability) SP "IMAP4rev1"
                     *(SP capability)
                       ; IMAP4rev1 servers which offer RFC 1730
                       ; compatibility MUST list "IMAP4" as the first
                       ; capability.

   CHAR8           = %x01-ff
                       ; any OCTET except NUL, %x00

   command         = tag SP (command-any / command-auth / command-nonauth /
                     command-select) CRLF
                       ; Modal based on state

   command-any     = "CAPABILITY" / "LOGOUT" / "NOOP" / x-command
                       ; Valid in all states

   command-auth    = append / create / delete / examine / list / lsub /
                     rename / select / status / subscribe / unsubscribe
                       ; Valid only in Authenticated or Selected state

   command-nonauth = login / authenticate
                       ; Valid only when in Not Authenticated state

   command-select  = "CHECK" / "CLOSE" / "EXPUNGE" / copy / fetch / store /
                     uid / search
                       ; Valid only when in Selected state

   continue-req    = "+" SP (resp-text / base64) CRLF

   copy            = "COPY" SP set SP mailbox

   create          = "CREATE" SP mailbox
                       ; Use of INBOX gives a NO error

   date            = date-text / DQUOTE date-text DQUOTE

   date-day        = 1*2DIGIT
                       ; Day of month

   date-day-fixed  = (SP DIGIT) / 2DIGIT
                       ; Fixed-format version of date-day

   date-month      = "Jan" / "Feb" / "Mar" / "Apr" / "May" / "Jun" /
                     "Jul" / "Aug" / "Sep" / "Oct" / "Nov" / "Dec"

   date-text       = date-day "-" date-month "-" date-year

   date-year       = 4DIGIT

   date-time       = DQUOTE date-day-fixed "-" date-month "-" date-year
                     SP time SP zone DQUOTE

   delete          = "DELETE" SP mailbox
                       ; Use of INBOX gives a NO error

   digit-nz        = %x31-39
                       ; 1-9

   envelope        = "(" env-date SP env-subject SP env-from SP env-sender SP
                     env-reply-to SP env-to SP env-cc SP env-bcc SP
                     env-in-reply-to SP env-message-id ")"

   env-bcc         = "(" 1*address ")" / nil

   env-cc          = "(" 1*address ")" / nil

   env-date        = nstring

   env-from        = "(" 1*address ")" / nil

   env-in-reply-to = nstring

   env-message-id  = nstring

   env-reply-to    = "(" 1*address ")" / nil

   env-sender      = "(" 1*address ")" / nil

   env-subject     = nstring

   env-to          = "(" 1*address ")" / nil

   examine         = "EXAMINE" SP mailbox

   fetch           = "FETCH" SP set SP ("ALL" / "FULL" / "FAST" / fetch-att /
                     "(" fetch-att *(SP fetch-att) ")")

   fetch-att       = "ENVELOPE" / "FLAGS" / "INTERNALDATE" /
                     "RFC822" [".HEADER" / ".SIZE" / ".TEXT"] /
                     "BODY" ["STRUCTURE"] / "UID" /
                     "BODY" [".PEEK"] section ["<" number "." nz-number ">"]

   flag            = "\Answered" / "\Flagged" / "\Deleted" /
                     "\Seen" / "\Draft" / flag-keyword / flag-extension
                       ; Does not include "\Recent"

   flag-extension  = "\" atom
                       ; Future expansion.  Client implementations
                       ; MUST accept flag-extension flags.  Server
                       ; implementations MUST NOT generate
                       ; flag-extension flags except as defined by
                       ; future standard or standards-track
                       ; revisions of this specification.

   flag-fetch      = flag / "\Recent"

   flag-keyword    = atom

   flag-list       = "(" [flag *(SP flag)] ")"

   flag-perm       = flag / "\*"

   greeting        = "*" SP (resp-cond-auth / resp-cond-bye) CRLF

   header-fld-name = astring

   header-list     = "(" header-fld-name *(SP header-fld-name) ")"

   list            = "LIST" SP mailbox SP list-mailbox

   list-mailbox    = 1*list-char / string

   list-char       = ATOM-CHAR / list-wildcards / resp-specials

   list-wildcards  = "%" / "*"

   literal         = "{" number "}" CRLF *CHAR8
                       ; Number represents the number of CHAR8s

   login           = "LOGIN" SP userid SP password

   lsub            = "LSUB" SP mailbox SP list-mailbox

   mailbox         = "INBOX" / astring
                       ; INBOX is case-insensitive.  All case variants of
                       ; INBOX (e.g. "iNbOx") MUST be interpreted as INBOX
                       ; not as an astring.  An astring which consists of
                       ; the case-insensitive sequence "I" "N" "B" "O" "X"
                       ; is considered to be INBOX and not an astring.
                       ;  Refer to section 5.1 for further
                       ; semantic details of mailbox names.

   mailbox-data    =  "FLAGS" SP flag-list / "LIST" SP mailbox-list /
                      "LSUB" SP mailbox-list / "SEARCH" *(SP nz-number) /
                      "STATUS" SP mailbox SP "("
                      [status-att SP number *(SP status-att SP number)] ")" /
                      number SP "EXISTS" / number SP "RECENT"

   mailbox-list    = "(" [mbx-list-flags] ")" SP
                      (DQUOTE QUOTED-CHAR DQUOTE / nil) SP mailbox

   mbx-list-flags  = *(mbx-list-oflag SP) mbx-list-sflag
                     *(SP mbx-list-oflag) /
                     mbx-list-oflag *(SP mbx-list-oflag)

   mbx-list-oflag  = "\Noinferiors" / flag-extension
                       ; Other flags; multiple possible per LIST response

   mbx-list-sflag  = "\Noselect" / "\Marked" / "\Unmarked"
                       ; Selectability flags; only one per LIST response

   media-basic     = ((DQUOTE ("APPLICATION" / "AUDIO" / "IMAGE" / "MESSAGE" /
                     "VIDEO") DQUOTE) / string) SP media-subtype
                       ; Defined in [MIME-IMT]

   media-message   = DQUOTE "MESSAGE" DQUOTE SP DQUOTE "RFC822" DQUOTE
                       ; Defined in [MIME-IMT]

   media-subtype   = string
                       ; Defined in [MIME-IMT]

   media-text      = DQUOTE "TEXT" DQUOTE SP media-subtype
                       ; Defined in [MIME-IMT]

   message-data    = nz-number SP ("EXPUNGE" / ("FETCH" SP msg-att))

   msg-att         = "(" (msg-att-dynamic / msg-att-static)
                      *(SP (msg-att-dynamic / msg-att-static)) ")"

   msg-att-dynamic = "FLAGS" SP "(" [flag-fetch *(SP flag-fetch)] ")"
                       ; MAY change for a message

   msg-att-static  = "ENVELOPE" SP envelope / "INTERNALDATE" SP date-time /
                     "RFC822" [".HEADER" / ".TEXT"] SP nstring /
                     "RFC822.SIZE" SP number / "BODY" ["STRUCTURE"] SP body /
                     "BODY" section ["<" number ">"] SP nstring /
                     "UID" SP uniqueid
                       ; MUST NOT change for a message

   nil             = "NIL"

   nstring         = string / nil

   number          = 1*DIGIT
                       ; Unsigned 32-bit integer
                       ; (0 <= n < 4,294,967,296)

   nz-number       = digit-nz *DIGIT
                       ; Non-zero unsigned 32-bit integer
                       ; (0 < n < 4,294,967,296)

   password        = astring

   quoted          = DQUOTE *QUOTED-CHAR DQUOTE

   QUOTED-CHAR     = <any TEXT-CHAR except quoted-specials> /
                     "\" quoted-specials

   quoted-specials = DQUOTE / "\"

   rename          = "RENAME" SP mailbox SP mailbox
                       ; Use of INBOX as a destination gives a NO error

   response        = *(continue-req / response-data) response-done

   response-data   = "*" SP (resp-cond-state / resp-cond-bye /
                     mailbox-data / message-data / capability-data) CRLF

   response-done   = response-tagged / response-fatal

   response-fatal  = "*" SP resp-cond-bye CRLF
                       ; Server closes connection immediately

   response-tagged = tag SP resp-cond-state CRLF

   resp-cond-auth  = ("OK" / "PREAUTH") SP resp-text
                       ; Authentication condition

   resp-cond-bye   = "BYE" SP resp-text

   resp-cond-state = ("OK" / "NO" / "BAD") SP resp-text
                       ; Status condition

   resp-specials   = "]"

   resp-text       = ["[" resp-text-code "]" SP] text

   resp-text-code  = "ALERT" /
                     "BADCHARSET" [SP "(" astring *(SP astring) ")" ] /
                     capability-data / "PARSE" /
                     "PERMANENTFLAGS" SP "(" [flag-perm *(SP flag-perm)] ")" /
                     "READ-ONLY" / "READ-WRITE" / "TRYCREATE" /
                     "UIDNEXT" SP nz-number / "UIDVALIDITY" SP nz-number /
                     "UNSEEN" SP nz-number /
                     atom [SP 1*<any TEXT-CHAR except "]">]

   search          = "SEARCH" [SP "CHARSET" SP astring] 1*(SP search-key)
                       ; CHARSET argument to MUST be registered with IANA

   search-key      = "ALL" / "ANSWERED" / "BCC" SP astring /
                     "BEFORE" SP date / "BODY" SP astring /
                     "CC" SP astring / "DELETED" / "FLAGGED" /
                     "FROM" SP astring / "KEYWORD" SP flag-keyword / "NEW" /
                     "OLD" / "ON" SP date / "RECENT" / "SEEN" /
                     "SINCE" SP date / "SUBJECT" SP astring /
                     "TEXT" SP astring / "TO" SP astring /
                     "UNANSWERED" / "UNDELETED" / "UNFLAGGED" /
                     "UNKEYWORD" SP flag-keyword / "UNSEEN" /
                       ; Above this line were in [IMAP2]
                     "DRAFT" / "HEADER" SP header-fld-name SP astring /
                     "LARGER" SP number / "NOT" SP search-key /
                     "OR" SP search-key SP search-key /
                     "SENTBEFORE" SP date / "SENTON" SP date /
                     "SENTSINCE" SP date / "SMALLER" SP number /
                     "UID" SP set / "UNDRAFT" / set /
                     "(" search-key *(SP search-key) ")"

   section         = "[" [section-spec] "]"

   section-msgtext = "HEADER" / "HEADER.FIELDS" [".NOT"] SP header-list /
                     "TEXT"
                       ; top-level or MESSAGE/RFC822 part

   section-part    = nz-number *("." nz-number)
                       ; body part nesting

   section-spec    = section-msgtext / (section-part ["." section-text])

   section-text    = section-msgtext / "MIME"
                       ; text other than actual body part (headers, etc.)

   select          = "SELECT" SP mailbox

   sequence-num    = nz-number / "*"
                       ; * is the largest number in use.  For message
                       ; sequence numbers, it is the number of messages
                       ; in the mailbox.  For unique identifiers, it is
                       ; the unique identifier of the last message in
                       ; the mailbox.

   set             = sequence-num / (sequence-num ":" sequence-num) /
                     (set "," set)
                       ; Identifies a set of messages.  For message
                       ; sequence numbers, these are consecutive
                       ; numbers from 1 to the number of messages in
                       ; the mailbox
                       ; Comma delimits individual numbers, colon
                       ; delimits between two numbers inclusive.
                       ; Example: 2,4:7,9,12:* is 2,4,5,6,7,9,12,13,
                       ; 14,15 for a mailbox with 15 messages.


   status          = "STATUS" SP mailbox SP "(" status-att *(SP status-att) ")"

   status-att      = "MESSAGES" / "RECENT" / "UIDNEXT" / "UIDVALIDITY" /
                     "UNSEEN"

   store           = "STORE" SP set SP store-att-flags

   store-att-flags = (["+" / "-"] "FLAGS" [".SILENT"]) SP
                     (flag-list / (flag *(SP flag)))

   string          = quoted / literal

   subscribe       = "SUBSCRIBE" SP mailbox

   tag             = 1*<any ASTRING-CHAR except "+">

   text            = 1*TEXT-CHAR

   TEXT-CHAR       = <any CHAR except CR and LF>

   time            = 2DIGIT ":" 2DIGIT ":" 2DIGIT
                       ; Hours minutes seconds

   uid             = "UID" SP (copy / fetch / search / store)
                       ; Unique identifiers used instead of message
                       ; sequence numbers

   uniqueid        = nz-number
                       ; Strictly ascending

   unsubscribe     = "UNSUBSCRIBE" SP mailbox

   userid          = astring

   x-command       = "X" atom <experimental command arguments>

   zone            = ("+" / "-") 4DIGIT
                       ; Signed four-digit value of hhmm representing
                       ; hours and minutes east of Greenwich (that is,
                       ; the amount that the given time differs from
                       ; Universal Time).  Subtracting the timezone
                       ; from the given time will give the UT form.
                       ; The Universal Time zone is "+0000".
*/


#ifndef MAILIMAP_TYPES_H

#define MAILIMAP_TYPES_H

#include <inttypes.h>
#include "mailstream.h"
#include "clist.h"

struct mailimap_address {
  char * personal_name; /* can be NULL */
  char * source_route;  /* can be NULL */
  char * mailbox_name;  /* can be NULL */
  char * host_name;     /* can be NULL */
};

enum {
  MAILIMAP_BODY_ERROR,
  MAILIMAP_BODY_1PART,
  MAILIMAP_BODY_MPART
};

struct mailimap_body {
  int type;
  struct mailimap_body_type_1part * body_1part; /* can be NULL */
  struct mailimap_body_type_mpart * body_mpart; /* can be NULL */
};


enum {
  MAILIMAP_BODY_EXTENSION_ERROR,
  MAILIMAP_BODY_EXTENSION_NSTRING,
  MAILIMAP_BODY_EXTENSION_NUMBER,
  MAILIMAP_BODY_EXTENSION_LIST
};

struct mailimap_body_extension {
  int type;
  char * nstring;    /* can be NULL */
  uint32_t number;
  clist * body_extension_list; /* list of (struct mailimap_body_extension *) */
                               /* can be NULL */
};

struct mailimap_body_ext_1part {
  char * body_md5;   /* != NULL */
  struct mailimap_body_fld_dsp * body_disposition; /* can be NULL */
  struct mailimap_body_fld_lang * body_language;   /* can be NULL */
  clist * body_extension_list; /* list of (struct mailimap_body_extension *) */
                               /* can be NULL */
};

struct mailimap_body_ext_mpart {
  struct mailimap_body_fld_param * body_parameter; /* != NULL */
  struct mailimap_body_fld_dsp * body_disposition; /* can be NULL */
  struct mailimap_body_fld_lang * body_language;   /* can be NULL */
  clist * body_extension_list; /* list of (struct mailimap_body_extension *) */
                               /* can be NULL */
};

struct mailimap_body_fields {
  struct mailimap_body_fld_param * body_parameter; /* != NULL */
  char * body_id;                                  /* can be NULL */
  char * body_description;                         /* can be NULL */
  struct mailimap_body_fld_enc * body_encoding;    /* != NULL */
  uint32_t body_size;
};

struct mailimap_body_fld_dsp {
  char * disposition_type;                     /* != NULL */
  struct mailimap_body_fld_param * attributes; /* != NULL */
};

enum {
  MAILIMAP_BODY_FLD_ENC_7BIT,
  MAILIMAP_BODY_FLD_ENC_8BIT,
  MAILIMAP_BODY_FLD_ENC_BINARY,
  MAILIMAP_BODY_FLD_ENC_BASE64,
  MAILIMAP_BODY_FLD_ENC_QUOTED_PRINTABLE,
  MAILIMAP_BODY_FLD_ENC_OTHER
};

struct mailimap_body_fld_enc {
  int type;
  char * value; /* can be NULL */
};

enum {
  MAILIMAP_BODY_FLD_LANG_ERROR,
  MAILIMAP_BODY_FLD_LANG_SINGLE,
  MAILIMAP_BODY_FLD_LANG_LIST
};

struct mailimap_body_fld_lang {
  int type;
  char * single; /* can be NULL */
  clist * list; /* list of string (gchar *) */
                /* can be NULL */
};


struct mailimap_single_body_fld_param {
  char * name;  /* != NULL */
  char * value; /* != NULL */
};

struct mailimap_body_fld_param {
  clist * list; /* list of (struct mailimap_single_body_fld_param *) */
                /* != NULL */
};


enum {
  MAILIMAP_BODY_TYPE_1PART_ERROR,
  MAILIMAP_BODY_TYPE_1PART_BASIC,
  MAILIMAP_BODY_TYPE_1PART_MSG,
  MAILIMAP_BODY_TYPE_1PART_TEXT
};

struct mailimap_body_type_1part {
  int type;
  struct mailimap_body_type_basic * body_type_basic; /* can be NULL */
  struct mailimap_body_type_msg * body_type_msg;     /* can be NULL */
  struct mailimap_body_type_text * body_type_text;   /* can be NULL */
  struct mailimap_body_ext_1part * body_ext_1part;   /* can be NULL */
};


struct mailimap_body_type_basic {
  struct mailimap_media_basic * media_basic; /* != NULL */
  struct mailimap_body_fields * body_fields; /* != NULL */
};

struct mailimap_body_type_mpart {
  clist * body_list; /* list of (struct mailimap_body *) */
                     /* != NULL */
  char * media_subtype; /* != NULL */
  struct mailimap_body_ext_mpart * body_ext_mpart; /* can be NULL */
};



struct mailimap_body_type_msg {
  struct mailimap_body_fields * body_fields; /* != NULL */
  struct mailimap_envelope * envelope;       /* != NULL */
  struct mailimap_body * body;               /* != NULL */
  uint32_t body_lines;
};


struct mailimap_body_type_text {
  char * media_text;                         /* != NULL */
  struct mailimap_body_fields * body_fields; /* != NULL */
  uint32_t body_lines;
};



enum {
  MAILIMAP_CAPABILITY_AUTH_TYPE,
  MAILIMAP_CAPABILITY_NAME
};

struct mailimap_capability {
  int type;
  char * auth_type; /* can be NULL */
  char * name;      /* can be NULL */
};



struct mailimap_capability_data {
  clist * list; /* list of (struct mailimap_capability *) */
                /* != NULL */
};


enum {
  MAILIMAP_CONTINUE_REQ_ERROR,
  MAILIMAP_CONTINUE_REQ_TEXT,
  MAILIMAP_CONTINUE_REQ_BASE64
};

struct mailimap_continue_req {
  int type;
  struct mailimap_resp_text * text; /* can be NULL */
  char * base64;                    /* can be NULL */
};



struct mailimap_date_time {
  int day;
  int month;
  int year;
  int hour;
  int min;
  int sec;
  int zone;
};



struct mailimap_envelope {
  char * date;                             /* can be NULL */
  char * subject;                          /* can be NULL */
  struct mailimap_env_from * from;         /* can be NULL */
  struct mailimap_env_sender * sender;     /* can be NULL */
  struct mailimap_env_reply_to * reply_to; /* can be NULL */
  struct mailimap_env_to * to;             /* can be NULL */
  struct mailimap_env_cc * cc;             /* can be NULL */
  struct mailimap_env_bcc * bcc;           /* can be NULL */
  char * in_reply_to;                      /* can be NULL */
  char * message_id;                       /* can be NULL */
};




struct mailimap_env_bcc {
  clist * list; /* list of (struct mailimap_address *) */
                /* != NULL */
};

struct mailimap_env_cc {
  clist * list; /* list of (struct mailimap_address *) */
                /* != NULL */
};



struct mailimap_env_from {
  clist * list; /* list of (struct mailimap_address *) */
                /* != NULL */
};



struct mailimap_env_reply_to {
  clist * list; /* list of (struct mailimap_address *) */
                /* != NULL */
};

struct mailimap_env_sender {
  clist * list; /* list of (struct mailimap_address *) */
                /* != NULL */
};


struct mailimap_env_to {
  clist * list; /* list of (struct mailimap_address *) */
                /* != NULL */
};


enum {
  MAILIMAP_FLAG_ANSWERED,
  MAILIMAP_FLAG_FLAGGED,
  MAILIMAP_FLAG_DELETED,
  MAILIMAP_FLAG_SEEN,
  MAILIMAP_FLAG_DRAFT,
  MAILIMAP_FLAG_KEYWORD,
  MAILIMAP_FLAG_EXTENSION
};

struct mailimap_flag {
  int type;
  char * keyword;   /* can be NULL */
  char * extension; /* can be NULL */
};



enum {
  MAILIMAP_FLAG_FETCH_ERROR,
  MAILIMAP_FLAG_FETCH_RECENT,
  MAILIMAP_FLAG_FETCH_OTHER
};

struct mailimap_flag_fetch {
  int type;
  struct mailimap_flag * flag; /* can be NULL */
};




enum {
  MAILIMAP_FLAG_PERM_ERROR,
  MAILIMAP_FLAG_PERM_FLAG,
  MAILIMAP_FLAG_PERM_ALL
};

struct mailimap_flag_perm {
  int type;
  struct mailimap_flag * flag; /* can be NULL */
};



struct mailimap_flag_list {
  clist * list; /* list of (struct mailimap_flag *) */
                /* != NULL */
};



enum {
  MAILIMAP_GREETING_RESP_COND_ERROR,
  MAILIMAP_GREETING_RESP_COND_AUTH,
  MAILIMAP_GREETING_RESP_COND_BYE
};

struct mailimap_greeting {
  int type;
  struct mailimap_resp_cond_auth * auth; /* can be NULL */
  struct mailimap_resp_cond_bye * bye;   /* can be NULL */
};




struct mailimap_header_list {
  clist * list; /* list of astring (gchar *) */
                /* != NULL */
};



struct mailimap_status_info {
  int att;
  uint32_t value;
};

struct mailimap_mailbox_data_status {
  char * mailbox;
  clist * status_info_list; /* list of (struct mailimap_status_info *) */
                            /* can be NULL */
};


enum {
  MAILIMAP_MAILBOX_DATA_ERROR,
  MAILIMAP_MAILBOX_DATA_FLAGS,
  MAILIMAP_MAILBOX_DATA_LIST,
  MAILIMAP_MAILBOX_DATA_LSUB,
  MAILIMAP_MAILBOX_DATA_SEARCH,
  MAILIMAP_MAILBOX_DATA_STATUS,
  MAILIMAP_MAILBOX_DATA_EXISTS,
  MAILIMAP_MAILBOX_DATA_RECENT
};

struct mailimap_mailbox_data {
  int type;
  struct mailimap_flag_list * flags;   /* can be NULL */
  struct mailimap_mailbox_list * list; /* can be NULL */
  struct mailimap_mailbox_list * lsub; /* can be NULL */
  clist * search;  /* list of nz-number (guint32 *) */
                   /* can be NULL */
  struct mailimap_mailbox_data_status *  status; /* can be NULL */
  uint32_t exists;
  uint32_t recent;
};



enum {
  MAILIMAP_MBX_LIST_FLAGS_SFLAG,
  MAILIMAP_MBX_LIST_FLAGS_NO_SFLAG
};

struct mailimap_mbx_list_flags {
  int type;
  clist * oflags; /* list of (struct mailimap_mbx_list_oflag *) */
                  /* != NULL */
  int sflag;
};



enum {
  MAILIMAP_MBX_LIST_OFLAG_ERROR,
  MAILIMAP_MBX_LIST_OFLAG_NOINFERIORS,
  MAILIMAP_MBX_LIST_OFLAG_FLAG_EXT
};

struct mailimap_mbx_list_oflag {
  int type;
  char * flag_ext; /* can be NULL */
};




enum {
  MAILIMAP_MBX_LIST_SFLAG_MARKED,
  MAILIMAP_MBX_LIST_SFLAG_NOSELECT,
  MAILIMAP_MBX_LIST_SFLAG_UNMARKED
};



struct mailimap_mailbox_list {
  struct mailimap_mbx_list_flags * mb_flag; /* can be NULL */
  char delimiter;
  char * mb; /* != NULL */
};



enum {
  MAILIMAP_MEDIA_BASIC_APPLICATION,
  MAILIMAP_MEDIA_BASIC_AUDIO,
  MAILIMAP_MEDIA_BASIC_IMAGE,
  MAILIMAP_MEDIA_BASIC_MESSAGE,
  MAILIMAP_MEDIA_BASIC_VIDEO,
  MAILIMAP_MEDIA_BASIC_OTHER
};

struct mailimap_media_basic {
  int type;
  char * basic_type; /* can be NULL */
  char * subtype;    /* != NULL */
};





enum {
  MAILIMAP_MESSAGE_DATA_ERROR,
  MAILIMAP_MESSAGE_DATA_EXPUNGE,
  MAILIMAP_MESSAGE_DATA_FETCH
};

struct mailimap_message_data {
  uint32_t number;
  int type;
  struct mailimap_msg_att * msg_att; /* can be NULL */
                                     /* if type = FETCH, can be NULL */
};



enum {
  MAILIMAP_MSG_ATT_ITEM_ERROR,
  MAILIMAP_MSG_ATT_ITEM_DYNAMIC,
  MAILIMAP_MSG_ATT_ITEM_STATIC
};


struct mailimap_msg_att_item {
  int type;
  struct mailimap_msg_att_dynamic * msg_att_dyn;   /* can be NULL */
  struct mailimap_msg_att_static * msg_att_static; /* can be NULL */
};

struct mailimap_msg_att {
  clist * list; /* list of (struct mailimap_msg_att_item *) */
                /* != NULL */
  uint32_t number; /* extra field to store the message number,
		     used for mailimap */
};



struct mailimap_msg_att_dynamic {
  clist * list; /* list of (struct mailimap_flag_fetch *) */
                /* can be NULL */
};


enum {
  MAILIMAP_MSG_ATT_ERROR,
  MAILIMAP_MSG_ATT_ENVELOPE,
  MAILIMAP_MSG_ATT_INTERNALDATE,
  MAILIMAP_MSG_ATT_RFC822,
  MAILIMAP_MSG_ATT_RFC822_HEADER,
  MAILIMAP_MSG_ATT_RFC822_TEXT,
  MAILIMAP_MSG_ATT_RFC822_SIZE,
  MAILIMAP_MSG_ATT_BODY,
  MAILIMAP_MSG_ATT_BODYSTRUCTURE,
  MAILIMAP_MSG_ATT_BODY_SECTION,
  MAILIMAP_MSG_ATT_UID
};

struct mailimap_msg_att_body_section {
  struct mailimap_section * section; /* != NULL */
  uint32_t origin_octet;
  char * body_part; /* can be NULL */
  size_t length;
};

struct mailimap_msg_att_static {
  int type;
  struct mailimap_envelope * env;            /* can be NULL */
  struct mailimap_date_time * internal_date; /* can be NULL */
  char * rfc822;        /* can be NULL */
  char * rfc822_header; /* can be NULL */
  char * rfc822_text;   /* can be NULL */
  size_t length;
  uint32_t rfc822_size;
  struct mailimap_body * bodystructure; /* can be NULL */
  struct mailimap_body * body;          /* can be NULL */
  struct mailimap_msg_att_body_section * body_section; /* can be NULL */
  uint32_t uid;
};




enum {
  MAILIMAP_RESP_ERROR,
  MAILIMAP_RESP_CONT_REQ,
  MAILIMAP_RESP_RESP_DATA
};

struct mailimap_cont_req_or_resp_data {
  int type;
  struct mailimap_continue_req * cont_req;   /* can be NULL */
  struct mailimap_response_data * resp_data; /* can be NULL */
};

struct mailimap_response {
  /* list of (struct mailiap_cont_req_or_resp_data *) */
  clist * cont_req_or_resp_data_list; /* can be NULL */
  struct mailimap_response_done * resp_done; /* != NULL */
};





enum {
  MAILIMAP_RESP_DATA_TYPE_ERROR,
  MAILIMAP_RESP_DATA_TYPE_COND_STATE,
  MAILIMAP_RESP_DATA_TYPE_COND_BYE,
  MAILIMAP_RESP_DATA_TYPE_MAILBOX_DATA,
  MAILIMAP_RESP_DATA_TYPE_MESSAGE_DATA,
  MAILIMAP_RESP_DATA_TYPE_CAPABILITY_DATA
};

struct mailimap_response_data {
  int type;
  struct mailimap_resp_cond_state * cond_state;      /* can be NULL */
  struct mailimap_resp_cond_bye * bye;               /* can be NULL */
  struct mailimap_mailbox_data * mailbox_data;       /* can be NULL */
  struct mailimap_message_data * message_data;       /* can be NULL */
  struct mailimap_capability_data * capability_data; /* can be NULL */
};




enum {
  MAILIMAP_RESP_DONE_TYPE_ERROR,
  MAILIMAP_RESP_DONE_TYPE_TAGGED,
  MAILIMAP_RESP_DONE_TYPE_FATAL
};

struct mailimap_response_done {
  int type;
  struct mailimap_response_tagged * tagged; /* can be NULL */
  struct mailimap_response_fatal * fatal;   /* can be NULL */
};


struct mailimap_response_fatal {
  struct mailimap_resp_cond_bye * bye; /* != NULL */
};


struct mailimap_response_tagged {
  char * tag; /* != NULL */
  struct mailimap_resp_cond_state * cond_state; /* != NULL */
};


enum {
  MAILIMAP_RESP_COND_AUTH_ERROR,
  MAILIMAP_RESP_COND_AUTH_OK,
  MAILIMAP_RESP_COND_AUTH_PREAUTH
};

struct mailimap_resp_cond_auth {
  int type;
  struct mailimap_resp_text * text; /* != NULL */
};


struct mailimap_resp_cond_bye {
  struct mailimap_resp_text * text; /* != NULL */
};


enum {
  MAILIMAP_RESP_COND_STATE_OK,
  MAILIMAP_RESP_COND_STATE_NO,
  MAILIMAP_RESP_COND_STATE_BAD
};

struct mailimap_resp_cond_state {
  int type;
  struct mailimap_resp_text * text; /* can be NULL */
};


struct mailimap_resp_text {
  struct mailimap_resp_text_code * resp_code; /* can be NULL */
  char * text; /* != NULL */
};



enum {
  MAILIMAP_RESP_TEXT_CODE_ALERT,
  MAILIMAP_RESP_TEXT_CODE_BADCHARSET,
  MAILIMAP_RESP_TEXT_CODE_CAPABILITY_DATA,
  MAILIMAP_RESP_TEXT_CODE_PARSE,
  MAILIMAP_RESP_TEXT_CODE_PERMANENTFLAGS,
  MAILIMAP_RESP_TEXT_CODE_READ_ONLY,
  MAILIMAP_RESP_TEXT_CODE_READ_WRITE,
  MAILIMAP_RESP_TEXT_CODE_TRY_CREATE,
  MAILIMAP_RESP_TEXT_CODE_UIDNEXT,
  MAILIMAP_RESP_TEXT_CODE_UIDVALIDITY,
  MAILIMAP_RESP_TEXT_CODE_UNSEEN,
  MAILIMAP_RESP_TEXT_CODE_OTHER
};

struct mailimap_resp_text_code {
  int type;
  clist * badcharset; /* list of astring (gchar *) */
                      /* can be NULL */
  struct mailimap_capability_data * cap_data; /* != NULL */
  clist * perm_flags; /* list of (struct mailimap_flag_perm *) */
                      /* can be NULL */
  uint32_t uidnext;
  uint32_t uidvalidity;
  uint32_t unseen;
  char * atom;       /* can be NULL */
  char * atom_value; /* can be NULL */
};

struct mailimap_section {
  struct mailimap_section_spec * section_spec; /* can be NULL */
};


enum {
  MAILIMAP_SECTION_MSGTEXT_HEADER,
  MAILIMAP_SECTION_MSGTEXT_HEADER_FIELDS,
  MAILIMAP_SECTION_MSGTEXT_HEADER_FIELDS_NOT,
  MAILIMAP_SECTION_MSGTEXT_TEXT
};

struct mailimap_section_msgtext {
  int type;
  struct mailimap_header_list * header_list; /* can be NULL */
};



struct mailimap_section_part {
  clist * section_id; /* list of nz-number (guint32 *) */
                      /* != NULL */
};



enum {
  MAILIMAP_SECTION_SPEC_SECTION_MSGTEXT,
  MAILIMAP_SECTION_SPEC_SECTION_PART
};

struct mailimap_section_spec {
  int type;
  struct mailimap_section_msgtext * section_msgtext; /* can be NULL */
  struct mailimap_section_part * section_part;       /* can be NULL */
  struct mailimap_section_text * section_text;       /* can be NULL */
};



enum {
  MAILIMAP_SECTION_TEXT_ERROR,
  MAILIMAP_SECTION_TEXT_SECTION_MSGTEXT,
  MAILIMAP_SECTION_TEXT_MIME
};

struct mailimap_section_text {
  int type;
  struct mailimap_section_msgtext * section_msgtext; /* can be NULL */
};




enum {
  MAILIMAP_STATUS_ATT_MESSAGES,
  MAILIMAP_STATUS_ATT_RECENT,
  MAILIMAP_STATUS_ATT_UIDNEXT,
  MAILIMAP_STATUS_ATT_UIDVALIDITY,
  MAILIMAP_STATUS_ATT_UNSEEN
};








/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */




/* sender only */





struct mailimap_set_item {
  uint32_t first;
  uint32_t last;
};

struct mailimap_set {
  clist * list;
};

struct mailimap_date {
  int day;
  int month;
  int year;
};

enum {
  MAILIMAP_FETCH_ATT_ENVELOPE,
  MAILIMAP_FETCH_ATT_FLAGS,
  MAILIMAP_FETCH_ATT_INTERNALDATE,
  MAILIMAP_FETCH_ATT_RFC822,
  MAILIMAP_FETCH_ATT_RFC822_HEADER,
  MAILIMAP_FETCH_ATT_RFC822_SIZE,
  MAILIMAP_FETCH_ATT_RFC822_TEXT,
  MAILIMAP_FETCH_ATT_BODY,
  MAILIMAP_FETCH_ATT_BODYSTRUCTURE,
  MAILIMAP_FETCH_ATT_UID,
  MAILIMAP_FETCH_ATT_BODY_SECTION,
  MAILIMAP_FETCH_ATT_BODY_PEEK_SECTION
};

struct mailimap_fetch_att {
  int type;
  struct mailimap_section * section;
  uint32_t offset;
  uint32_t size;
};





enum {
  MAILIMAP_FETCH_TYPE_ALL,
  MAILIMAP_FETCH_TYPE_FULL,
  MAILIMAP_FETCH_TYPE_FAST,
  MAILIMAP_FETCH_TYPE_FETCH_ATT,
  MAILIMAP_FETCH_TYPE_FETCH_ATT_LIST
};

struct mailimap_fetch_type {
  int type;
  struct mailimap_fetch_att * fetch_att;
  clist * fetch_att_list; /* list of (struct mailimap_fetch_att *) */
};


struct mailimap_store_att_flags {
  int sign;
  int silent;
  struct mailimap_flag_list * flag_list;
};


enum {
  MAILIMAP_SEARCH_KEY_ALL,
  MAILIMAP_SEARCH_KEY_ANSWERED,
  MAILIMAP_SEARCH_KEY_BCC,
  MAILIMAP_SEARCH_KEY_BEFORE,
  MAILIMAP_SEARCH_KEY_BODY,
  MAILIMAP_SEARCH_KEY_CC,
  MAILIMAP_SEARCH_KEY_DELETED,
  MAILIMAP_SEARCH_KEY_FLAGGED,
  MAILIMAP_SEARCH_KEY_FROM,
  MAILIMAP_SEARCH_KEY_KEYWORD,
  MAILIMAP_SEARCH_KEY_NEW,
  MAILIMAP_SEARCH_KEY_OLD,
  MAILIMAP_SEARCH_KEY_ON,
  MAILIMAP_SEARCH_KEY_RECENT,
  MAILIMAP_SEARCH_KEY_SEEN,
  MAILIMAP_SEARCH_KEY_SINCE,
  MAILIMAP_SEARCH_KEY_SUBJECT,
  MAILIMAP_SEARCH_KEY_TEXT,
  MAILIMAP_SEARCH_KEY_TO,
  MAILIMAP_SEARCH_KEY_UNANSWERED,
  MAILIMAP_SEARCH_KEY_UNDELETED,
  MAILIMAP_SEARCH_KEY_UNFLAGGED,
  MAILIMAP_SEARCH_KEY_UNKEYWORD,
  MAILIMAP_SEARCH_KEY_UNSEEN,
  MAILIMAP_SEARCH_KEY_DRAFT,
  MAILIMAP_SEARCH_KEY_HEADER,
  MAILIMAP_SEARCH_KEY_LARGER,
  MAILIMAP_SEARCH_KEY_NOT,
  MAILIMAP_SEARCH_KEY_OR,
  MAILIMAP_SEARCH_KEY_SENTBEFORE,
  MAILIMAP_SEARCH_KEY_SENTON,
  MAILIMAP_SEARCH_KEY_SENTSINCE,
  MAILIMAP_SEARCH_KEY_SMALLER,
  MAILIMAP_SEARCH_KEY_UID,
  MAILIMAP_SEARCH_KEY_UNDRAFT,
  MAILIMAP_SEARCH_KEY_SET,
  MAILIMAP_SEARCH_KEY_MULTIPLE
};

struct mailimap_search_key {
  int type;
  char * bcc;
  struct mailimap_date * before;
  char * body;
  char * cc;
  char * from;
  char * keyword;
  struct mailimap_date * on;
  struct mailimap_date * since;
  char * subject;
  char * text;
  char * to;
  char * unkeyword;
  char * header_name;
  char * header_value;
  uint32_t larger;
  struct mailimap_search_key * not;
  struct mailimap_search_key * or1;
  struct mailimap_search_key * or2;
  struct mailimap_date * sentbefore;
  struct mailimap_date * senton;
  struct mailimap_date * sentsince;
  uint32_t smaller;
  struct mailimap_set * uid;
  struct mailimap_set * set;
  clist * multiple; /* list of (struct mailimap_search_key *) */
};

struct mailimap_status_att_list {
  clist * list;
};

/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */

/* tools functions */

uint32_t * mailimap_number_alloc_new(uint32_t number);
void mailimap_number_alloc_free(uint32_t * pnumber);


/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */



/* functions declaration */

void mailimap_addr_host_free(char * addr_host);

void mailimap_addr_mailbox_free(char * addr_mailbox);

void mailimap_addr_adl_free(char * addr_adl);

void mailimap_addr_name_free(char * addr_name);

struct mailimap_address *
mailimap_address_new(char * personal_name, char * source_route,
		     char * mailbox_name, char * host_name);

void mailimap_address_free(struct mailimap_address * addr);

void mailimap_astring_free(char * astring);

void mailimap_atom_free(char * atom);

void mailimap_auth_type_free(char * auth_type);

void mailimap_base64_free(char * base64);

struct mailimap_body *
mailimap_body_new(int type,
		  struct mailimap_body_type_1part * body_1part,
		  struct mailimap_body_type_mpart * body_mpart);


void mailimap_body_free(struct mailimap_body * body);

struct mailimap_body_extension *
mailimap_body_extension_new(int type, char * nstring,
			    uint32_t number, clist * body_extension_list);

void mailimap_body_extension_free(struct mailimap_body_extension * be);


struct mailimap_body_ext_1part *
mailimap_body_ext_1part_new(char * body_md5,
			    struct mailimap_body_fld_dsp * body_disposition,
			    struct mailimap_body_fld_lang * body_language,
			    clist * body_extension_list);


void
mailimap_body_ext_1part_free(struct mailimap_body_ext_1part * body_ext_1part);

struct mailimap_body_ext_mpart *
mailimap_body_ext_mpart_new(struct mailimap_body_fld_param * body_parameter,
			    struct mailimap_body_fld_dsp * body_disposition,
			    struct mailimap_body_fld_lang * body_language,
			    clist * body_extension_list);

void
mailimap_body_ext_mpart_free(struct mailimap_body_ext_mpart * body_ext_mpart);

struct mailimap_body_fields *
mailimap_body_fields_new(struct mailimap_body_fld_param * body_parameter,
			 char * body_id,
			 char * body_description,
			 struct mailimap_body_fld_enc * body_encoding,
			 uint32_t body_size);

void
mailimap_body_fields_free(struct mailimap_body_fields * body_fields);

void mailimap_body_fld_desc_free(char * body_fld_desc);

struct mailimap_body_fld_dsp *
mailimap_body_fld_dsp_new(char * disposition_type,
			  struct mailimap_body_fld_param * attributes);

void mailimap_body_fld_dsp_free(struct mailimap_body_fld_dsp * bfd);

struct mailimap_body_fld_enc *
mailimap_body_fld_enc_new(int type, char * value);

void mailimap_body_fld_enc_free(struct mailimap_body_fld_enc * bfe);

void mailimap_body_fld_id_free(char * body_fld_id);

struct mailimap_body_fld_lang *
mailimap_body_fld_lang_new(int type, char * single, clist * list);

void
mailimap_body_fld_lang_free(struct mailimap_body_fld_lang * fld_lang);

void mailimap_body_fld_md5_free(char * body_fld_md5);

struct mailimap_single_body_fld_param *
mailimap_single_body_fld_param_new(char * name, char * value);

void
mailimap_single_body_fld_param_free(struct mailimap_single_body_fld_param * p);

struct mailimap_body_fld_param *
mailimap_body_fld_param_new(clist * list);

void
mailimap_body_fld_param_free(struct mailimap_body_fld_param * fld_param);

struct mailimap_body_type_1part *
mailimap_body_type_1part_new(int type,
			     struct mailimap_body_type_basic * body_type_basic,
			     struct mailimap_body_type_msg * body_type_msg,
			     struct mailimap_body_type_text * body_type_text,
			     struct mailimap_body_ext_1part * body_ext_1part);

void
mailimap_body_type_1part_free(struct mailimap_body_type_1part * bt1p);

struct mailimap_body_type_basic *
mailimap_body_type_basic_new(struct mailimap_media_basic * media_basic,
			     struct mailimap_body_fields * body_fields);

void mailimap_body_type_basic_free(struct mailimap_body_type_basic *
				   body_type_basic);

struct mailimap_body_type_mpart *
mailimap_body_type_mpart_new(clist * body_list, char * media_subtype,
			     struct mailimap_body_ext_mpart * body_ext_mpart);

void mailimap_body_type_mpart_free(struct mailimap_body_type_mpart *
				   body_type_mpart);

struct mailimap_body_type_msg *
mailimap_body_type_msg_new(struct mailimap_body_fields * body_fields,
			   struct mailimap_envelope * envelope,
			   struct mailimap_body * body,
			   uint32_t body_fld_lines);

void
mailimap_body_type_msg_free(struct mailimap_body_type_msg * body_type_msg);

struct mailimap_body_type_text *
mailimap_body_type_text_new(char * media_text,
			    struct mailimap_body_fields * body_fields,
			    uint32_t body_lines);

void
mailimap_body_type_text_free(struct mailimap_body_type_text * body_type_text);

struct mailimap_capability *
mailimap_capability_new(int type, char * auth_type, char * name);

void mailimap_capability_free(struct mailimap_capability * c);

struct mailimap_capability_data *
mailimap_capability_data_new(clist * list);

void
mailimap_capability_data_free(struct mailimap_capability_data * cap_data);

struct mailimap_continue_req *
mailimap_continue_req_new(int type, struct mailimap_resp_text * text,
			  char * base64);

void mailimap_continue_req_free(struct mailimap_continue_req * cont_req);

struct mailimap_date_time *
mailimap_date_time_new(int day, int month, int year, int hour,
		       int min, int sec, int zone);

void mailimap_date_time_free(struct mailimap_date_time * date_time);

struct mailimap_envelope *
mailimap_envelope_new(char * date, char * subject,
		      struct mailimap_env_from * from,
		      struct mailimap_env_sender * sender,
		      struct mailimap_env_reply_to * reply_to,
		      struct mailimap_env_to * to,
		      struct mailimap_env_cc* cc,
		      struct mailimap_env_bcc * bcc,
		      char * in_reply_to, char * message_id);

void mailimap_envelope_free(struct mailimap_envelope * env);

struct mailimap_env_bcc * mailimap_env_bcc_new(clist * list);

void mailimap_env_bcc_free(struct mailimap_env_bcc * env_bcc);

struct mailimap_env_cc * mailimap_env_cc_new(clist * list);

void mailimap_env_cc_free(struct mailimap_env_cc * env_cc);

void mailimap_env_date_free(char * date);

struct mailimap_env_from * mailimap_env_from_new(clist * list);

void mailimap_env_from_free(struct mailimap_env_from * env_from);

void mailimap_env_in_reply_to_free(char * in_reply_to);

void mailimap_env_message_id_free(char * message_id);

struct mailimap_env_reply_to * mailimap_env_reply_to_new(clist * list);

void
mailimap_env_reply_to_free(struct mailimap_env_reply_to * env_reply_to);

struct mailimap_env_sender * mailimap_env_sender_new(clist * list);

void mailimap_env_sender_free(struct mailimap_env_sender * env_sender);

void mailimap_env_subject_free(char * subject);

struct mailimap_env_to * mailimap_env_to_new(clist * list);

void mailimap_env_to_free(struct mailimap_env_to * env_to);

struct mailimap_flag * mailimap_flag_new(int type,
					 char * keyword,
					 char * extension);

void mailimap_flag_free(struct mailimap_flag * f);

void mailimap_flag_extension_free(char * flag_extension);

struct mailimap_flag_fetch *
mailimap_flag_fetch_new(int type, struct mailimap_flag * flag);

void mailimap_flag_fetch_free(struct mailimap_flag_fetch * flag_fetch);

void mailimap_flag_keyword_free(char * flag_keyword);

struct mailimap_flag_list *
mailimap_flag_list_new(clist * list);

void mailimap_flag_list_free(struct mailimap_flag_list * flag_list);

struct mailimap_flag_perm *
mailimap_flag_perm_new(int type, struct mailimap_flag * flag);

void mailimap_flag_perm_free(struct mailimap_flag_perm * flag_perm);

struct mailimap_greeting *
mailimap_greeting_new(int type,
		      struct mailimap_resp_cond_auth * auth,
		      struct mailimap_resp_cond_bye * bye);

void mailimap_greeting_free(struct mailimap_greeting * greeting);

void
mailimap_header_fld_name_free(char * header_fld_name);

struct mailimap_header_list *
mailimap_header_list_new(clist * list);

void
mailimap_header_list_free(struct mailimap_header_list * header_list);

void mailimap_literal_free(char * literal);

void mailimap_mailbox_free(char * mailbox);

struct mailimap_status_info *
mailimap_status_info_new(int att, uint32_t value);

void mailimap_status_info_free(struct mailimap_status_info * info);

void
mailimap_mailbox_data_search_free(clist * data_search);

struct mailimap_mailbox_data_status *
mailimap_mailbox_data_status_new(char * mailbox,
				 clist * status_info_list);

void
mailimap_mailbox_data_status_free(struct mailimap_mailbox_data_status * info);

struct mailimap_mailbox_data *
mailimap_mailbox_data_new(int type, struct mailimap_flag_list * flags,
			  struct mailimap_mailbox_list * list,
			  struct mailimap_mailbox_list * lsub,
			  clist * search,
			  struct mailimap_mailbox_data_status * status,
			  uint32_t exists,
			  uint32_t recent);

void
mailimap_mailbox_data_free(struct mailimap_mailbox_data * mb_data);

struct mailimap_mbx_list_flags *
mailimap_mbx_list_flags_new(int type, clist * oflags, int sflag);

void
mailimap_mbx_list_flags_free(struct mailimap_mbx_list_flags * mbx_list_flags);

struct mailimap_mbx_list_oflag *
mailimap_mbx_list_oflag_new(int type, char * flag_ext);

void
mailimap_mbx_list_oflag_free(struct mailimap_mbx_list_oflag * oflag);

struct mailimap_mailbox_list *
mailimap_mailbox_list_new(struct mailimap_mbx_list_flags * mbx_flags,
			  char delimiter, char * mb);

void
mailimap_mailbox_list_free(struct mailimap_mailbox_list * mb_list);

struct mailimap_media_basic *
mailimap_media_basic_new(int type, char * basic_type, char * subtype);

void
mailimap_media_basic_free(struct mailimap_media_basic * media_basic);

void mailimap_media_subtype_free(char * media_subtype);

void mailimap_media_text_free(char * media_text);

struct mailimap_message_data *
mailimap_message_data_new(uint32_t number, int type,
			  struct mailimap_msg_att * msg_att);

void
mailimap_message_data_free(struct mailimap_message_data * msg_data);

struct mailimap_msg_att_item *
mailimap_msg_att_item_new(int type,
			  struct mailimap_msg_att_dynamic * msg_att_dyn,
			  struct mailimap_msg_att_static * msg_att_static);

void
mailimap_msg_att_item_free(struct mailimap_msg_att_item * item);

struct mailimap_msg_att * mailimap_msg_att_new(clist * list);

void mailimap_msg_att_free(struct mailimap_msg_att * msg_att);

struct mailimap_msg_att_dynamic *
mailimap_msg_att_dynamic_new(clist * list);

void
mailimap_msg_att_dynamic_free(struct mailimap_msg_att_dynamic * msg_att_dyn);

struct mailimap_msg_att_body_section *
mailimap_msg_att_body_section_new(struct mailimap_section * section,
				  uint32_t origin_octet,
				  char * body_part,
				  size_t length);

void
mailimap_msg_att_body_section_free(struct mailimap_msg_att_body_section * 
				   msg_att_body_section);

struct mailimap_msg_att_static *
mailimap_msg_att_static_new(int type, struct mailimap_envelope * env,
			    struct mailimap_date_time * internal_date,
			    char * rfc822,
			    char * rfc822_header,
			    char * rfc822_text,
			    size_t length,
			    uint32_t rfc822_size,
			    struct mailimap_body * bodystructure,
			    struct mailimap_body * body,
			    struct mailimap_msg_att_body_section *
			    body_section,
			    uint32_t uid);


void mailimap_msg_att_envelope_free(struct mailimap_envelope * env);

void
mailimap_msg_att_internaldate_free(struct mailimap_date_time * date_time);

void
mailimap_msg_att_rfc822_free(char * str);

void
mailimap_msg_att_rfc822_header_free(char * str);

void
mailimap_msg_att_rfc822_text_free(char * str);

void
mailimap_msg_att_body_free(struct mailimap_body * body);

void
mailimap_msg_att_bodystructure_free(struct mailimap_body * body);

void
mailimap_msg_att_static_free(struct mailimap_msg_att_static * item);

void mailimap_nstring_free(char * str);

struct mailimap_cont_req_or_resp_data *
mailimap_cont_req_or_resp_data_new(int type,
				   struct mailimap_continue_req * cont_req,
				   struct mailimap_response_data * resp_data);

void
mailimap_cont_req_or_resp_data_free(struct mailimap_cont_req_or_resp_data *
				    cont_req_or_resp_data);

struct mailimap_response *
mailimap_response_new(clist * cont_req_or_resp_data_list,
		      struct mailimap_response_done * resp_done);

void
mailimap_response_free(struct mailimap_response * resp);

struct mailimap_response_data *
mailimap_response_data_new(int type,
			   struct mailimap_resp_cond_state * cond_state,
			   struct mailimap_resp_cond_bye * bye,
			   struct mailimap_mailbox_data * mailbox_data,
			   struct mailimap_message_data * message_data,
			   struct mailimap_capability_data * capability_data);

void
mailimap_response_data_free(struct mailimap_response_data * resp_data);

struct mailimap_response_done *
mailimap_response_done_new(int type,
			   struct mailimap_response_tagged * tagged,
			   struct mailimap_response_fatal * fatal);

void mailimap_response_done_free(struct mailimap_response_done *
				 resp_done);

struct mailimap_response_fatal *
mailimap_response_fatal_new(struct mailimap_resp_cond_bye * bye);

void mailimap_response_fatal_free(struct mailimap_response_fatal * resp_fatal);

struct mailimap_response_tagged *
mailimap_response_tagged_new(char * tag,
			     struct mailimap_resp_cond_state * cond_state);

void
mailimap_response_tagged_free(struct mailimap_response_tagged * tagged);

struct mailimap_resp_cond_auth *
mailimap_resp_cond_auth_new(int type,
			    struct mailimap_resp_text * text);

void
mailimap_resp_cond_auth_free(struct mailimap_resp_cond_auth * cond_auth);

struct mailimap_resp_cond_bye *
mailimap_resp_cond_bye_new(struct mailimap_resp_text * text);

void
mailimap_resp_cond_bye_free(struct mailimap_resp_cond_bye * cond_bye);

struct mailimap_resp_cond_state *
mailimap_resp_cond_state_new(int type,
			     struct mailimap_resp_text * text);
void
mailimap_resp_cond_state_free(struct mailimap_resp_cond_state * cond_state);

struct mailimap_resp_text *
mailimap_resp_text_new(struct mailimap_resp_text_code * resp_code,
		       char * text);

void mailimap_resp_text_free(struct mailimap_resp_text * resp_text);

struct mailimap_resp_text_code *
mailimap_resp_text_code_new(int type, clist * badcharset,
			    struct mailimap_capability_data * cap_data,
			    clist * perm_flags,
			    uint32_t uidnext, uint32_t uidvalidity,
			    uint32_t unseen, char * atom, char * atom_value);

void
mailimap_resp_text_code_free(struct mailimap_resp_text_code * resp_text_code);

struct mailimap_section *
mailimap_section_new(struct mailimap_section_spec * section_spec);

void mailimap_section_free(struct mailimap_section * section);

struct mailimap_section_msgtext *
mailimap_section_msgtext_new(int type,
			     struct mailimap_header_list * header_list);

void
mailimap_section_msgtext_free(struct mailimap_section_msgtext * msgtext);

struct mailimap_section_part *
mailimap_section_part_new(clist * section_id);

void
mailimap_section_part_free(struct mailimap_section_part * section_part);

struct mailimap_section_spec *
mailimap_section_spec_new(int type,
			  struct mailimap_section_msgtext * section_msgtext,
			  struct mailimap_section_part * section_part,
			  struct mailimap_section_text * section_text);

void
mailimap_section_spec_free(struct mailimap_section_spec * section_spec);

struct mailimap_section_text *
mailimap_section_text_new(int type,
			  struct mailimap_section_msgtext * section_msgtext);

void
mailimap_section_text_free(struct mailimap_section_text * section_text);

void
mailimap_string_free(char * str);

void mailimap_tag_free(char * tag);

void mailimap_text_free(char * text);







/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */
/* ************************************************************************* */


/* sender only */


struct mailimap_set_item *
mailimap_set_item_new(uint32_t first, uint32_t last);

void mailimap_set_item_free(struct mailimap_set_item * set_item);

struct mailimap_set * mailimap_set_new(clist * list);

void mailimap_set_free(struct mailimap_set * set);

struct mailimap_date * mailimap_date_new(int day, int month, int year);

void mailimap_date_free(struct mailimap_date * date);


struct mailimap_fetch_att *
mailimap_fetch_att_new(int type, struct mailimap_section * section,
		       uint32_t offset, uint32_t size);


void mailimap_fetch_att_free(struct mailimap_fetch_att * fetch_att);



struct mailimap_fetch_type *
mailimap_fetch_type_new(int type,
			struct mailimap_fetch_att * fetch_att,
			clist * fetch_att_list);


void mailimap_fetch_type_free(struct mailimap_fetch_type * fetch_type);

struct mailimap_store_att_flags *
mailimap_store_att_flags_new(int sign, int silent,
			     struct mailimap_flag_list * flag_list);

void mailimap_store_att_flags_free(struct mailimap_store_att_flags *
				   store_att_flags);


struct mailimap_search_key *
mailimap_search_key_new(int type,
			char * bcc,
			struct mailimap_date * before,
			char * body,
			char * cc,
			char * from,
			char * keyword,
			struct mailimap_date * on,
			struct mailimap_date * since,
			char * subject,
			char * text,
			char * to,
			char * unkeyword,
			char * header_name,
			char * header_value,
			uint32_t larger,
			struct mailimap_search_key * not,
			struct mailimap_search_key * or1,
			struct mailimap_search_key * or2,
			struct mailimap_date * sentbefore,
			struct mailimap_date * senton,
			struct mailimap_date * sentsince,
			uint32_t smaller,
			struct mailimap_set * uid,
			struct mailimap_set * set,
			clist * multiple);


void mailimap_search_key_free(struct mailimap_search_key * key);

struct mailimap_status_att_list * mailimap_status_att_list_new(clist * list);

void mailimap_status_att_list_free(struct mailimap_status_att_list *
				   status_att_list);



/* main */



enum {
  MAILIMAP_STATE_DISCONNECTED,
  MAILIMAP_STATE_NON_AUTHENTICATED,
  MAILIMAP_STATE_AUTHENTICATED,
  MAILIMAP_STATE_SELECTED,
  MAILIMAP_STATE_LOGOUT
};

struct mailimap {
  char * response;               /* response message */
  
  /* internals */
  mailstream * stream;

  size_t progr_rate;
  progress_function * progr_fun;

  MMAPString * stream_buffer;        /* buffer for lines reading */
  MMAPString * response_buffer;      /* buffer for responses */

  int state;                     /* state */
  int tag;

  struct mailimap_connection_info * connection_info;
  struct mailimap_selection_info * selection_info;
  struct mailimap_response_info * response_info;
};

typedef struct mailimap mailimap;

enum {
  MAILIMAP_NO_ERROR = 0,
  MAILIMAP_NO_ERROR_AUTHENTICATED = 1,
  MAILIMAP_NO_ERROR_NON_AUTHENTICATED = 2,
  MAILIMAP_ERROR_BAD_STATE,
  MAILIMAP_ERROR_STREAM,
  MAILIMAP_ERROR_PARSE,
  MAILIMAP_ERROR_CONNECTION_REFUSED,
  MAILIMAP_ERROR_MEMORY,
  MAILIMAP_ERROR_FATAL,
  MAILIMAP_ERROR_PROTOCOL,
  MAILIMAP_ERROR_DONT_ACCEPT_CONNECTION,
  MAILIMAP_ERROR_APPEND,
  MAILIMAP_ERROR_NOOP,
  MAILIMAP_ERROR_LOGOUT,
  MAILIMAP_ERROR_CAPABILITY,
  MAILIMAP_ERROR_CHECK,
  MAILIMAP_ERROR_CLOSE,
  MAILIMAP_ERROR_EXPUNGE,
  MAILIMAP_ERROR_COPY,
  MAILIMAP_ERROR_UID_COPY,
  MAILIMAP_ERROR_CREATE,
  MAILIMAP_ERROR_DELETE,
  MAILIMAP_ERROR_EXAMINE,
  MAILIMAP_ERROR_FETCH,
  MAILIMAP_ERROR_UID_FETCH,
  MAILIMAP_ERROR_LIST,
  MAILIMAP_ERROR_LOGIN,
  MAILIMAP_ERROR_LSUB,
  MAILIMAP_ERROR_RENAME,
  MAILIMAP_ERROR_SEARCH,
  MAILIMAP_ERROR_UID_SEARCH,
  MAILIMAP_ERROR_SELECT,
  MAILIMAP_ERROR_STATUS,
  MAILIMAP_ERROR_STORE,
  MAILIMAP_ERROR_UID_STORE,
  MAILIMAP_ERROR_SUBSCRIBE,
  MAILIMAP_ERROR_UNSUBSCRIBE,
  MAILIMAP_ERROR_STARTTLS,
  MAILIMAP_ERROR_INVAL,
};

struct mailimap_connection_info {
  struct mailimap_capability_data * capability;
};

struct mailimap_connection_info *
mailimap_connection_info_new();

void
mailimap_connection_info_free(struct mailimap_connection_info * conn_info);

enum {
  MAILIMAP_MAILBOX_READONLY,
  MAILIMAP_MAILBOX_READWRITE
};

struct mailimap_selection_info {
  clist * perm_flags; /* list of (struct flag_perm *) */
  int perm;
  uint32_t uidnext;
  uint32_t uidvalidity;
  uint32_t unseen;
  struct mailimap_flag_list * flags;
  uint32_t exists;
  uint32_t recent;
};

struct mailimap_selection_info *
mailimap_selection_info_new();

void
mailimap_selection_info_free(struct mailimap_selection_info * sel_info);


struct mailimap_response_info {
  char * alert;
  char * parse;
  clist * badcharset;
  int trycreate;
  clist * mailbox_list; /* list of (struct mailimap_mailbox_list *) */
  clist * mailbox_lsub; /* list of (struct mailimap_mailbox_list *) */
  clist * search_result; /* list of (guint 32 *) */
  struct mailimap_mailbox_data_status * status;
  clist * expunged; /* list of (guint 32 *) */
  clist * fetch_list; /* list of (struct mailimap_msg_att *) */
};

struct mailimap_response_info *
mailimap_response_info_new();

void
mailimap_response_info_free(struct mailimap_response_info * resp_info);

#endif

