#  PMail - GNOME/GTK/Python email client
#  Copyright (C) 2000 Scott Bender <sbender@harmony-ds.com>

#  This file is part of PMail.

#  PMail is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2, or (at your option)
#  any later version.

#  PMail is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with GNU Emacs; see the file COPYING.  If not, write to
#  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
#  Boston, MA 02111-1307, USA.  

from gtk import *
import GtkExtra
import threading
import time

import mygnome

import config
import GladeWindow
import MessageList
import TreeWindow
import Preferences
import ProgressBar
import Compose
import gutil
import mail
import SearchMessages
import MsgWindow
import update
try:
  import _pmailextensions
except:
  pass

window = None

class MainWindow(GladeWindow.Window):
  def __init__(self):
    if mygnome.have_gnome:
      gladefile = 'gnomeapp.glade'
    else:
      gladefile = 'all.glade'
    GladeWindow.Window.__init__(self, gladefile)
    global window
    window = self
    #self._debug = 1

    self._started_up = 0
    self._progress_stack = []
    self._progress_stack_lock = threading.RLock()
    self._use_progress_popup = not mygnome.have_gnome
    self._check_lock = threading.RLock()

    w,h = config.getWindowParam(self, 'size', (650, 450))
    self._window.set_usize(w, h)
    self._window.set_title('%s %s' % (config.app_name, config.version))

    self._window.connect("destroy", self.do_quit)

    self._messageWindow = MsgWindow.create_message_window(self)
    self._messageList = MessageList.Window(self, self._messageWindow)
    self._treeWindow = TreeWindow.Window(self, self._messageList)
    self._messageList._treeWindow = self._treeWindow

    self._hpane.add1(self._treeWindow)
    self._hpane.set_position(config.getWindowParam(self, 'hSize', 150))
    self._treeWindow.show()

    self._hpane.add2(self._messageList)
    self._messageList.show()
    vSize = config.getWindowParam(self, 'vSize', 150)
    self._vpane.set_position(vSize)
    self._vpane.add2(self._messageWindow)
    self._messageWindow.show()

    sigs = { 'new_msg': self.new_message,
             'check_mail': self.check_mail,
             'delete': self._messageList.delete_mail,
             'preferences': self.do_prefs,
             'search': None,
             'quit': self.do_quit,
             'expunge': self._messageList.expunge,
             'reply': self.reply,
             'forward': self.forward,
             'search_mailbox': self.search_mailbox,
             'flag': self._messageList.flag_mail,
             'header_view': self.header_view,
             'about': self.about,
             }
    self._glade.signal_autoconnect(sigs)
    
    self._menubar = self._glade.get_widget_by_longname('MainWindow.dock1.dockitem1.menubar')

    plugins = mail.Plugin.get_plugins()

    items = filter(lambda x: x['name'] == 'plugins',
                   self._menubar.children())
    item = items[0]
    if len(plugins):
      sub = GtkMenu()
      item.set_submenu(sub)
      for plugin in plugins:
        plugin._ui = self
        name = plugin.name()
        menu = plugin.menu()
        if menu:
          sitem = GtkMenuItem(name)
          sitem.set_submenu(menu)
          sub.append(sitem)
          sitem.show()
    
    self._started_up = 1


  def selected_mailbox(self):
    return self._messageList.selected_mailbox()

  def selected_messages(self):
    return self._messageList.selected_messages()

  def check_mail(self, d=None):
    box = self._messageList.selected_mailbox()
    if box:
      force = 0
      for source in mail.get_sources():
        if isinstance(source, mail.POP.POPSource) \
           and source['storage_mailbox'] == box:
          force = source.check_mail() or force
      self._messageList.check_mail(force_list=force)

  def show(self):
    GladeWindow.Window.show(self)

    for source in mail.get_sources():
      update.update()
      if source['check_at_startup']:
        source.check_mail()
        self._treeWindow.update()

  def show_status(self, cntx, mesg):
    #c = self._status.get_context_id(cntx)
    #self._status.push(c, '   ' + mesg)
    if mygnome.have_gnome:
      self._status.push(mesg)

  def pop_status(self, cntx):
    #c = self._status.get_context_id(cntx)
    #self._status.pop(c)
    if mygnome.have_gnome:
      self._status.pop()

  def do_quit(self, _b=None, _e=None):
    self.store_config()
    self._messageWindow.remove_temp_files()
    mainquit()

  def do_prefs(self, _b=None, _e=None):
    Preferences.show(self)

  def about(self, _b):
    about = gnome.ui.GnomeAbout(config.app_name, config.version,
                                'Copyright 2000 Scott Bender',
                                ['Scott Bender <sbender@harmony-ds.com>'],
                                'http://www.scottbender.net/pmail')
                                
    about.set_parent(self._window)
    about.show()

  def new_message(self, b):
    Compose.show(self)

  def forward(self, b, msg=None, type=None):
    if not msg:
      msgs = self._messageList.selected_messages()
      if len(msgs) == 1:
        msg = msgs[0]
      else:
        return

    if not type:
      name = b['name']
      if name  == 'attachment':
        type = mail.ATTACHMENT
      elif name == 'inline':
        type = mail.INLINE
      elif name == 'quoted':
        type = mail.QUOTED
      else:
        type = mail.get_config('forward_type')
        
    Compose.forward(self, msg, type)

  def reply(self, b, msg=None, type=None):
    if not msg:
      msgs = self._messageList.selected_messages()
      if len(msgs) == 1:
        msg = msgs[0]
      else:
        return

    if not type:
      name = b['name']
      if name  == 'attachment':
        type = mail.ATTACHMENT
      elif name == 'inline':
        type = mail.INLINE
      elif name == 'quoted':
        type = mail.QUOTED
      else:
        type = mail.get_config('reply_type')
          
    Compose.reply(self, msg, type)

  def search_mailbox(self, b):
    SearchMessages.show(self, self._treeWindow._selected_mailbox)

  def preferences_changed(self):
    self._treeWindow.update_tree()

  def header_view(self, d):
    name = d['name']
    if name == 'normal':
      val = mail.NORMAL_HEADERS
    elif name == 'brief':
      val = mail.BRIEF_HEADERS
    elif name == 'all':
      val = mail.ALL_HEADERS
    mail.set_config('headerView', val)
    self._messageWindow.redisplay()
      

  def name(self):
    return "GtkMainWindow"

  def store_config(self):
    self._messageList.store_config()
    self._treeWindow.store_config()
    a = self._treeWindow.get_allocation()
    config.setWindowParam(self, 'hSize', a[2])
    a = self._hpane.get_allocation()
    config.setWindowParam(self, 'vSize', a[3])
    a = self._window.get_allocation()
    config.setWindowParam(self, 'size', (a[2], a[3]))

  def in_main_thread(self):
    return threading.currentThread().getName() == 'MainThread'

  def threads_enter(self):
    if not self.in_main_thread():
      threads_enter()

  def threads_leave(self):
    if not self.in_main_thread():
      threads_leave()

  def __latest_progess(self):
    if len(self._progress_stack):
      res = -1

      r = range(0, len(self._progress_stack))
      r.reverse()
      for i in r:
        if self._progress_stack[i] != None:
          res = i
          break
      return res
    else:
      return -1
    
  def start_progress(self, message, count):
    self.debug('start_progress: %d' % count)
    self.threads_enter()      
    
    if self._use_progress_popup:
      self._progress = ProgressBar.show('Progress',
                                        message, 
                                        count,
                                        self._window)
      res = self._progress
    else:
      self._status.push(message)
      self._status.set_progress(0)
      self._progress_stack_lock.acquire()
      idx = self.__latest_progess() + 1
      if idx >= len(self._progress_stack):
        self._progress_stack.append(count)
      else:
        self._progress_stack[idx] = count
      self._progress_stack_lock.release()
      res = idx

    self.threads_leave()
    if self.in_main_thread():
      update.update()
      
    self.debug('start_progress: done')
    return res

  def update_progress(self, num, progress):
    #self.debug('update progress: %d' % num)    
    self.threads_enter()      
      
    if self._use_progress_popup:
      progress.update(num)
    else:
      self._progress_stack_lock.acquire()
      if progress == self.__latest_progess():
        f = float(num)/float(self._progress_stack[progress])
        self._status.set_progress(f)
      self._progress_stack_lock.release()

    self.threads_leave()
    if self.in_main_thread():
      update.update()
        

  def progress_aborted(self, progress):
    if self._use_progress_popup:
      if not progress:
        progress = self._progress
      return progress.user_aborted()
    else:
      return 0
    

  def finish_progress(self, progress):
    self.debug('finish_progress')
    self.threads_enter()
    
    if self._use_progress_popup:
      progress.destroy()
    else:
      self._progress_stack_lock.acquire()
      if progress == self.__latest_progess():
        self._status.pop()
        self._status.set_progress(0)
        self._progress_stack[progress] = None
      self._progress_stack_lock.release()

    self.threads_leave()
    if self.in_main_thread():
      update.update()
    
  def prompt_password(self, source):
    self.threads_enter()
    res = gutil.get_password('Mail Server Password',
                             'Please enter the password for: %s' %
                             source.name(),
                             self._window)
    self.threads_leave()
    return res

  def alert(self, message):
    self.threads_enter()
    gutil.alert_box(message)
    self.threads_leave()

  def display_message(self, message):
    self._treeWindow.select_mailbox(message.mailbox())
    self._messageList.select_message(message)


  def check_thread_func(self, source):
    res = source.check_mail()
    #self.threads_enter()
    self._treeWindow.update()
    #self.threads_leave()
    return res

  def check_timer_func(self, source):
    #thread = threading.Thread(None, self.check_thread_func, None, (source,))
    #thread.start()

    if not self._started_up:
      return

    self._check_lock.acquire()
    #print 'checking %s at %s' % (source.name(), time.asctime(time.localtime(time.time())))
    
    threads_enter()
    res = self.check_thread_func(source)
    if res and source == self._messageList.selected_mailbox() \
       and self._messageList.selected_mailbox().source():
      self._messageList.list_messages()
    threads_leave()
    #print 'done checking %s at %s' % (source.name(), time.asctime(time.localtime(time.time())))    
    self._check_lock.release()
    return 1

  def add_check_timer(self, source, seconds):
    #seconds = 20
    tout = seconds*1000
    tag = timeout_add(tout, self.check_timer_func, source)
    return tag
    #return 1
    
  def remove_check_timer(self, tag):
    timeout_remove(tag)
    pass

