#  PMail - GNOME/GTK/Python email client
#  Copyright (C) 2000 Scott Bender <sbender@harmony-ds.com>

#  This file is part of PMail.

#  PMail is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2, or (at your option)
#  any later version.

#  PMail is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.

#  You should have received a copy of the GNU General Public License
#  along with GNU Emacs; see the file COPYING.  If not, write to
#  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
#  Boston, MA 02111-1307, USA.  


from gtk import *
import GtkExtra
import string
import re

import gtkf.MainWindow
import gtkf.GladeWindow
import mail.Plugin
import gtkf.gutil
import config
from Base import *

_filters = []
INCOMMING=1
OUTGOING=2
MANUAL=3

class Plugin(mail.Plugin.Plugin):
  def __init__(self):
    mail.Plugin.Plugin.__init__(self)
    self._filter_window = None

  def name(self):
    return 'Filters'

  def new_mail_arrived(self, messages):
    for filt in _filters:
      if filt['type'] == INCOMMING:
        filt.apply(messages)

  def new_mail_sent(self, message):
    for filt in _filters:
      if filt['type'] == OUTGOING:
        filt.apply([message])

  def menu(self):
    mf = GtkExtra.MenuFactory(GtkExtra.MENU_FACTORY_MENU)
    entries = [
      ('Edit...', None, self.edit),
      ]
    mf.add_entries(entries)
    menu =  mf.get_menu('')
    apply_menu = self._apply_filter_menu()
    item = GtkMenuItem('Apply Filter')
    item.set_submenu(apply_menu)
    menu.append(item)
    item.show()
    return menu

  def edit(self, d):
    if not self._filter_window:
      self._filter_window = FilterWindow()
    self._filter_window._window.show()

  def _apply_filter_menu(self, message=None, type=MANUAL):
    mf = GtkExtra.MenuFactory(GtkExtra.MENU_FACTORY_MENU)
    entries = []

    for filt in _filters:
      if not type or filt['type'] == type:
        entries.append((filt['name'], None, self.apply_filter, filt, message))
    mf.add_entries(entries)
    menu = mf.get_menu('')
    return menu
    
  def message_list_popup_items(self, message):
    item = GtkMenuItem('Apply Filter')
    item.set_submenu(self._apply_filter_menu(message))
    return [item]

  def apply_filter(self, d, filter, message=None):
    if not message:
      import gtkf
      main = gtkf.main_window()
      messages = main.selected_messages()
    else:
      messages = [message]

    filter.apply(messages)
    

  def application_initialized(self):
    pass

InvalidCriteria = 'InvalidCriteria'

def do_move(msg, value):
  box = mail.mailbox_from_config(value)
  if msg.append(box):
    msg.delete()

def do_copy(msg, value):
  box = mail.mailbox_from_config(value)
  msg.append(box)

def do_forward(msg, value):
  m = mail.NewMessage.NewMessage(string.split(value, ','),
                                 subject=mail.get_subject(msg, 'Fwd:'))
  m.start_message(1)
  m.add_messsage_attachment(msg)
  m.send()

def do_replywith(msg, value):
  to = message.get_header('Reply-to')
  if not to:
    to = message.get_header('From')    
  m = mail.NewMessage.NewMessage([to],
                                 subject=mail.get_subject(msg, 'Re:'))
  m.start_message(1)
  m.set_body(value)
  m.add_messsage_attachment(msg)
  m.send()

def do_pythoncode(message, value):
  x = compile(value, '<string>', 'exec')
  exec x

def run_regex(regex, string):
  try:
    regex = re.compile(regex)
    return regex.match(string, re.I)
  #except pcre.error:
  #  raise InvalidCriteria
  except re.error:
    raise InvalidCriteria, regex

def run_code(match, value):
  x = compile(match, '<string>', 'exec')
  exec x
  return result

_filter_operators = {'Contains': lambda s, v: string.find(v, s) != -1,
                     'Not Contains': lambda s, v: string.find(v, s) == -1,
                     'Equals': lambda s, v: s == v,
                     'Not Equals': lambda s, v: s != v,
                     'Starts With': lambda s, v: v[:len(s)] == s,
                     'Ends With': lambda s, v: v[:-len(s)] == s,
                     'Python Regex': run_regex,
                     'Python Code': run_code,
                    }
_sorted_operator_keys = list(_filter_operators.keys())
_sorted_operator_keys.sort()

_filter_fields = ['Subject', 'From', 'To', 'Body', 'CC', 'Python Code']
_filter_fields.sort()

FOLDER = 1
ADDRESS = 2
TEXT = 3
_filter_actions = {
  'Move To': [FOLDER, do_move],
  'Copy To': [FOLDER, do_copy],
  'Flag': [None, lambda msg, val: msg.set_flagged(1)],
  'Delete': [None, lambda msg, val: msg.delete()],
  'Forward To': [ADDRESS, do_forward],
  'Reply With': [TEXT, do_replywith],
  'Python Code': [TEXT, do_pythoncode],
  }
_sorted_actions = list(_filter_actions.keys())
_sorted_actions.sort()


class Filter(PMailObject):
  def __init__(self):
    PMailObject.__init__(self)
    #self._debug = 1
    self._criteria = None
    self._actions = None
    self._type = INCOMMING
    self._folder = None
    self._folder_config = None
    self._name = None
    self._enabled = 1

    self.add_config_params(['criteria', 'actions', 'type', 'name',
                            'folder_config', 'enabled'])

  def criteria_matches(self, message):
    res = 1
    for c in self._criteria:
      field = c['field']
      if field == 'Body':
        value = message.body()
      elif field == 'Python Code':
        value = None
      else:
        value = message.get_header(field)

      bool = c.get('bool')

      operator = c['operator']

      if field == 'Python Code':
        x = compile(c['value'], '<string>', 'exec')
        exec x
        r = result
      else:
        try:
          f = _filter_operators.get(operator)
          r =  f(c['value'], value)
        except InvalidCriteria, x:
          gtkf.gutil.alert_box('Invalid Filter Criteria: \'%s\'' % x)
          return 0

      if bool == 2:
        #Not
        if r:
          return 0
        else:
          r = 1
      if bool == 1 and res:
        return 1
      elif bool == 1: #start over
        res = 1
        
      if res == 1:
        res = r
      
    return res

  def perform_actions(self, message):
    #make sure we to move or delete before anything else
    actions = filter(lambda c: c['action'] != 'Delete'
                     and c['action'] != 'Move To', self._actions)
    actions = actions + filter(lambda c: c['action'] == 'Delete'
                              or c['action'] == 'Move To', self._actions)

    for act in actions:
      action = act['action']
      f = _filter_actions.get(action)[1]
      if self._debug:
        self.debug('performing %s/%s on message(%s)' %
                   (action, act.get('value'), message.get_header('subject')))
      f(message, act.get('value'))
    

  def apply(self, messages):
    if not self._enabled:
      return
    
    self.debug('Running filter \'%s\'' % self._name)
    for msg in messages:
      if (self._type == MANUAL or not msg.seen()) \
         and (self['folder'] == None or self._folder == msg.mailbox()):
        if self.criteria_matches(msg):
          self.debug('message(%s) matched criteria for \'%s\'' %
                     (msg.get_header('subject'), self._name))
          #msg.set_seen(1)
          self.perform_actions(msg)

  def __getitem__(self, key):
    if key == 'folder' \
       and self._folder == None \
       and self._folder_config:
      self._folder = mail.mailbox_from_config(self._folder_config)
    return PMailObject.__getitem__(self, key)

  def dump(self):
    print 'name:', self._name
    print 'enabled:', self._enabled
    print 'type:', self._type
    print 'folder:', self._folder

    print 'criteria:'
    for c in self._criteria:
      print '  {'
      print '    bool:', c.get('bool')
      print '    field:', c['field']
      print '    operator:', c['operator']
      print '    value:', c['value']
      print '  }'

    print 'actions:'
    for c in self._actions:
      print '  {'
      print '    action:', c['action']
      print '    value:', c.get('value')
      print '  }'


class FilterWindow(gtkf.GladeWindow.Window):
  def __init__(self):
    gtkf.GladeWindow.Window.__init__(self, 'filters.glade', 'window')

    self._debug = 1

    sigs = { 'more_criteria': self.more_criteria,
             'less_criteria': self.less_criteria,
             'more_actions': self.more_actions,
             'less_actions': self.less_actions,
             'update_filter': self.update_filter,
             'add_filter': self.add_filter,
             'select_filter': self.select_filter,
             'close': self.close,
             'delete_filter': self.delete_filter,
             'clear': self.clear,
             }
    self._glade.signal_autoconnect(sigs)

    self._bools = []
    self._fields = []
    self._operators = []
    self._values = []
    self._criteria_hboxes = []

    self._actions = []
    self._action_values = []
    self._action_hboxes = []

    self._current_filter = None

    w,h = config.getWindowParam(self, 'size', (650, 450))
    self._window.set_usize(w, h)
    
    gtkf.gutil.folder_option_menu(self._folder, do_none=0)
    
    self.add_criteria_widgets()
    self.add_actions_widgets()
    self.update_filter_list()

  def clear(self, d):
    self._name.set_text('')
    self._values[0].set_text('')

    while (len(self._actions) > 1):
      self.less_actions()

    while (len(self._bools) > 1):
      self.less_criteria()

  def close(self, d):
    self._window.hide()

  def name(self):
    return 'Filters'

  def select_filter(self, d, e):
    idx = self._filter_list.child_position(e)
    self._current_filter = _filters[idx]
    self.set_values()
    

  def add_filter(self, d):
    self._current_filter = Filter()
    if self.get_values():
      _filters.append(self._current_filter)
      self.update_filter_list()

  def delete_filter(self, d):
    if self._current_filter:
      _filters.remove(self._current_filter)
      self.update_filter_list()

  def update_filter(self, d):
    if self._current_filter:
      self.get_values()
      self.update_filter_list()

  def get_values(self):
    filt = self._current_filter

    if self._incomming.get_active():
      type = INCOMMING
    elif self._outgoing.get_active():
      type = OUTGOING
    elif self._manual.get_active():
      type = MANUAL

    filt['type'] = type
    filt['enabled'] = self._enabled.get_active()

    if type == INCOMMING:
      idx, item = gtkf.gutil.selected_menu_item(self._folder.get_menu())
      f = item.get_data('folder')
      filt['folder'] = f
      filt['folder_config'] = f \
                              and mail.mailbox_config_value(f) \
                              or None

      if not f:
        gtkf.gutil.alert_box('Please select a folder for the filter',
                             self._window)
        return 0
    else:
      filt['folder'] = None
      filt['folder_config'] = None
      
    filt['name'] = self._name.get_text()

    if not filt['name']:
      gtkf.gutil.alert_box('Please enter a name for the filter', self._window)
      return 0

    filt['criteria'] = []
    for i in range(len(self._bools)):
      c = {}
      if i > 0:
        c['bool'], skip = gtkf.gutil.selected_menu_item(
          self._bools[i].get_menu())
      idx, item = gtkf.gutil.selected_menu_item(self._fields[i].get_menu())
      c['field'] = item.get_data('data')
      idx, item = gtkf.gutil.selected_menu_item(self._operators[i].get_menu())
      c['operator'] = item.get_data('data')
      c['value'] = self._values[i].get_text()
      if not len(c['value']):
        gtkf.gutil.alert_box('Please enter a value for your criteria',
                             self._window)
        return 0
      filt['criteria'].append(c)

    filt['actions'] = []
    for i in range(len(self._actions)):
      c = {}
      idx, item = gtkf.gutil.selected_menu_item(self._actions[i].get_menu())
      c['action'] = item.get_data('data')
      type = _filter_actions.get(c['action'])[0]
      if type == ADDRESS:
        c['value'] = self._action_values[i].get_text()
      elif type == TEXT:
        c['value'] = self._action_values[i].get_chars(0, -1)
      elif type == FOLDER:
        idx, item = gtkf.gutil.selected_menu_item(self._action_values[i].get_menu())
        f = item.get_data('folder')
        if f:
          c['value'] = mail.mailbox_config_value(f)

      if type != None and not c.get('value'):
        gtkf.gutil.alert_box('Please select a value for your \'%s\' Action.'
                             % c['action'], self._window)
        return 0
        
      filt['actions'].append(c)

    if self._debug:
      self._current_filter.dump()

    return 1

  def set_values(self):
    filt = self._current_filter

    if filt['type'] == INCOMMING:
      self._incomming.set_active(1)
    elif filt['type'] == OUTGOING:
      self._outgoing.set_active(1)
    else:
      self._manual.set_active(1)

    gtkf.gutil.select_folder_option(self._folder, filt['folder'])
    self._name.set_text(filt['name'])
    self._enabled.set_active(filt['enabled'])

    for i in range(len(filt['criteria'])):
      c = filt['criteria'][i]
      if i >= len(self._bools):
        self.add_criteria_widgets()
      if i > 0:
        self._bools[i].set_history(c['bool'])

      self._fields[i].set_history(_filter_fields.index(c['field']))
      self._operators[i].set_history(_sorted_operator_keys.index(c['operator']))
      self._values[i].set_text(c['value'])
    
    while len(self._bools) > len(filt['criteria']):
      self.less_criteria()

    for i in range(len(filt['actions'])):
      c = filt['actions'][i]
      
      if i >= len(self._actions):
        self.add_actions_widgets()

      self._actions[i].set_history(_sorted_actions.index(c['action']))
      self.action_selected(self._actions[i].get_menu())
      type = _filter_actions.get(c['action'])[0]
      if type == ADDRESS:
        self._action_values[i].set_text(c['value'])
      elif type == TEXT:
        self._action_values[i].insert_defaults(c['value'])
      elif type == FOLDER:
        box = mail.mailbox_from_config(c['value'])
        gtkf.gutil.select_folder_option(self._action_values[i], box)
                   
    while len(self._actions) > len(filt['actions']):
      self.less_actions()
  
  def update_filter_list(self):
    self._filter_list.clear_items(0, -1)

    items = []
    for filt in _filters:
      item = GtkListItem(label=filt['name'])
      item.show()
      items.append(item)
      
    if len(items):
      self._filter_list.append_items(items)
      

  def more_criteria(self, d):
    self.add_criteria_widgets()

  def less_criteria(self, d=None):
    row = len(self._bools)-1
    if row > 0:
      self._criteria_box.remove(self._criteria_hboxes[row])
      del self._criteria_hboxes[row]
      del self._bools[row]
      del self._fields[row]
      del self._operators[row]
      del self._values[row]

  def more_actions(self, d):
    self.add_actions_widgets()

  def less_actions(self, d=None):
    row = len(self._actions)-1
    if row > 0:
      self._action_box.remove(self._action_hboxes[row])
      del self._action_hboxes[row]
      del self._actions[row]
      del self._action_values[row]

  def action_selected(self, menu):
    idx, item = gtkf.gutil.selected_menu_item(menu)    
    type = _filter_actions.get(_sorted_actions[idx])[0]
    w = None
    if type == FOLDER:
      w = GtkOptionMenu()
      gtkf.gutil.folder_option_menu(w, do_none=0)
    elif type == ADDRESS:
      w = GtkEntry()
    elif type == TEXT:
      w = GtkText()
      w.set_editable(1)

    action_idx = menu.get_data('row')
    if self._action_values[action_idx] != None:
      self._action_hboxes[action_idx].remove(self._action_values[action_idx])
      self._action_values[action_idx] = None
      
    if w:
      #self._action_hboxes[action_idx].pack_end(w)
      self._action_hboxes[action_idx].attach(w, 1, 2, 0, 1, yoptions=0)
      self._action_values[action_idx] = w
      w.show()

  def add_criteria_widgets(self):
    row = len(self._bools)

    self._bools.append(GtkOptionMenu())
    self._fields.append(GtkOptionMenu())
    self._operators.append(GtkOptionMenu())
    self._values.append(GtkEntry())

    hbox = GtkHBox()
    self._criteria_hboxes.append(hbox)
    self._criteria_box.pack_start(hbox)

    gtkf.gutil.fill_option_menu(self._bools[row], ['And', 'Or', 'Not'])
    gtkf.gutil.fill_option_menu(self._fields[row], _filter_fields,
                                _filter_fields)
    gtkf.gutil.fill_option_menu(self._operators[row], _sorted_operator_keys,
                                _sorted_operator_keys)

    hbox.pack_start(self._bools[row])
    hbox.pack_start(self._fields[row])
    hbox.pack_start(self._operators[row])
    hbox.pack_start(self._values[row])
    hbox.show()

    self._bools[row].show()
    self._fields[row].show()
    self._operators[row].show()
    self._values[row].show()

    self._bools[row].set_history(0)
    self._fields[row].set_history(0)
    self._operators[row].set_history(0)
    if row == 0:
      self._bools[row].set_sensitive(0)

  def add_actions_widgets(self):
    row = len(self._actions)

    self._actions.append(GtkOptionMenu())
    self._action_values.append(None)

    #hbox = GtkHBox()
    hbox = GtkTable(1, 2)
    self._action_hboxes.append(hbox)
    self._action_box.pack_start(hbox)

    gtkf.gutil.fill_option_menu(self._actions[row], _sorted_actions,
                                _sorted_actions)
    self._actions[row].set_history(0)

    #hbox.pack_start(self._actions[row])
    hbox.attach(self._actions[row], 0, 1, 0, 1, yoptions=0)
    hbox.show()

    self._actions[row].get_menu().connect('selection_done',
                                          self.action_selected)
    self._actions[row].get_menu().set_data('row', row)

    self._actions[row].show()

def save_config():
  dicts = []
  for filt in _filters:
    dicts.append(filt.get_config())
  config.setParam(None, 'Mail', 'filters', dicts)  
    
def load_config():
  dicts = config.getParam(None, 'Mail', 'filters', None)

  if dicts:
    for dict in dicts:
      filt = Filter()
      filt.read_config(dict)
      _filters.append(filt)

#print 'Registering Filter plugin'
config.register_config_hook(None, load_config, save_config, None)
mail.Plugin.register_plugin(Plugin())
