#
# encode.rb
#
#   Copyright (c) 1998-2002 Minero Aoki <aamine@loveruby.net>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'nkf'
require 'tmail/base64'
require 'tmail/stringio'
require 'tmail/utils'


module TMail

  module StrategyInterface

    def create_dest( obj )
      case obj
      when nil
        StringOutput.new
      when String
        StringOutput.new(obj)
      when IO, StringOutput
        obj
      else
        raise TypeError, 'cannot handle this type of object for dest'
      end
    end
    module_function :create_dest

    def encoded( eol = "\r\n", charset = 'j', dest = nil )
      accept_strategy HFencoder, eol, charset, dest
    end

    def decoded( eol = "\n", charset = 'e', dest = nil )
      accept_strategy HFdecoder, eol, charset, dest
    end

    alias to_s decoded
  
    def accept_strategy( klass, eol, charset, dest0 )
      dest = create_dest( dest0 ||= '' )
      strategy = klass.new( dest, charset )
      accept strategy, dest
      dest0
    end

  end


  ###
  ### decode
  ###

  class HFdecoder

    encoded = '=\?(?:iso-2022-jp|euc-jp|shift_jis)\?[QB]\?[a-z0-9+/=]+\?='
    ENCODED_WORDS = /#{encoded}(?:\s+#{encoded})*/i

    OUTPUT_ENCODING = {
      'EUC'  => 'e',
      'SJIS' => 's',
    }

    def self.decode( str, encoding = nil )
      encoding ||= (OUTPUT_ENCODING[$KCODE] || 'j')
      opt = '-m' + encoding
      str.gsub( ENCODED_WORDS ) {|s| NKF.nkf(opt, s) }
    end


    def initialize( dest, code = nil )
      @f = StrategyInterface.create_dest(dest)
      @encoding = (/\A[ejs]/ === code) ? code[0,1] : nil
    end

    def decode( str )
      type.decode( str, @encoding )
    end
    private :decode

    def terminate
    end

    def header_line( str )
      @f << decode(str)
    end

    def header_name( nm )
      @f << nm << ': '
    end

    def header_body( str )
      @f << decode(str)
    end
      
    def space
      @f << ' '
    end

    alias spc space

    def lwsp( str )
      @f << str
    end
      
    def meta( str )
      @f << str
    end

    def text( str )
      @f << decode(str)
    end

    alias phrase text

    def kv_pair( k, v )
      @f << k << '=' << v
    end

  end


  ###
  ### encode
  ###

  class HFencoder

    include TextUtils

    unless defined? BENCODE_DEBUG then
      BENCODE_DEBUG = false
    end

    def self.encode( str )
      e = new()
      e.header_body str
      e.terminate
      e.dest.string
    end

    SPACER       = "\t"
    MAX_LINE_LEN = 70

    def initialize( dest = nil, encoding = nil, limit = nil )
      @f     = StrategyInterface.create_dest(dest)
      @opt   = '-j'  #'-' + (/\A[ejs]/ === encoding ? encoding[0,1] : 'j')
      reset
    end

    def reset
      @text = ''
      @lwsp = ''
      @curlen = 0
    end

    def terminate
      add_lwsp ''
      reset
    end

    def dest
      @f
    end


    #
    # add
    #

    def header_line( line )
      scanadd line
    end

    def header_name( name )
      add_text name.split('-').map {|i| i.capitalize }.join('-')
      add_text ':'
      add_lwsp ' '
    end

    def header_body( str )
      scanadd NKF.nkf(@opt, str)
    end

    def space
      add_lwsp ' '
    end

    alias spc space

    def lwsp( str )
      add_lwsp str.sub(/[\r\n]+[^\r\n]*\z/, '')
    end

    def meta( str )
      add_text str
    end

    def text( str )
      scanadd NKF.nkf(@opt, str)
    end

    def phrase( str )
      str = NKF.nkf(@opt, str)
      if CONTROL_CHAR === str then
        scanadd str
      else
        add_text quote_phrase(str)
      end
    end

    def kv_pair( k, v )   ### line folding is not used yet
      v = NKF.nkf( @opt, v )

      if token_safe? v then
        add_text k + '=' + v

      elsif not CONTROL_CHAR === v then
        add_text k + '=' + quote_token(v)
      
      else
        # RFC2231 encoding
        kv = k + '*=' + "iso-2022-jp'ja'" + encode_value(v)
        add_text kv
      end
    end

    def encode_value( str )
      str.gsub( TOKEN_UNSAFE ) {|s| '%%%02x' % s[0] }
    end


    private


    MPREFIX = '=?iso-2022-jp?B?'
    MSUFFIX = '?='

    ESC_ASCII     = "\e(B"
    ESC_ISO2022JP = "\e$B"


    def scanadd( str, force = false )
      types = ''
      strs = []

      until str.empty? do
        if m = /\A[^\e\t\r\n ]+/.match(str) then
          types << (force ? 'j' : 'a')
          strs.push m[0]

        elsif m = /\A[\t\r\n ]+/.match(str) then
          types << 's'
          strs.push m[0]

        elsif m = /\A\e../.match(str) then
          esc = m[0]
          str = m.post_match
          if esc != ESC_ASCII and m = /\A[^\e]+/.match(str) then
            types << 'j'
            strs.push m[0]
          end

        else
          raise 'TMail FATAL: encoder scan fail'
        end
        str = m.post_match
      end

      do_encode types, strs
    end

    def do_encode( types, strs )
      #
      # sbuf = (J|A)(J|A|S)*(J|A)
      #
      #   A: ascii only, no need to encode
      #   J: jis, etc. need to encode
      #   S: LWSP
      #
      # (J|A)*J(J|A)* -> W
      # W(SW)*        -> E
      #
      # result = (A|E)(S(A|E))*
      #
      if BENCODE_DEBUG then
        puts "\n-- do_encode ------------"
        puts types.split(//).join(' ')
        p strs
      end

      e = /[ja]*j[ja]*(?:s[ja]*j[ja]*)*/

      while m = e.match(types) do
        pre = m.pre_match
        unless pre.empty? then
          concat_a_s pre, strs[ 0, pre.size ]
        end

        concat_e m[0], strs[ m.begin(0) ... m.end(0) ]

        types = m.post_match
        strs.slice! 0, m.end(0)
      end
      concat_a_s types, strs
    end

    def concat_a_s( types, strs )
      i = 0
      types.each_byte do |t|
        case t
        when ?a then add_text strs[i]
        when ?s then add_lwsp strs[i]
        else
          raise "TMail FATAL: unknown flag: #{t.chr}"
        end
        i += 1
      end
    end
    
    def concat_e( types, strs )
      if BENCODE_DEBUG then
        puts '---- concat_e'
        puts "types=#{types.split('').join(' ')}"
        puts "strs =#{strs.inspect}"
      end

      raise "TMail FATAL: exist: #{@text.inspect}" unless @text.empty?

      chunk = ''
      strs.each_with_index do |s,i|
        m = 'extract_' + types[i,1]
        until s.empty? do
          unless c = send(m, chunk.size, s) then
            add_with_encode chunk unless chunk.empty?
            flush
            chunk = ''
            fold
            c = __send__( m, 0, s )
            raise 'TMail FATAL: extract fail' unless c
          end
          chunk << c
        end
      end
      add_with_encode chunk unless chunk.empty?
    end

    def extract_j( csize, str )
      size = maxbyte( csize, str.size ) - 6
      size = size % 2 == 0 ? size : size - 1
      return nil if size <= 0

      c = str[ 0, size ]
      str[ 0, size ] = ''
      c = ESC_ISO2022JP + c + ESC_ASCII
      c
    end

    def extract_a( csize, str )
      size = maxbyte( csize, str.size )
      return nil if size <= 0

      c = str[ 0, size ]
      str[ 0, size ] = ''
      c
    end

    alias extract_s extract_a

    def maxbyte( csize, ssize )
      rest = restsize - MPREFIX.size - MSUFFIX.size
      rest / 4 * 3 - csize
    end

    #def encsize( len )
    #  amari = (if len % 3 == 0 then 0 else 1 end)
    #  (len / 3 + amari) * 4 + PRESIZE
    #end


    #
    # length-free buffer
    #

    def add_text( str )
      @text << str
#puts '---- text -------------------------------------'
#puts "+ #{str.inspect}"
#puts "txt >>>#{@text.inspect}<<<"
    end

    def add_with_encode( str )
      @text << MPREFIX << Base64.encode(str) << MSUFFIX
    end

    def add_lwsp( lwsp )
#puts '---- lwsp -------------------------------------'
#puts "+ #{lwsp.inspect}"
      if restsize <= 0 then
        fold
      end
      flush
      @lwsp = lwsp
    end

    def flush
#puts '---- flush ----'
#puts "spc >>>#{@lwsp.inspect}<<<"
#puts "txt >>>#{@text.inspect}<<<"
      @f << @lwsp << @text
      @curlen += @lwsp.size + @text.size
      @text = ''
      @lwsp = ''
    end

    def fold
#puts '---- fold ----'
      @f.puts
      @curlen = 0
      @lwsp = SPACER
    end

    def restsize
      MAX_LINE_LEN - (@curlen + @lwsp.size + @text.size)
    end

  end   # class HFencoder

end    # module TMail
