#
# mbox.rb
#
#   Copyright (c) 1998-2002 Minero Aoki <aamine@loveruby.net>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'tmail/port'
require 'socket'


module TMail

  class MhMailbox

    PORT_CLASS = MhPort

    def initialize( dir )
      edir = File.expand_path(dir)
      raise ArgumentError, "not directory: #{dir}"\
                              unless FileTest.directory? edir
      @dirname = edir
      @last_file = nil
      @last_atime = nil
    end

    def directory
      @dirname
    end

    attr_accessor :last_atime

    def inspect
      "#<#{self.class} #{@directory}>"
    end

    def close
    end

    def new_port
      PORT_CLASS.new(next_file_name())
    end

    def each_port
      mail_files().each do |path|
        yield PORT_CLASS.new(path)
      end
      @last_atime = Time.now
    end

    alias each      each_port

    def reverse_each_port
      mail_files().reverse_each do |path|
        yield PORT_CLASS.new(path)
      end
      @last_atime = Time.now
    end

    alias reverse_each reverse_each_port

    def each_new_port( mtime = nil, &block )
      mtime ||= @last_atime
      return each_port(&block) unless mtime
      return unless File.mtime(@dirname) >= mtime

      mail_files().each do |path|
        yield PORT_CLASS.new(path) if File.mtime(path) > mtime
      end
      @last_atime = Time.now
    end

    private

    def mail_files
      Dir.entries(@dirname)\
              .select {|s| /\A\d+\z/ === s }\
              .map {|s| s.to_i }\
              .sort\
              .map {|i| "#{@dirname}/#{i}" }\
              .select {|path| FileTest.file? path }
    end

    def next_file_name
      unless n = @last_file then
        n = 0
        Dir.entries(@dirname)\
                .select {|s| /\A\d+\z/ === s }\
                .map {|s| s.to_i }.sort\
        .each do |i|
          next unless FileTest.file? "#{@dirname}/#{i}"
          n = i
        end
      end
      begin
        n += 1
      end while FileTest.exist? "#{@dirname}/#{n}"
      @last_file = n

      "#{@dirname}/#{n}"
    end

  end   # MhMailbox

  MhLoader = MhMailbox



  class UNIXMbox

    class << self

      def create_from_line( port )
        sprintf 'From %s %s',
                fromaddr(), TextUtils.time2str(File.mtime(port.filename))
      end

      private

      def fromaddr
        h = HeaderField.new_from_port(port, 'Return-Path') ||
            HeaderField.new_from_port(port, 'From')
        h or return 'nobody'
        a = h.addrs[0] or return 'nobody'
        a.spec
      end

    end


    def initialize( fname, tmpdir = '/tmp' )
      @filename = fname

      Dir.mkdir tmpdir + "/tmail_mbox_#{$$}_#{id}"
      @real = MHMailbox.new(tmpdir)
      @closed = false

      @finalizer = UNIXMbox.mkfinal( @real, @filename )
      ObjectSpace.define_finalizer self, @finalizer
    end

    def UNIXMbox.mkfinal( ld, mbfile )
      lambda {
          File.open( mbfile, 'w' ) {|f|
              ld.each do |port|
                f.puts get_fromline(port)
                port.ropen {|r|
                    f.puts r.read
                }
              end
          }

          Dir.foreach(ld.dirname) do |fname|
            next if /\A\.\.?\z/ === fname
            File.unlink ld.dirname + '/' + fname
          end
          Dir.unlink ld.dirname
      }
    end

    def close
      return if @closed
      
      ObjectSpace.undefine_finalizer self
      @finalizer.call
      @finalizer = nil
      @real = nil
      @closed = true
    end

    def each_port( &block )
      close_check
      update
      @real.each_port(&block)
    end

    alias each each_port

    def reverse_each_port( &block )
      close_check
      update
      @real.reverse_each_port(&block)
    end

    alias reverse_each reverse_each_port

    def each_new_port( mtime = nil )
      close_check
      update
      @real.each_new_port( mtime ) {|p| yield p }
    end

    def new_port
      close_check
      @real.new_port
    end

    private

    FROMLINE = /\AFrom \S+ \w+ (\w+) (\d+) (\d+):(\d+):(\d+) (\d+)/

    def update
      return if FileTest.zero? @filename
      wf = t = p = nil

      lock( @filename ) {|f|
        begin
          f.each do |line|
            if /\AFrom / === line then
              wf.close if wf
              File.utime t, t, p.filename if t

              p = @real.new_port
              wf = p.wopen
              if m = FROMLINE.match(line) then
                t = Time.local( m[6].to_i, m[1], m[2].to_i,
                                m[3].to_i, m[4].to_i, m[5].to_i )
              else
                t = nil
              end
            else
              wf << line if wf
            end
          end
        ensure
          if wf and not wf.closed? then
            wf.close
            File.utime t, t, p.filename if t
          end
        end
      }
      File.truncate @filename, 0
    end
  
    def lock( fname )
      begin
        f = File.open( fname )
        f.flock File::LOCK_EX
        yield f
      ensure
        f.flock File::LOCK_UN
        f.close if f and not f.closed?
      end
    end

    def close_check
      @closed and raise ArgumentError, 'accessing already closed mbox'
    end

  end   # UNIXMbox

  MboxLoader = UNIXMbox



  class Maildir

    PORT_CLASS = MaildirPort

    def initialize( dir = nil )
      @dirname = dir || ENV['MAILDIR']
      raise ArgumentError, "not directory: #{@dirname}"\
                              unless FileTest.directory? @dirname
      @new = "#{@dirname}/new"
      @tmp = "#{@dirname}/tmp"
      @cur = "#{@dirname}/cur"
    end

    def directory
      @dirname
    end

    def inspect
      "#<#{self.class} #{@dirname}>"
    end

    def close
    end

    def each_port
      mail_files(@cur).each do |path|
        yield PORT_CLASS.new(path)
      end
    end

    alias each each_port

    def reverse_each_port
      mail_files(@cur).reverse_each do |path|
        yield PORT_CLASS.new(path)
      end
    end

    alias reverse_each reverse_each_port

    def new_port
      fn = nil
      tmp = nil
      i = 0

      while true do
        fn = "#{Time.now.to_i}.#{$$}.#{Socket.gethostname}"
        tmp = "#{@tmp}/#{fn}"
        break unless FileTest.exist? tmp
        i += 1
        raise IOError, "can't create new file in maildir" if i == 3
        sleep 1
      end
      File.open(tmp, 'w') {|f| f.write "\n\n" }
      cur = "#{@cur}/#{fn}"
      File.rename tmp, cur

      PORT_CLASS.new(cur)
    end

    def each_new_port
      mails_files(@new).each do |path|
        dest = @cur + '/' + File.basename(path)
        File.rename path, dest
        yield PORT_CLASS.new(dest)
      end

      check_tmp
    end


    TOO_OLD = 60 * 60 * 36   # 36 hour

    def check_tmp
      old = Time.now.to_i - TOO_OLD
      
      each_filename(@tmp) do |full, fname|
        if FileTest.file? full and
           File.stat(full).mtime.to_i < nt
          File.unlink full
        end
      end
    end

    private

    def mail_files( dir )
      Dir.entries(dir)\
              .select {|s| s[0] != ?. }\
              .sort_by {|s| s.slice(/\A\d+/).to_i }\
              .map {|s| "#{dir}/#{s}" }\
              .select {|path| FileTest.file? path }
    end

    def each_filename( dir )
      Dir.foreach(dir) do |fname|
        path = "#{dir}/#{fname}"
        if fname[0] != ?. and FileTest.file? path then
          yield path, fname
        end
      end
    end
    
  end   # Maildir

  MaildirLoader = Maildir

end   # module TMail
