#
# facade.rb
#
#   Copyright (c) 1998-2001 Minero Aoki <aamine@loveruby.net>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#


module TMail

  class Mail

    ###
    ### attributes
    ###

    class << self
    
      def tmail_attr_reader( header, name, getter )
        module_eval s = <<-"}", __FILE__, __LINE__ + 1
          def #{name.id2name}( default = nil )
            if header = @header['#{header.downcase}'] then
              header.#{getter}
            else
              default
            end
          end
        }
      end

      def tmail_attr_writer( header, name, setter = 'addrs.replace' )
        module_eval <<-"}", __FILE__, __LINE__ + 1
          def #{name.id2name}=( *arr )
            store '#{header}', ''
            @header['#{header.downcase}'].#{setter} arr.flatten
            arr
          end
        }
      end

      def tmail_s_attr_writer( header, name = nil )
        name = name ? name.id2name : header.downcase.gsub(/-/, '_')
        module_eval <<-"}", __FILE__, __LINE__ + 1
          def #{name}=( str )
            if str then
              store '#{header}', str
            else
              @header.delete '#{header.downcase}'
            end
            str
          end
        }
      end

      def tmail_sa_attr_writer( header )
        name = header.downcase.gsub(/-/, '_')
        module_eval <<-"}", __FILE__, __LINE__ + 1
          def #{name}=( *strs )
            strs.flatten!
            if strs.empty? then
              @header.delte '#{header.downcase}'
            else
              store '#{header}', strs.join(', ')
            end
            strs
          end
        }
      end
    
    end


    #
    # date time
    #

    tmail_attr_reader 'Date', :date, 'date'

    def date=( time )
      if time then
        store 'Date', ''
        @header['date'].date = time
      else
        @header.delete 'date'
      end
      time
    end

    def strftime( fmt, default = nil )
      if t = date then
        t.strftime(fmt)
      else
        default
      end
    end


    #
    # destination
    #

    tmail_attr_reader 'To',  :to_addrs,  'addrs'
    tmail_attr_reader 'Cc',  :cc_addrs,  'addrs'
    tmail_attr_reader 'Bcc', :bcc_addrs, 'addrs'

    tmail_attr_writer 'To',  :to_addrs
    tmail_attr_writer 'Cc',  :cc_addrs
    tmail_attr_writer 'Bcc', :bcc_addrs

    def to( default = nil )
      addr, = to_addrs(nil)
      addr || default
    end

    tmail_sa_attr_writer 'To'
    tmail_sa_attr_writer 'Cc'
    tmail_sa_attr_writer 'Bcc'


    #
    # originator
    #

    tmail_attr_reader 'From', :from_addrs, 'addrs'
    tmail_attr_writer 'From', :from_addrs

    def from( default = nil )
      if a = from_addr then
        a.phrase || a.address
      else
        default
      end
    end

    tmail_sa_attr_writer 'From'

    # obsolete
    def from_addr( default = nil )
      addr, = from_addrs(nil)
      addr || default
    end

    # obsolete
    def from_address( default = nil )
      if a = from_addr(nil) then
        a.address
      else
        default
      end
    end

    # obsolete
    alias from_address= from_addrs=

    # obsolete
    def from_phrase( default = nil )
      if a = from_addr(nil) then
        a.phrase
      else
        default
      end
    end


    tmail_attr_reader 'Reply-To', :reply_to_addrs, 'addrs'
    tmail_attr_writer 'Reply-To', :reply_to_addrs
    tmail_sa_attr_writer 'Reply-To'


    tmail_attr_reader 'Sender', :sender, 'addr'

    def sender( default )
      if f = @header['sender'] then
        if a = f.addr then
          return a.address
        end
      end
      default
    end

    tmail_s_attr_writer 'Sender'


    #
    # subject
    #

    tmail_attr_reader 'Subject', :subject, 'body'
    tmail_s_attr_writer 'Subject'


    #
    # identity & threading
    #

    tmail_attr_reader 'Message-Id', :message_id, 'msgid'
    tmail_s_attr_writer 'Message-Id'

    alias msgid  message_id
    alias msgid= message_id=

    tmail_attr_reader 'In-Reply-To', :in_reply_to, 'msgids'
    tmail_sa_attr_writer 'In-Reply-To'

    tmail_attr_reader 'References', :references, 'refs'
    tmail_sa_attr_writer 'References'


    #
    # MIME headers
    #

    tmail_attr_reader 'Mime-Version', :mime_version, 'version'

    def mime_version=( m, opt = nil )
      if opt then
        if f = @header['mime-version'] then
          f.major = m
          f.minor = opt
        else
          store 'Mime-Version', "#{m}.#{opt}"
        end
      else
        store 'Mime-Version', m
      end
      m
    end


    tmail_attr_reader 'Content-Type', :main_type, 'main'
    tmail_attr_reader 'Content-Type', :sub_type,  'sub'

    def set_content_type( str, sub = nil, param = nil )
      if sub then
        main, sub = str, sub
      else
        main, sub = str.split('/', 2)
      end
      if f = @header['content-type'] then
        f.main = main
        f.sub  = sub
      else
        store 'Content-Type', "#{main}/#{sub}"
      end
      @header['content-type'].params.replace param if param

      str
    end

    alias content_type= set_content_type

    tmail_attr_reader 'Content-Type', :charset, "params['charset']"

    def charset=( str )
      if str then
        if f = @header[ 'content-type' ] then
          f.params['charset'] = str
        else
          store 'Content-Type', "text/plain; charset=#{str}"
        end
      end
      str
    end


    tmail_attr_reader 'Content-Transfer-Encoding', :transfer_encoding, 'encoding'
    tmail_s_attr_writer 'Content-Transfer-Encoding', :transfer_encoding
    alias encoding                   transfer_encoding
    alias encoding=                  transfer_encoding=
    alias content_transfer_encoding  transfer_encoding
    alias content_transfer_encoding= transfer_encoding=


    tmail_attr_reader 'Content-Disposition', :disposition, 'disposition'
    tmail_s_attr_writer 'Content-Disposition', :disposition
    alias content_disposition  disposition
    alias content_disposition= disposition=

    def set_content_disposition( str, params = nil )
      if f = @header['content-disposition'] then
        f.disposition = str
      else
        f = store( 'Content-Disposition', str )
      end
      f.params.replace params if params
    end


    ###
    ### utils
    ###

    def create_reply
      mail = TMail::Mail.parse('')
      mail.subject = 'Re: ' + subject('').sub( /\A(?:\[[^\]]+])?(?:\s*Re:)*\s*/i, '' )
      mail.to_addrs = reply_addresses([])
      mail.in_reply_to = [message_id(nil)].compact
      mail.references = references([]) + [message_id(nil)].compact
      mail.mime_version = '1.0'
      mail
    end


    def base64_encode
      store 'Content-Transfer-Encoding', 'Base64'
      self.body = Base64.folding_encode(self.body)
    end

    def base64_decode
      if /base64/i === (@header['content-transfer-encoding'] || '') then
        store 'Content-Transfer-Encoding', '8bit'
        self.body = Base64.decode(self.body)
      end
    end


    def destinations( default = nil )
      ret = []
      %w( to cc bcc ).each do |nm|
        if head = @header[nm] then
          head.addrs.each {|i| ret.push i.address }
        end
      end
      ret.empty? ? default : ret
    end

    def each_destination( &block )
      destinations([]).each do |i|
        if Address === i then
          yield i
        else
          i.each( &block )
        end
      end
    end

    alias each_dest each_destination


    def reply_addresses( default = nil )
      reply_to_addrs(nil) or from_addrs(nil) or default
    end

    def error_reply_addresses( default = nil )
      if s = sender(nil) then
        [s]
      else
        from_addrs(default)
      end
    end


    def multipart?
      main_type('').downcase == 'multipart'
    end

  end   # class Mail

end   # module TMail
