package PDF::API2::Util;

use vars qw($VERSION @ISA @EXPORT %colors %winfonts);
use Math::Trig;
use POSIX qw( HUGE_VAL floor );

( $VERSION ) = '$Revisioning: 20011128.170503 $ ' =~ /\$Revisioning:\s+([^\s]+)/;

use Exporter;
@ISA = qw(Exporter);
@EXPORT = qw( 
	pdfkey digest digestx digest16 digest32 
	float floats floats5 intg intgs 
	mMin mMax 
	cRGB cRGB8 RGBasCMYK HSVtoRGB
	namecolor
	opts_from_ttf
	opts_from_pfm
);            

BEGIN {
	if($^O eq 'MSWin32') {
		eval q|
			use Win32;
			use Win32::TieRegistry;
			$Registry->Delimiter("/");
		|;
	}

        %colors=(
        	aliceblue => [0.941176470588235,0.972549019607843,1],
        	antiquewhite => [0.980392156862745,0.92156862745098,0.843137254901961],
        	aqua => [0,1,1],
        	aquamarine => [0.498039215686275,1,0.831372549019608],
        	azure => [0.941176470588235,1,1],
        	beige => [0.96078431372549,0.96078431372549,0.862745098039216],
        	bisque => [1,0.894117647058824,0.768627450980392],
        	black => [0,0,0],
        	blanchedalmond => [1,0.92156862745098,0.803921568627451],
        	blue => [0,0,1],
        	blueviolet => [0.541176470588235,0.168627450980392,0.886274509803922],
        	brown => [0.647058823529412,0.164705882352941,0.164705882352941],
        	burlywood => [0.870588235294118,0.72156862745098,0.529411764705882],
        	cadetblue => [0.372549019607843,0.619607843137255,0.627450980392157],
        	chartreuse => [0.498039215686275,1,0],
        	chocolate => [0.823529411764706,0.411764705882353,0.117647058823529],
        	coral => [1,0.498039215686275,0.313725490196078],
        	cornflowerblue => [0.392156862745098,0.584313725490196,0.929411764705882],
        	cornsilk => [1,0.972549019607843,0.862745098039216],
        	crimson => [0.862745098039216,0.0784313725490196,0.235294117647059],
        	cyan => [0,1,1],
        	darkblue => [0,0,0.545098039215686],
        	darkcyan => [0,0.545098039215686,0.545098039215686],
        	darkgoldenrod => [0.72156862745098,0.525490196078431,0.0431372549019608],
        	darkgray => [0.662745098039216,0.662745098039216,0.662745098039216],
        	darkgreen => [0,0.392156862745098,0],
        	darkgrey => [0.662745098039216,0.662745098039216,0.662745098039216],
        	darkkhaki => [0.741176470588235,0.717647058823529,0.419607843137255],
        	darkmagenta => [0.545098039215686,0,0.545098039215686],
        	darkolivegreen => [0.333333333333333,0.419607843137255,0.184313725490196],
        	darkorange => [1,0.549019607843137,0],
        	darkorchid => [0.6,0.196078431372549,0.8],
        	darkred => [0.545098039215686,0,0],
        	darksalmon => [0.913725490196078,0.588235294117647,0.47843137254902],
        	darkseagreen => [0.56078431372549,0.737254901960784,0.56078431372549],
        	darkslateblue => [0.282352941176471,0.23921568627451,0.545098039215686],
        	darkslategray => [0.184313725490196,0.309803921568627,0.309803921568627],
        	darkslategrey => [0.184313725490196,0.309803921568627,0.309803921568627],
        	darkturquoise => [0,0.807843137254902,0.819607843137255],
        	darkviolet => [0.580392156862745,0,0.827450980392157],
        	deeppink => [1,0.0784313725490196,0.576470588235294],
        	deepskyblue => [0,0.749019607843137,1],
        	dimgray => [0.411764705882353,0.411764705882353,0.411764705882353],
        	dimgrey => [0.411764705882353,0.411764705882353,0.411764705882353],
        	dodgerblue => [0.117647058823529,0.564705882352941,1],
        	firebrick => [0.698039215686274,0.133333333333333,0.133333333333333],
        	floralwhite => [1,0.980392156862745,0.941176470588235],
        	forestgreen => [0.133333333333333,0.545098039215686,0.133333333333333],
        	fuchsia => [1,0,1],
        	gainsboro => [0.862745098039216,0.862745098039216,0.862745098039216],
        	ghostwhite => [0.972549019607843,0.972549019607843,1],
        	gold => [1,0.843137254901961,0],
        	goldenrod => [0.854901960784314,0.647058823529412,0.125490196078431],
        	gray => [0.501960784313725,0.501960784313725,0.501960784313725],
        	grey => [0.501960784313725,0.501960784313725,0.501960784313725],
        	green => [0,0.501960784313725,0],
        	greenyellow => [0.67843137254902,1,0.184313725490196],
        	honeydew => [0.941176470588235,1,0.941176470588235],
        	hotpink => [1,0.411764705882353,0.705882352941177],
        	indianred => [0.803921568627451,0.36078431372549,0.36078431372549],
        	indigo => [0.294117647058824,0,0.509803921568627],
        	ivory => [1,1,0.941176470588235],
        	khaki => [0.941176470588235,0.901960784313726,0.549019607843137],
        	lavender => [0.901960784313726,0.901960784313726,0.980392156862745],
        	lavenderblush => [1,0.941176470588235,0.96078431372549],
        	lawngreen => [0.486274509803922,0.988235294117647,0],
        	lemonchiffon => [1,0.980392156862745,0.803921568627451],
        	lightblue => [0.67843137254902,0.847058823529412,0.901960784313726],
        	lightcoral => [0.941176470588235,0.501960784313725,0.501960784313725],
        	lightcyan => [0.87843137254902,1,1],
        	lightgoldenrodyellow => [0.980392156862745,0.980392156862745,0.823529411764706],
        	lightgray => [0.827450980392157,0.827450980392157,0.827450980392157],
        	lightgreen => [0.564705882352941,0.933333333333333,0.564705882352941],
        	lightgrey => [0.827450980392157,0.827450980392157,0.827450980392157],
        	lightpink => [1,0.713725490196078,0.756862745098039],
        	lightsalmon => [1,0.627450980392157,0.47843137254902],
        	lightseagreen => [0.125490196078431,0.698039215686274,0.666666666666667],
        	lightskyblue => [0.529411764705882,0.807843137254902,0.980392156862745],
        	lightslategray => [0.466666666666667,0.533333333333333,0.6],
        	lightslategrey => [0.466666666666667,0.533333333333333,0.6],
        	lightsteelblue => [0.690196078431373,0.768627450980392,0.870588235294118],
        	lightyellow => [1,1,0.87843137254902],
        	lime => [0,1,0],
        	limegreen => [0.196078431372549,0.803921568627451,0.196078431372549],
        	linen => [0.980392156862745,0.941176470588235,0.901960784313726],
        	magenta => [1,0,1],
        	maroon => [0.501960784313725,0,0],
        	mediumaquamarine => [0.4,0.803921568627451,0.666666666666667],
        	mediumblue => [0,0,0.803921568627451],
        	mediumorchid => [0.729411764705882,0.333333333333333,0.827450980392157],
        	mediumpurple => [0.576470588235294,0.43921568627451,0.858823529411765],
        	mediumseagreen => [0.235294117647059,0.701960784313725,0.443137254901961],
        	mediumslateblue => [0.482352941176471,0.407843137254902,0.933333333333333],
        	mediumspringgreen => [0,0.980392156862745,0.603921568627451],
        	mediumturquoise => [0.282352941176471,0.819607843137255,0.8],
        	mediumvioletred => [0.780392156862745,0.0823529411764706,0.52156862745098],
        	midnightblue => [0.0980392156862745,0.0980392156862745,0.43921568627451],
        	mintcream => [0.96078431372549,1,0.980392156862745],
        	mistyrose => [1,0.894117647058824,0.882352941176471],
        	moccasin => [1,0.894117647058824,0.709803921568627],
        	navajowhite => [1,0.870588235294118,0.67843137254902],
        	navy => [0,0,0.501960784313725],
        	oldlace => [0.992156862745098,0.96078431372549,0.901960784313726],
        	olive => [0.501960784313725,0.501960784313725,0],
        	olivedrab => [0.419607843137255,0.556862745098039,0.137254901960784],
        	orange => [1,0.647058823529412,0],
        	orangered => [1,0.270588235294118,0],
        	orchid => [0.854901960784314,0.43921568627451,0.83921568627451],
        	palegoldenrod => [0.933333333333333,0.909803921568627,0.666666666666667],
        	palegreen => [0.596078431372549,0.984313725490196,0.596078431372549],
        	paleturquoise => [0.686274509803922,0.933333333333333,0.933333333333333],
        	palevioletred => [0.858823529411765,0.43921568627451,0.576470588235294],
        	papayawhip => [1,0.937254901960784,0.835294117647059],
        	peachpuff => [1,0.854901960784314,0.725490196078431],
        	peru => [0.803921568627451,0.52156862745098,0.247058823529412],
        	pink => [1,0.752941176470588,0.796078431372549],
        	plum => [0.866666666666667,0.627450980392157,0.866666666666667],
        	powderblue => [0.690196078431373,0.87843137254902,0.901960784313726],
        	purple => [0.501960784313725,0,0.501960784313725],
        	red => [1,0,0],
        	rosybrown => [0.737254901960784,0.56078431372549,0.56078431372549],
        	royalblue => [0.254901960784314,0.411764705882353,0.882352941176471],
        	saddlebrown => [0.545098039215686,0.270588235294118,0.0745098039215686],
        	salmon => [0.980392156862745,0.501960784313725,0.447058823529412],
        	sandybrown => [0.956862745098039,0.643137254901961,0.376470588235294],
        	seagreen => [0.180392156862745,0.545098039215686,0.341176470588235],
        	seashell => [1,0.96078431372549,0.933333333333333],
        	sienna => [0.627450980392157,0.32156862745098,0.176470588235294],
        	silver => [0.752941176470588,0.752941176470588,0.752941176470588],
        	skyblue => [0.529411764705882,0.807843137254902,0.92156862745098],
        	slateblue => [0.415686274509804,0.352941176470588,0.803921568627451],
        	slategray => [0.43921568627451,0.501960784313725,0.564705882352941],
        	slategrey => [0.43921568627451,0.501960784313725,0.564705882352941],
        	snow => [1,0.980392156862745,0.980392156862745],
        	springgreen => [0,1,0.498039215686275],
        	steelblue => [0.274509803921569,0.509803921568627,0.705882352941177],
        	tan => [0.823529411764706,0.705882352941177,0.549019607843137],
        	teal => [0,0.501960784313725,0.501960784313725],
        	thistle => [0.847058823529412,0.749019607843137,0.847058823529412],
        	tomato => [1,0.388235294117647,0.27843137254902],
        	turquoise => [0.250980392156863,0.87843137254902,0.815686274509804],
        	violet => [0.933333333333333,0.509803921568627,0.933333333333333],
        	wheat => [0.96078431372549,0.870588235294118,0.701960784313725],
        	white => [1,1,1],
        	whitesmoke => [0.96078431372549,0.96078431372549,0.96078431372549],
        	yellow => [1,1,0],
        	yellowgreen => [0.603921568627451,0.803921568627451,0.196078431372549],
                none => [0,0,0],
        );

}

sub mMin {
	my $n=HUGE_VAL;
	map { $n=($n>$_) ? $_ : $n } @_;
	return($n);	
}

sub mMax {
	my $n=-(HUGE_VAL);
	map { $n=($n<$_) ? $_ : $n } @_;
	return($n);	
}

sub cRGB {
	my @cmy=(map { 1-$_ } @_);
	my $k=mMin(@cmy);
	return((map { $_-$k } @cmy),$k);
}

sub cRGB8 {
	return cRGB(map { $_/255 } @_);
}

sub RGBasCMYK {
	my @rgb=@_;
	my @cmy=(map { 1-$_ } @rgb);
	my $k=mMin(@cmy);
	return((map { $_-$k } @cmy),$k);
}

sub HSVtoRGB ($$$) {
	my ($h,$s,$v)=@_;
	my ($r,$g,$b,$i,$f,$p,$q,$t);

        if( $s == 0 ) {
                ## achromatic (grey)
                return ($v,$v,$v);
        }

        $h %= 360;                      
        $h /= 60;                       ## sector 0 to 5
        $i = POSIX::floor( $h );
        $f = $h - $i;                   ## factorial part of h
        $p = $v * ( 1 - $s );
        $q = $v * ( 1 - $s * $f );
        $t = $v * ( 1 - $s * ( 1 - $f ) );

	if($i<1) {
		$r = $v;
                $g = $t;
                $b = $p;
	} elsif($i<2){
		$r = $q;
                $g = $v;
                $b = $p;
	} elsif($i<3){
		$r = $p;
                $g = $v;
                $b = $t;
	} elsif($i<4){
		$r = $p;
                $g = $q;
                $b = $v;
	} elsif($i<5){
		$r = $t;
                $g = $p;
                $b = $v;
	} else {
		$r = $v;
                $g = $p;
                $b = $q;
	}
	return ($r,$g,$b);
}

sub namecolor {
	my $name=lc(shift @_);
	$name=~s/[^\#!%a-z0-9]//cg;
	my $col;
	if($name=~/^#/) {
		my ($r,$g,$b,$h);
		if(length($name)<5) {		# zb. #fa4,          #cf0
			$r=hex(substr($name,1,1))/0xf;
			$g=hex(substr($name,2,1))/0xf;
			$b=hex(substr($name,3,1))/0xf;
		} elsif(length($name)<8) {	# zb. #ffaa44,       #ccff00
			$r=hex(substr($name,1,2))/0xff;
			$g=hex(substr($name,3,2))/0xff;
			$b=hex(substr($name,5,2))/0xff;
		} elsif(length($name)<11) {	# zb. #fffaaa444,    #cccfff000
			$r=hex(substr($name,1,3))/0xfff;
			$g=hex(substr($name,4,3))/0xfff;
			$b=hex(substr($name,7,3))/0xfff;
		} else {			# zb. #ffffaaaa4444, #ccccffff0000
			$r=hex(substr($name,1,4))/0xffff;
			$g=hex(substr($name,5,4))/0xffff;
			$b=hex(substr($name,9,4))/0xffff;
		}
		$col=[$r,$g,$b];
	} elsif($name=~/^%/) {
		my ($r,$g,$b,$c,$y,$m,$k);
		if(length($name)<6) {		# zb. %cmyk
			$c=hex(substr($name,1,1))/0xf;
			$m=hex(substr($name,2,1))/0xf;
			$y=hex(substr($name,3,1))/0xf;
			$k=hex(substr($name,4,1))/0xf;
		} elsif(length($name)<10) {	# zb. %ccmmyykk
			$c=hex(substr($name,1,2))/0xff;
			$m=hex(substr($name,3,2))/0xff;
			$y=hex(substr($name,5,2))/0xff;
			$k=hex(substr($name,7,2))/0xff;
		} elsif(length($name)<14) {	# zb. %cccmmmyyykkk
			$c=hex(substr($name,1,3))/0xfff;
			$m=hex(substr($name,4,3))/0xfff;
			$y=hex(substr($name,7,3))/0xfff;
			$k=hex(substr($name,10,3))/0xfff;
		} else {			# zb. %ccccmmmmyyyykkkk
			$c=hex(substr($name,1,4))/0xffff;
			$m=hex(substr($name,5,4))/0xffff;
			$y=hex(substr($name,9,4))/0xffff;
			$k=hex(substr($name,13,4))/0xffff;
		}
		$r=1-$c-$k;
		$g=1-$m-$k;
		$b=1-$y-$k;
		$col=[$r,$g,$b];
	} elsif($name=~/^!/) {
		my ($r,$g,$b,$h,$s,$v);
		if(length($name)<5) {		
			$h=360*hex(substr($name,1,1))/0xf;
			$s=hex(substr($name,2,1))/0xf;
			$v=hex(substr($name,3,1))/0xf;
		} elsif(length($name)<8) {
			$h=360*hex(substr($name,1,2))/0xff;
			$s=hex(substr($name,3,2))/0xff;
			$v=hex(substr($name,5,2))/0xff;
		} elsif(length($name)<11) {	
			$h=360*hex(substr($name,1,3))/0xfff;
			$s=hex(substr($name,4,3))/0xfff;
			$v=hex(substr($name,7,3))/0xfff;
		} else {		
			$h=360*hex(substr($name,1,4))/0xffff;
			$s=hex(substr($name,5,4))/0xffff;
			$v=hex(substr($name,9,4))/0xffff;
		}
		($r,$g,$b)=HSVtoRGB($h,$s,$v);
		$col=[$r,$g,$b];
	} else {
		$col = $colors{$name} || [0,0,0];
	}
	return(@{$col});
}

sub pdfkey {
	my $ddata=join('',@_);
	my $mdkey='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789gT';
	my $xdata="0" x 8;
	my $off=0;
	foreach my $set (0..(length($ddata)<<1)) {
		$off+=vec($ddata,$set,4);
		$off+=vec($xdata,($set & 7),8);
		vec($xdata,($set & 7),8)=vec($mdkey,($off & 0x3f),8);
	}
	return($xdata);
}

sub digestx {
	my $len=shift @_;
	my $mask=$len-1;
	my $ddata=join('',@_);
	my $mdkey='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789gT';
	my $xdata="0" x $len;
	my $off=0;
	my $set;
	foreach $set (0..(length($ddata)<<1)) {
		$off+=vec($ddata,$set,4);
		$off+=vec($xdata,($set & $mask),8);
		vec($xdata,($set & ($mask<<1 |1)),4)=vec($mdkey,($off & 0x7f),4);
	}
	
#	foreach $set (0..$mask) {
#		vec($xdata,$set,8)=(vec($xdata,$set,8) & 0x7f) | 0x40;
#	}

#	$off=0;
#	foreach $set (0..$mask) {
#		$off+=vec($xdata,$set,8);
#		vec($xdata,$set,8)=vec($mdkey,($off & 0x3f),8);
#	}

	return($xdata);
}

sub digest {
	return(digestx(32,@_));
}

sub digest16 {
	return(digestx(16,@_));
}

sub digest32 {
	return(digestx(32,@_));
}

sub xlog10 {
	my $n = shift;
	if($n) {
    		return log(abs($n))/log(10);
	} else { return 0; }
}

sub float {
	my $f=shift @_;
	my $mxd=shift @_||4;
	$f=0 if(abs($f)<0.0000000000000001);
	my $ad=floor(xlog10($f)-$mxd);
	if(abs($f-int($f)) < (10**(-$mxd))) {
		# just in case we have an integer
		return sprintf('%i',$f);
	} elsif($ad>0){
		return sprintf('%f',$f);
	} else {
		return sprintf('%.'.abs($ad).'f',$f);
	}
}
sub floats { return map { float($_); } @_; }
sub floats5 { return map { float($_,5); } @_; }


sub intg {
	my $f=shift @_;
	return sprintf('%i',$f);
}
sub intgs { return map { intg($_); } @_; }

my @cp1252 = (1 .. 127,
       0x20AC, 0x0081, 0x201A, 0x0192, 0x201E, 0x2026, 0x2020, 0x2021,
       0x02C6, 0x2030, 0x0160, 0x2039, 0x0152, 0x008D, 0x017D, 0x008F,
       0x0090, 0x2018, 0x2019, 0x201C, 0x201D, 0x2022, 0x2013, 0x2014,
       0x02DC, 0x2122, 0x0161, 0x203A, 0x0153, 0x009D, 0x017E, 0x0178,
       0xA0 .. 0xFF);

%winfonts=(
	'arialblack' => {
		'-widths' => [ 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 333, 333, 500, 660, 666, 1000, 889, 277, 389, 389, 556, 660, 333, 333, 333, 277, 666, 666, 666, 666, 666, 666, 666, 666, 666, 666, 333, 333, 660, 660, 660, 610, 740, 777, 777, 777, 777, 722, 666, 833, 833, 389, 666, 833, 666, 943, 833, 833, 722, 833, 777, 722, 722, 833, 777, 1000, 777, 777, 722, 389, 277, 389, 660, 500, 333, 666, 666, 666, 666, 666, 389, 666, 666, 333, 333, 666, 333, 1000, 666, 666, 666, 666, 443, 610, 443, 666, 610, 943, 666, 610, 556, 389, 277, 389, 660, 750, 666, 750, 277, 666, 500, 1000, 666, 666, 333, 1000, 722, 333, 1000, 750, 722, 750, 750, 277, 277, 500, 500, 500, 500, 1000, 333, 950, 610, 333, 1000, 750, 556, 777, 333, 333, 666, 666, 660, 666, 277, 666, 333, 799, 399, 666, 660, 333, 799, 500, 399, 660, 399, 399, 333, 666, 850, 333, 333, 399, 399, 666, 1000, 1000, 1000, 610, 777, 777, 777, 777, 777, 777, 1000, 777, 722, 722, 722, 722, 389, 389, 389, 389, 777, 833, 833, 833, 833, 833, 833, 660, 833, 833, 833, 833, 833, 777, 722, 666, 666, 666, 666, 666, 666, 666, 1000, 666, 666, 666, 666, 666, 333, 333, 333, 333, 666, 666, 666, 666, 666, 666, 666, 660, 666, 666, 666, 666, 666, 610, 666, 610 ],
		'-descent' => -309,
		'-firstchar' => 1,
		'-fontname' => 'ArialBlack',
		'-capheight' => 715,
		'-italicangle' => '0',
		'-flags' => 40,
		'-ascent' => 1100,
		'-xheight' => 518,
		'-fontbbox' => [ -193, -306, 1687, 1083 ],
		'-type' => 'TrueType',
		'-lastchar' => 255
	},
	'arialbolditalic' => {
		'-widths' => [ 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 277, 333, 474, 556, 556, 889, 722, 237, 333, 333, 389, 583, 277, 333, 277, 277, 556, 556, 556, 556, 556, 556, 556, 556, 556, 556, 333, 333, 583, 583, 583, 610, 975, 722, 722, 722, 722, 666, 610, 777, 722, 277, 556, 722, 610, 833, 722, 777, 666, 777, 722, 666, 610, 722, 666, 943, 666, 666, 610, 333, 277, 333, 583, 556, 333, 556, 610, 556, 610, 556, 333, 610, 610, 277, 277, 556, 277, 889, 610, 610, 610, 610, 389, 556, 333, 610, 556, 777, 556, 556, 500, 389, 279, 389, 583, 750, 556, 750, 277, 556, 500, 1000, 556, 556, 333, 1000, 666, 333, 1000, 750, 610, 750, 750, 277, 277, 500, 500, 350, 556, 1000, 333, 1000, 556, 333, 943, 750, 500, 666, 277, 333, 556, 556, 556, 556, 279, 556, 333, 736, 370, 556, 583, 333, 736, 552, 399, 548, 333, 333, 333, 576, 556, 277, 333, 333, 365, 556, 833, 833, 833, 610, 722, 722, 722, 722, 722, 722, 1000, 722, 666, 666, 666, 666, 277, 277, 277, 277, 722, 722, 777, 777, 777, 777, 777, 583, 777, 722, 722, 722, 722, 666, 666, 610, 556, 556, 556, 556, 556, 556, 889, 556, 556, 556, 556, 556, 277, 277, 277, 277, 610, 610, 610, 610, 610, 610, 610, 548, 610, 610, 610, 610, 610, 556, 610, 556 ],
		'-descent' => -211,
		'-firstchar' => 1,
		'-fontname' => 'ArialBoldItalic',
		'-capheight' => 715,
		'-italicangle' => -12,
		'-flags' => 104,
		'-ascent' => 905,
		'-xheight' => 518,
		'-type' => 'TrueType',
		'-fontbbox' => [ -559, -376, 1156, 1030 ],
		'-lastchar' => 255
	},
	'arialbold' => {
		'-widths' => [ 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 277, 333, 474, 556, 556, 889, 722, 237, 333, 333, 389, 583, 277, 333, 277, 277, 556, 556, 556, 556, 556, 556, 556, 556, 556, 556, 333, 333, 583, 583, 583, 610, 975, 722, 722, 722, 722, 666, 610, 777, 722, 277, 556, 722, 610, 833, 722, 777, 666, 777, 722, 666, 610, 722, 666, 943, 666, 666, 610, 333, 277, 333, 583, 556, 333, 556, 610, 556, 610, 556, 333, 610, 610, 277, 277, 556, 277, 889, 610, 610, 610, 610, 389, 556, 333, 610, 556, 777, 556, 556, 500, 389, 279, 389, 583, 750, 556, 750, 277, 556, 500, 1000, 556, 556, 333, 1000, 666, 333, 1000, 750, 610, 750, 750, 277, 277, 500, 500, 350, 556, 1000, 333, 1000, 556, 333, 943, 750, 500, 666, 277, 333, 556, 556, 556, 556, 279, 556, 333, 736, 370, 556, 583, 333, 736, 552, 399, 548, 333, 333, 333, 576, 556, 277, 333, 333, 365, 556, 833, 833, 833, 610, 722, 722, 722, 722, 722, 722, 1000, 722, 666, 666, 666, 666, 277, 277, 277, 277, 722, 722, 777, 777, 777, 777, 777, 583, 777, 722, 722, 722, 722, 666, 666, 610, 556, 556, 556, 556, 556, 556, 889, 556, 556, 556, 556, 556, 277, 277, 277, 277, 610, 610, 610, 610, 610, 610, 610, 548, 610, 610, 610, 610, 610, 556, 610, 556 ],
		'-descent' => -211,
		'-firstchar' => 1,
		'-fontname' => 'ArialBold',
		'-capheight' => 715,
		'-italicangle' => '0',
		'-flags' => 40,
		'-ascent' => 905,
		'-xheight' => 518,
		'-type' => 'TrueType',
		'-fontbbox' => [ -627, -376, 2033, 1047 ],
		'-lastchar' => 255
	},
	'arialitalic' => {
		'-widths' => [ 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 277, 277, 354, 556, 556, 889, 666, 190, 333, 333, 389, 583, 277, 333, 277, 277, 556, 556, 556, 556, 556, 556, 556, 556, 556, 556, 277, 277, 583, 583, 583, 556, 1015, 666, 666, 722, 722, 666, 610, 777, 722, 277, 500, 666, 556, 833, 722, 777, 666, 777, 722, 666, 610, 722, 666, 943, 666, 666, 610, 277, 277, 277, 469, 556, 333, 556, 556, 500, 556, 556, 277, 556, 556, 222, 222, 500, 222, 833, 556, 556, 556, 556, 333, 500, 277, 556, 500, 722, 500, 500, 500, 333, 259, 333, 583, 750, 556, 750, 222, 556, 333, 1000, 556, 556, 333, 1000, 666, 333, 1000, 750, 610, 750, 750, 222, 222, 333, 333, 350, 556, 1000, 333, 1000, 500, 333, 943, 750, 500, 666, 277, 333, 556, 556, 556, 556, 259, 556, 333, 736, 370, 556, 583, 333, 736, 552, 399, 548, 333, 333, 333, 576, 537, 277, 333, 333, 365, 556, 833, 833, 833, 610, 666, 666, 666, 666, 666, 666, 1000, 722, 666, 666, 666, 666, 277, 277, 277, 277, 722, 722, 777, 777, 777, 777, 777, 583, 777, 722, 722, 722, 722, 666, 666, 610, 556, 556, 556, 556, 556, 556, 889, 500, 556, 556, 556, 556, 277, 277, 277, 277, 556, 556, 556, 556, 556, 556, 556, 548, 610, 556, 556, 556, 556, 500, 556, 500 ],
		'-descent' => -211,
		'-firstchar' => 1,
		'-fontname' => 'ArialItalic',
		'-capheight' => 715,
		'-italicangle' => -12,
		'-flags' => 104,
		'-ascent' => 905,
		'-xheight' => 518,
		'-type' => 'TrueType',
		'-fontbbox' => [ -517, -324, 1081, 1024 ],
		'-lastchar' => 255
	},
	'arial' => {
		'-widths' => [ 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 750, 277, 277, 354, 556, 556, 889, 666, 190, 333, 333, 389, 583, 277, 333, 277, 277, 556, 556, 556, 556, 556, 556, 556, 556, 556, 556, 277, 277, 583, 583, 583, 556, 1015, 666, 666, 722, 722, 666, 610, 777, 722, 277, 500, 666, 556, 833, 722, 777, 666, 777, 722, 666, 610, 722, 666, 943, 666, 666, 610, 277, 277, 277, 469, 556, 333, 556, 556, 500, 556, 556, 277, 556, 556, 222, 222, 500, 222, 833, 556, 556, 556, 556, 333, 500, 277, 556, 500, 722, 500, 500, 500, 333, 259, 333, 583, 750, 556, 750, 222, 556, 333, 1000, 556, 556, 333, 1000, 666, 333, 1000, 750, 610, 750, 750, 222, 222, 333, 333, 350, 556, 1000, 333, 1000, 500, 333, 943, 750, 500, 666, 277, 333, 556, 556, 556, 556, 259, 556, 333, 736, 370, 556, 583, 333, 736, 552, 399, 548, 333, 333, 333, 576, 537, 277, 333, 333, 365, 556, 833, 833, 833, 610, 666, 666, 666, 666, 666, 666, 1000, 722, 666, 666, 666, 666, 277, 277, 277, 277, 722, 722, 777, 777, 777, 777, 777, 583, 777, 722, 722, 722, 722, 666, 666, 610, 556, 556, 556, 556, 556, 556, 889, 500, 556, 556, 556, 556, 277, 277, 277, 277, 556, 556, 556, 556, 556, 556, 556, 548, 610, 556, 556, 556, 556, 500, 556, 500 ],
		'-descent' => -211,
		'-firstchar' => 1,
		'-fontname' => 'Arial',
		'-capheight' => 715,
		'-italicangle' => '0',
		'-flags' => 40,
		'-ascent' => 905,
		'-xheight' => 518,
		'-type' => 'TrueType',
		'-fontbbox' => [ -664, -324, 2028, 1037 ],
		'-lastchar' => 255
	},
	'comicsansmsbold' => {
		'-widths' => [ 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 433, 237, 437, 842, 693, 820, 654, 433, 366, 366, 529, 610, 433, 610, 433, 511, 610, 610, 610, 610, 610, 610, 610, 610, 610, 610, 433, 433, 610, 610, 610, 565, 931, 731, 630, 618, 721, 624, 606, 679, 768, 546, 665, 610, 550, 882, 812, 798, 532, 876, 640, 693, 695, 736, 674, 1039, 723, 635, 693, 376, 549, 376, 610, 626, 556, 555, 593, 513, 587, 559, 508, 530, 577, 280, 403, 540, 273, 776, 523, 525, 534, 520, 480, 486, 471, 520, 486, 684, 590, 552, 538, 366, 421, 366, 610, 500, 618, 500, 433, 588, 429, 675, 610, 596, 556, 1241, 693, 610, 1193, 500, 693, 500, 500, 226, 226, 433, 433, 610, 441, 882, 556, 882, 507, 610, 896, 500, 538, 635, 476, 237, 610, 793, 610, 610, 421, 634, 556, 795, 610, 638, 610, 610, 795, 626, 610, 610, 610, 610, 556, 610, 760, 610, 556, 610, 610, 638, 610, 610, 610, 565, 731, 731, 731, 731, 731, 731, 1086, 618, 624, 624, 624, 624, 546, 546, 546, 546, 721, 812, 798, 798, 798, 798, 798, 610, 798, 736, 736, 736, 736, 635, 520, 533, 555, 555, 555, 555, 555, 555, 911, 513, 559, 559, 559, 559, 280, 280, 280, 280, 508, 523, 525, 525, 525, 525, 525, 610, 533, 520, 520, 520, 520, 552, 534, 552 ],
		'-descent' => -291,
		'-firstchar' => 1,
		'-fontname' => 'ComicSansMSBold',
		'-capheight' => 758,
		'-italicangle' => '0',
		'-flags' => 34,
		'-ascent' => 1102,
		'-xheight' => 539,
		'-type' => 'TrueType',
		'-fontbbox' => [ -111, -291, 1229, 1102 ],
		'-lastchar' => 255
	},
	'comicsansms' => {
		'-widths' => [ 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 298, 237, 424, 842, 693, 820, 654, 388, 366, 366, 529, 480, 276, 416, 249, 511, 610, 450, 610, 610, 610, 610, 610, 610, 610, 610, 298, 298, 381, 510, 381, 523, 931, 731, 630, 602, 721, 624, 606, 679, 768, 546, 665, 610, 550, 882, 796, 798, 520, 876, 628, 693, 679, 736, 649, 1039, 723, 635, 693, 376, 549, 376, 581, 626, 556, 511, 593, 513, 587, 547, 508, 530, 577, 280, 403, 540, 273, 776, 523, 525, 534, 520, 480, 486, 471, 520, 486, 684, 590, 520, 538, 366, 421, 366, 597, 500, 610, 500, 298, 426, 411, 675, 596, 596, 556, 1241, 693, 381, 1193, 500, 693, 500, 500, 180, 180, 393, 393, 387, 441, 882, 556, 882, 403, 381, 896, 500, 538, 635, 298, 237, 623, 793, 611, 635, 404, 634, 556, 795, 526, 577, 480, 416, 795, 626, 409, 480, 650, 650, 556, 520, 693, 249, 556, 650, 449, 577, 650, 650, 650, 523, 731, 731, 731, 731, 731, 731, 1086, 602, 624, 624, 624, 624, 546, 546, 546, 546, 721, 796, 798, 798, 798, 798, 798, 480, 798, 736, 736, 736, 736, 635, 520, 443, 511, 511, 511, 511, 511, 511, 911, 513, 547, 547, 547, 547, 280, 280, 280, 280, 508, 523, 525, 525, 525, 525, 525, 480, 525, 520, 520, 520, 520, 520, 534, 431 ],
		'-descent' => -291,
		'-firstchar' => 1,
		'-fontname' => 'ComicSansMS',
		'-capheight' => 758,
		'-italicangle' => '0',
		'-flags' => 34,
		'-ascent' => 1102,
		'-xheight' => 539,
		'-type' => 'TrueType',
		'-fontbbox' => [ -92, -312, 1186, 1102 ],
		'-lastchar' => 255
	},
	'couriernewbolditalic' => {
		'-widths' => [ 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600 ],
		'-descent' => -300,
		'-firstchar' => 1,
		'-fontname' => 'CourierNewBoldItalic',
		'-capheight' => 592,
		'-italicangle' => -12,
		'-flags' => 107,
		'-ascent' => 832,
		'-xheight' => 443,
		'-type' => 'TrueType',
		'-fontbbox' => [ -102, -376, 836, 1003 ],
		'-lastchar' => 255
	},
	'couriernewbold' => {
		'-widths' => [ 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600 ],
		'-descent' => -300,
		'-firstchar' => 1,
		'-fontname' => 'CourierNewBold',
		'-capheight' => 592,
		'-italicangle' => '0',
		'-flags' => 43,
		'-ascent' => 832,
		'-xheight' => 443,
		'-type' => 'TrueType',
		'-fontbbox' => [ -46, -710, 701, 1221 ],
		'-lastchar' => 255
	},
	'couriernewitalic' => {
		'-widths' => [ 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600 ],
		'-descent' => -300,
		'-firstchar' => 1,
		'-fontname' => 'CourierNewItalic',
		'-capheight' => 571,
		'-italicangle' => -12,
		'-flags' => 107,
		'-ascent' => 832,
		'-xheight' => 422,
		'-type' => 'TrueType',
		'-fontbbox' => [ -67, -273, 800, 1000 ],
		'-lastchar' => 255
	},
	'couriernew' => {
		'-widths' => [ 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600 ],
		'-descent' => -300,
		'-firstchar' => 1,
		'-fontname' => 'CourierNew',
		'-capheight' => 571,
		'-italicangle' => '0',
		'-flags' => 43,
		'-ascent' => 832,
		'-xheight' => 422,
		'-type' => 'TrueType',
		'-fontbbox' => [ -21, -679, 637, 1020 ],
		'-lastchar' => 255
	},
	'tahomabold' => {
		'-widths' => [ 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 292, 342, 489, 818, 636, 1198, 781, 275, 454, 454, 636, 818, 312, 431, 312, 577, 636, 636, 636, 636, 636, 636, 636, 636, 636, 636, 363, 363, 818, 818, 818, 566, 919, 684, 686, 667, 757, 615, 581, 745, 764, 483, 500, 696, 572, 893, 770, 770, 657, 770, 726, 633, 612, 738, 674, 1027, 684, 670, 622, 454, 577, 454, 818, 636, 545, 598, 631, 527, 629, 593, 382, 629, 640, 301, 362, 602, 301, 953, 640, 617, 629, 629, 433, 514, 415, 640, 578, 889, 604, 575, 525, 623, 636, 623, 818, 1000, 636, 1000, 275, 636, 489, 1000, 636, 636, 545, 1675, 633, 424, 1036, 1000, 622, 1000, 1000, 275, 275, 489, 489, 636, 636, 909, 545, 860, 514, 424, 985, 1000, 525, 670, 292, 342, 636, 636, 636, 636, 636, 636, 545, 928, 507, 703, 818, 431, 928, 636, 519, 818, 539, 539, 545, 650, 636, 363, 545, 539, 539, 703, 1127, 1127, 1127, 566, 684, 684, 684, 684, 684, 684, 988, 667, 615, 615, 615, 615, 483, 483, 483, 483, 773, 770, 770, 770, 770, 770, 770, 818, 770, 738, 738, 738, 738, 670, 659, 645, 598, 598, 598, 598, 598, 598, 937, 527, 593, 593, 593, 593, 301, 301, 301, 301, 619, 640, 617, 617, 617, 617, 617, 818, 617, 640, 640, 640, 640, 575, 629, 575 ],
		'-descent' => -206,
		'-firstchar' => 1,
		'-fontname' => 'TahomaBold',
		'-capheight' => 727,
		'-italicangle' => '0',
		'-flags' => 40,
		'-ascent' => 1000,
		'-xheight' => 548,
		'-type' => 'TrueType',
		'-fontbbox' => [ -670, -207, 1625, 1065 ],
		'-lastchar' => 255
	},
	'tahoma' => {
		'-widths' => [ 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 312, 332, 401, 727, 545, 976, 673, 210, 382, 382, 545, 727, 302, 363, 302, 382, 545, 545, 545, 545, 545, 545, 545, 545, 545, 545, 353, 353, 727, 727, 727, 473, 909, 599, 589, 600, 678, 561, 521, 667, 675, 373, 416, 587, 497, 770, 667, 707, 551, 707, 620, 557, 583, 655, 596, 901, 580, 576, 559, 382, 382, 382, 727, 545, 545, 524, 552, 461, 552, 526, 318, 552, 557, 228, 281, 498, 228, 839, 557, 542, 552, 552, 360, 446, 334, 557, 498, 742, 495, 498, 444, 480, 382, 480, 727, 1000, 545, 1000, 210, 545, 397, 817, 545, 545, 545, 1390, 557, 382, 976, 1000, 559, 1000, 1000, 210, 210, 401, 401, 454, 545, 909, 545, 875, 446, 382, 908, 1000, 444, 576, 312, 332, 545, 545, 545, 545, 382, 545, 545, 928, 493, 573, 727, 363, 928, 545, 470, 727, 493, 493, 545, 567, 545, 353, 545, 493, 493, 573, 1000, 1000, 1000, 473, 599, 599, 599, 599, 599, 599, 913, 600, 561, 561, 561, 561, 373, 373, 373, 373, 698, 667, 707, 707, 707, 707, 707, 727, 707, 655, 655, 655, 655, 576, 565, 548, 524, 524, 524, 524, 524, 524, 879, 461, 526, 526, 526, 526, 228, 228, 228, 228, 545, 557, 542, 542, 542, 542, 542, 727, 542, 557, 557, 557, 557, 498, 552, 498 ],
		'-descent' => -206,
		'-firstchar' => 1,
		'-fontname' => 'Tahoma',
		'-capheight' => 727,
		'-italicangle' => '0',
		'-flags' => 40,
		'-ascent' => 1000,
		'-xheight' => 545,
		'-type' => 'TrueType',
		'-fontbbox' => [ -609, -207, 1338, 1034 ],
		'-lastchar' => 255
	},
	'timesnewromanbolditalic' => {
		'-widths' => [ 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 250, 389, 555, 500, 500, 833, 777, 277, 333, 333, 500, 569, 250, 333, 250, 277, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 333, 333, 569, 569, 569, 500, 832, 666, 666, 666, 722, 666, 666, 722, 777, 389, 500, 666, 610, 889, 722, 722, 610, 722, 666, 556, 610, 722, 666, 889, 666, 610, 610, 333, 277, 333, 569, 500, 333, 500, 500, 443, 500, 443, 333, 500, 556, 277, 277, 500, 277, 777, 556, 500, 500, 500, 389, 389, 277, 556, 443, 666, 500, 443, 389, 348, 220, 348, 569, 777, 500, 777, 333, 500, 500, 1000, 500, 500, 333, 1000, 556, 333, 943, 777, 610, 777, 777, 333, 333, 500, 500, 350, 500, 1000, 333, 1000, 389, 333, 722, 777, 389, 610, 250, 389, 500, 500, 500, 500, 220, 500, 333, 747, 266, 500, 605, 333, 747, 500, 399, 548, 299, 299, 333, 576, 500, 250, 333, 299, 299, 500, 750, 750, 750, 500, 666, 666, 666, 666, 666, 666, 943, 666, 666, 666, 666, 666, 389, 389, 389, 389, 722, 722, 722, 722, 722, 722, 722, 569, 722, 722, 722, 722, 722, 610, 610, 500, 500, 500, 500, 500, 500, 500, 722, 443, 443, 443, 443, 443, 277, 277, 277, 277, 500, 556, 500, 500, 500, 500, 500, 548, 500, 556, 556, 556, 556, 443, 500, 443 ],
		'-descent' => -216,
		'-firstchar' => 1,
		'-fontname' => 'TimesNewRomanBoldItalic',
		'-capheight' => 662,
		'-italicangle' => -16,
		'-flags' => 106,
		'-ascent' => 891,
		'-xheight' => 452,
		'-type' => 'TrueType',
		'-fontbbox' => [ -547, -306, 1206, 1032 ],
		'-lastchar' => 255
	},
	'timesnewromanbold' => {
		'-widths' => [ 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 250, 333, 555, 500, 500, 1000, 833, 277, 333, 333, 500, 569, 250, 333, 250, 277, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 333, 333, 569, 569, 569, 500, 930, 722, 666, 722, 722, 666, 610, 777, 777, 389, 500, 777, 666, 943, 722, 777, 610, 777, 722, 556, 666, 722, 722, 1000, 722, 722, 666, 333, 277, 333, 581, 500, 333, 500, 556, 443, 556, 443, 333, 500, 556, 277, 333, 556, 277, 833, 556, 500, 556, 556, 443, 389, 333, 556, 500, 722, 500, 500, 443, 394, 220, 394, 520, 777, 500, 777, 333, 500, 500, 1000, 500, 500, 333, 1000, 556, 333, 1000, 777, 666, 777, 777, 333, 333, 500, 500, 350, 500, 1000, 333, 1000, 389, 333, 722, 777, 443, 722, 250, 333, 500, 500, 500, 500, 220, 500, 333, 747, 299, 500, 569, 333, 747, 500, 399, 548, 299, 299, 333, 576, 540, 250, 333, 299, 330, 500, 750, 750, 750, 500, 722, 722, 722, 722, 722, 722, 1000, 722, 666, 666, 666, 666, 389, 389, 389, 389, 722, 722, 777, 777, 777, 777, 777, 569, 777, 722, 722, 722, 722, 722, 610, 556, 500, 500, 500, 500, 500, 500, 722, 443, 443, 443, 443, 443, 277, 277, 277, 277, 500, 556, 500, 500, 500, 500, 500, 548, 500, 556, 556, 556, 556, 500, 556, 500 ],
		'-descent' => -216,
		'-firstchar' => 1,
		'-fontname' => 'TimesNewRomanBold',
		'-capheight' => 662,
		'-italicangle' => '0',
		'-flags' => 42,
		'-ascent' => 891,
		'-xheight' => 456,
		'-type' => 'TrueType',
		'-fontbbox' => [ -558, -306, 2033, 1025 ],
		'-lastchar' => 255
	},
	'timesnewromanitalic' => {
		'-widths' => [ 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 250, 333, 419, 500, 500, 833, 777, 213, 333, 333, 500, 674, 250, 333, 250, 277, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 333, 333, 674, 674, 674, 500, 919, 610, 610, 666, 722, 610, 610, 722, 722, 333, 443, 666, 556, 833, 666, 722, 610, 722, 610, 500, 556, 722, 610, 833, 610, 556, 556, 389, 277, 389, 421, 500, 333, 500, 500, 443, 500, 443, 277, 500, 500, 277, 277, 443, 277, 722, 500, 500, 500, 500, 389, 389, 277, 500, 443, 666, 443, 443, 389, 399, 274, 399, 541, 777, 500, 777, 333, 500, 556, 889, 500, 500, 333, 1000, 500, 333, 943, 777, 556, 777, 777, 333, 333, 556, 556, 350, 500, 889, 333, 979, 389, 333, 666, 777, 389, 556, 250, 389, 500, 500, 500, 500, 274, 500, 333, 759, 275, 500, 674, 333, 759, 500, 399, 548, 299, 299, 333, 576, 522, 250, 333, 299, 310, 500, 750, 750, 750, 500, 610, 610, 610, 610, 610, 610, 889, 666, 610, 610, 610, 610, 333, 333, 333, 333, 722, 666, 722, 722, 722, 722, 722, 674, 722, 722, 722, 722, 722, 556, 610, 500, 500, 500, 500, 500, 500, 500, 666, 443, 443, 443, 443, 443, 277, 277, 277, 277, 500, 500, 500, 500, 500, 500, 500, 548, 500, 500, 500, 500, 500, 443, 500, 443 ],
		'-descent' => -216,
		'-firstchar' => 1,
		'-fontname' => 'TimesNewRomanItalic',
		'-capheight' => 662,
		'-italicangle' => -16,
		'-flags' => 106,
		'-ascent' => 891,
		'-xheight' => 441,
		'-type' => 'TrueType',
		'-fontbbox' => [ -497, -306, 1120, 1023 ],
		'-lastchar' => 255
	},
	'timesnewroman' => {
		'-widths' => [ 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 777, 250, 333, 408, 500, 500, 833, 777, 180, 333, 333, 500, 563, 250, 333, 250, 277, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 277, 277, 563, 563, 563, 443, 920, 722, 666, 666, 722, 610, 556, 722, 722, 333, 389, 722, 610, 889, 722, 722, 556, 722, 666, 556, 610, 722, 722, 943, 722, 722, 610, 333, 277, 333, 469, 500, 333, 443, 500, 443, 500, 443, 333, 500, 500, 277, 277, 500, 277, 777, 500, 500, 500, 500, 333, 389, 277, 500, 500, 722, 500, 500, 443, 479, 200, 479, 541, 777, 500, 777, 333, 500, 443, 1000, 500, 500, 333, 1000, 556, 333, 889, 777, 610, 777, 777, 333, 333, 443, 443, 350, 500, 1000, 333, 979, 389, 333, 722, 777, 443, 722, 250, 333, 500, 500, 500, 500, 200, 500, 333, 759, 275, 500, 563, 333, 759, 500, 399, 548, 299, 299, 333, 576, 453, 250, 333, 299, 310, 500, 750, 750, 750, 443, 722, 722, 722, 722, 722, 722, 889, 666, 610, 610, 610, 610, 333, 333, 333, 333, 722, 722, 722, 722, 722, 722, 722, 563, 722, 722, 722, 722, 722, 722, 556, 500, 443, 443, 443, 443, 443, 443, 666, 443, 443, 443, 443, 443, 277, 277, 277, 277, 500, 500, 500, 500, 500, 500, 500, 548, 500, 500, 500, 500, 500, 500, 500, 500 ],
		'-descent' => -216,
		'-firstchar' => 1,
		'-fontname' => 'TimesNewRoman',
		'-capheight' => 662,
		'-italicangle' => '0',
		'-flags' => 42,
		'-ascent' => 891,
		'-xheight' => 447,
		'-type' => 'TrueType',
		'-fontbbox' => [ -568, -306, 2028, 1006 ],
		'-lastchar' => 255
	},
	'verdanabolditalic' => {
		'-widths' => [ 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 341, 402, 587, 867, 710, 1271, 862, 332, 543, 543, 710, 867, 361, 479, 361, 689, 710, 710, 710, 710, 710, 710, 710, 710, 710, 710, 402, 402, 867, 867, 867, 616, 963, 776, 761, 723, 830, 683, 650, 811, 837, 545, 555, 770, 637, 947, 846, 850, 732, 850, 782, 710, 681, 812, 763, 1128, 763, 736, 691, 543, 689, 543, 867, 710, 710, 667, 699, 588, 699, 664, 422, 699, 712, 341, 402, 670, 341, 1058, 712, 685, 699, 699, 497, 593, 455, 712, 648, 979, 668, 650, 596, 710, 543, 710, 867, 1000, 710, 1000, 332, 710, 587, 1048, 710, 710, 710, 1777, 710, 543, 1135, 1000, 691, 1000, 1000, 332, 332, 587, 587, 710, 710, 1000, 710, 963, 593, 543, 1067, 1000, 596, 736, 341, 402, 710, 710, 710, 710, 543, 710, 710, 963, 597, 849, 867, 479, 963, 710, 587, 867, 597, 597, 710, 721, 710, 361, 710, 597, 597, 849, 1181, 1181, 1181, 616, 776, 776, 776, 776, 776, 776, 1093, 723, 683, 683, 683, 683, 545, 545, 545, 545, 830, 846, 850, 850, 850, 850, 850, 867, 850, 812, 812, 812, 812, 736, 734, 712, 667, 667, 667, 667, 667, 667, 1018, 588, 664, 664, 664, 664, 341, 341, 341, 341, 679, 712, 685, 685, 685, 685, 685, 867, 685, 712, 712, 712, 712, 650, 699, 650 ],
		'-descent' => -209,
		'-firstchar' => 1,
		'-fontname' => 'VerdanaBoldItalic',
		'-capheight' => 727,
		'-italicangle' => -13,
		'-flags' => 104,
		'-ascent' => 1005,
		'-xheight' => 548,
		'-type' => 'TrueType',
		'-fontbbox' => [ -166, -207, 1704, 1000 ],
		'-lastchar' => 255
	},
	'verdanabold' => {
		'-widths' => [ 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 341, 402, 587, 867, 710, 1271, 862, 332, 543, 543, 710, 867, 361, 479, 361, 689, 710, 710, 710, 710, 710, 710, 710, 710, 710, 710, 402, 402, 867, 867, 867, 616, 963, 776, 761, 723, 830, 683, 650, 811, 837, 545, 555, 770, 637, 947, 846, 850, 732, 850, 782, 710, 681, 812, 763, 1128, 763, 736, 691, 543, 689, 543, 867, 710, 710, 667, 699, 588, 699, 664, 422, 699, 712, 341, 402, 670, 341, 1058, 712, 686, 699, 699, 497, 593, 455, 712, 649, 979, 668, 650, 596, 710, 543, 710, 867, 1000, 710, 1000, 332, 710, 587, 1048, 710, 710, 710, 1777, 710, 543, 1135, 1000, 691, 1000, 1000, 332, 332, 587, 587, 710, 710, 1000, 710, 963, 593, 543, 1067, 1000, 596, 736, 341, 402, 710, 710, 710, 710, 543, 710, 710, 963, 597, 849, 867, 479, 963, 710, 587, 867, 597, 597, 710, 721, 710, 361, 710, 597, 597, 849, 1181, 1181, 1181, 616, 776, 776, 776, 776, 776, 776, 1093, 723, 683, 683, 683, 683, 545, 545, 545, 545, 830, 846, 850, 850, 850, 850, 850, 867, 850, 812, 812, 812, 812, 736, 734, 712, 667, 667, 667, 667, 667, 667, 1018, 588, 664, 664, 664, 664, 341, 341, 341, 341, 679, 712, 686, 686, 686, 686, 686, 867, 686, 712, 712, 712, 712, 650, 699, 650 ],
		'-descent' => -209,
		'-firstchar' => 1,
		'-fontname' => 'VerdanaBold',
		'-capheight' => 727,
		'-italicangle' => '0',
		'-flags' => 40,
		'-ascent' => 1005,
		'-xheight' => 548,
		'-type' => 'TrueType',
		'-fontbbox' => [ -73, -207, 1707, 1000 ],
		'-lastchar' => 255
	},
	'verdanaitalic' => {
		'-widths' => [ 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 351, 393, 458, 818, 635, 1076, 726, 268, 454, 454, 635, 818, 363, 454, 363, 454, 635, 635, 635, 635, 635, 635, 635, 635, 635, 635, 454, 454, 818, 818, 818, 545, 1000, 682, 685, 698, 765, 632, 574, 775, 751, 420, 454, 692, 556, 842, 748, 787, 603, 787, 695, 683, 616, 731, 682, 990, 685, 615, 685, 454, 454, 454, 818, 635, 635, 600, 623, 520, 623, 595, 351, 621, 632, 274, 344, 586, 274, 973, 632, 606, 623, 623, 426, 520, 394, 632, 590, 818, 591, 590, 525, 634, 454, 634, 818, 1000, 635, 1000, 268, 635, 458, 818, 635, 635, 635, 1518, 683, 454, 1069, 1000, 685, 1000, 1000, 268, 268, 458, 458, 545, 635, 1000, 635, 976, 520, 454, 980, 1000, 525, 615, 351, 393, 635, 635, 635, 635, 454, 635, 635, 1000, 545, 644, 818, 454, 1000, 635, 541, 818, 541, 541, 635, 641, 635, 363, 635, 541, 545, 644, 1000, 1000, 1000, 545, 682, 682, 682, 682, 682, 682, 989, 698, 632, 632, 632, 632, 420, 420, 420, 420, 765, 748, 787, 787, 787, 787, 787, 818, 787, 731, 731, 731, 731, 615, 605, 620, 600, 600, 600, 600, 600, 600, 954, 520, 595, 595, 595, 595, 274, 274, 274, 274, 611, 632, 606, 606, 606, 606, 606, 818, 606, 632, 632, 632, 632, 590, 623, 590 ],
		'-descent' => -209,
		'-firstchar' => 1,
		'-fontname' => 'VerdanaItalic',
		'-capheight' => 727,
		'-italicangle' => -13,
		'-flags' => 104,
		'-ascent' => 1005,
		'-xheight' => 545,
		'-type' => 'TrueType',
		'-fontbbox' => [ -131, -206, 1460, 1000 ],
		'-lastchar' => 255
	},
	'verdana' => {
		'-widths' => [ 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 351, 393, 458, 818, 635, 1076, 726, 268, 454, 454, 635, 818, 363, 454, 363, 454, 635, 635, 635, 635, 635, 635, 635, 635, 635, 635, 454, 454, 818, 818, 818, 545, 1000, 683, 685, 698, 770, 632, 574, 775, 751, 420, 454, 692, 556, 842, 748, 787, 603, 787, 695, 683, 616, 731, 683, 988, 685, 615, 685, 454, 454, 454, 818, 635, 635, 600, 623, 520, 623, 595, 351, 623, 632, 274, 344, 591, 274, 972, 632, 606, 623, 623, 426, 520, 394, 632, 591, 818, 591, 591, 525, 634, 454, 634, 818, 1000, 635, 1000, 268, 635, 458, 818, 635, 635, 635, 1521, 683, 454, 1069, 1000, 685, 1000, 1000, 268, 268, 458, 458, 545, 635, 1000, 635, 976, 520, 454, 981, 1000, 525, 615, 351, 393, 635, 635, 635, 635, 454, 635, 635, 1000, 545, 644, 818, 454, 1000, 635, 541, 818, 541, 541, 635, 641, 635, 363, 635, 541, 545, 644, 1000, 1000, 1000, 545, 683, 683, 683, 683, 683, 683, 984, 698, 632, 632, 632, 632, 420, 420, 420, 420, 775, 748, 787, 787, 787, 787, 787, 818, 787, 731, 731, 731, 731, 615, 605, 620, 600, 600, 600, 600, 600, 600, 955, 520, 595, 595, 595, 595, 274, 274, 274, 274, 611, 632, 606, 606, 606, 606, 606, 818, 606, 632, 632, 632, 632, 591, 623, 591 ],
		'-descent' => -209,
		'-firstchar' => 1,
		'-fontname' => 'Verdana',
		'-capheight' => 727,
		'-italicangle' => '0',
		'-flags' => 40,
		'-ascent' => 1005,
		'-xheight' => 545,
		'-type' => 'TrueType',
		'-fontbbox' => [ -49, -206, 1446, 1000 ],
		'-lastchar' => 255
	},
	'webdings' => {
		'-widths' => [ 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 1000, 1000, 1000, 1000, 990, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 537, 537, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 558, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 500, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1220, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 672, 672, 1000, 842, 516, 1000, 1000, 1000, 1000, 428, 605, 1000, 1000, 1000, 1000, 662, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1291, 1000, 1000, 1000, 1000 ],
		'-descent' => -200,
		'-firstchar' => 1,
		'-fontname' => 'Webdings',
		'-capheight' => 604,
		'-italicangle' => '0',
		'-flags' => 34,
		'-ascent' => 799,
		'-xheight' => 604,
		'-type' => 'TrueType',
		'-fontbbox' => [ -1, -200, 4000, 799 ],
		'-lastchar' => 255
	},
	'wingdings' => {
		'-widths' => [ 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 500, 1000, 1030, 1144, 1301, 1343, 893, 1216, 458, 1083, 891, 1132, 1132, 1171, 1171, 1440, 1443, 1096, 1343, 698, 698, 891, 554, 602, 1072, 947, 1078, 939, 891, 891, 891, 891, 909, 909, 587, 792, 674, 674, 941, 941, 548, 548, 891, 843, 843, 843, 1110, 660, 849, 1088, 888, 880, 650, 812, 746, 746, 722, 693, 794, 885, 891, 895, 891, 1156, 1054, 963, 1090, 940, 933, 945, 1024, 928, 1096, 1064, 779, 1049, 1270, 746, 952, 746, 891, 891, 891, 891, 457, 746, 986, 891, 577, 1060, 1060, 891, 891, 891, 530, 530, 500, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 312, 457, 312, 891, 891, 891, 891, 891, 952, 457, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 891, 1047, 1047, 1047, 1047, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1252, 1252, 794, 794, 891, 891, 891, 891, 891, 891, 979, 979, 891, 891, 775, 775, 775, 775, 1067, 1067, 891, 891, 872, 872, 872, 872, 891, 891, 810, 810, 1060, 810, 781, 781, 781, 781, 481, 385, 635, 785, 891, 891, 1034 ],
		'-descent' => -210,
		'-firstchar' => 1,
		'-fontname' => 'Wingdings',
		'-capheight' => 722,
		'-italicangle' => '0',
		'-flags' => 34,
		'-ascent' => 898,
		'-xheight' => 722,
		'-type' => 'TrueType',
		'-fontbbox' => [ 0, -210, 1358, 898 ],
		'-lastchar' => 255
	},
);

my $winfontsloaded=0;
my $psffontsloaded=0;

sub opts_from_ttf {
	my ($file,%opt) = @_;

	if($winfontsloaded==0){
		$winfontsloaded=1;

		if($^O eq 'MSWin32') {
			 
			my $osname= Win32::IsWinNT() ? 'Windows NT' : 'Windows';

			my $subKey= $Registry->{"HKEY_LOCAL_MACHINE/SOFTWARE/Microsoft/$osname/CurrentVersion/Fonts/"};
			my $fontDir=$Registry->{"HKEY_CURRENT_USER/Software/Microsoft/Windows/CurrentVersion/Explorer/Shell Folders"}->{Fonts};
			foreach my $k (keys %{$subKey}) {
				next unless($k=~/\(TrueType\)/);
				my $kk=lc($k);
				$kk=~s|^/||;
				$kk=~s|\s+\(truetype\).*$||g;
				$kk=~s/\s+//g;

				my $f=$subKey->{$k}=~/[\\\/]/ ? $subKey->{$k} : "$fontDir/$subKey->{$k}";
				if(-f $f) {
					my %ff=opts_from_ttf($f);
			#	#	print STDERR " $k :: $kk\n";
					$winfonts{$kk}={%ff} if(%ff);
				}
			}	
		} else {
		##	print STDERR " FFF \n";
		}
	}



	my $alias=lc($file);
	$alias=~s/[\s]+//g;
	 
	if(defined $winfonts{$alias}){
		my %wf=%{$winfonts{$alias}};
		my $subf;
		if((($opt{-ttopts}||0)&0x07) == 2) {
			$subf=',Bold';
		} elsif((($opt{-ttopts}||0)&0x07) == 3) {
			$subf=',Italic';
		} elsif((($opt{-ttopts}||0)&0x07) == 4) {
			$subf=',BoldItalic';
		} else {
			$subf='';
		}
		$wf{-fontname}.=$subf;
		return %wf;
	} elsif($opt{-lookup}==1) {
		return undef;
	}

	my $font = Font::TTF::Font->open($file) || return undef; 

	$Font::TTF::Name::utf8 = 1;
    
	$font->{'post'}->read;
    
    	$opt{-ttfile}=$file;
	$opt{-type}="TrueType";
	my $altname = $font->{'name'}->read->find_name(1) || return undef;
	my $subf = $font->{'name'}->find_name(2);
    	$altname =~ s/\s//og;
    	$subf =~ s/\s//og;
    	$altname .= ",$subf" unless($subf =~ m/^Regular$/o);
	my $name = $font->{'name'}->read->find_name(4) || return undef;
    	$name =~ s/\s//og;
    	$subf='';
    	if($opt{-ttopts}&0x07 == 2) {
		$subf=',Bold';
    	} elsif($opt{-ttopts}&0x07 == 3) {
		$subf=',Italic';
    	} elsif($opt{-ttopts}&0x07 == 4) {
		$subf=',BoldItalic';
    	} else {
		$subf='';
    	}
	$opt{-fontname}="$name$subf";
	$opt{-altname}=$altname;
	my $upem = $font->{'head'}->read->{'unitsPerEm'};
	$opt{-ascent}=int($font->{'hhea'}->read->{'Ascender'} * 1000 / $upem);
	$opt{-descent}=int($font->{'hhea'}{'Descender'} * 1000 / $upem);

eval {
	$opt{-capheight}=int(
		$font->{'loca'}->read->{'glyphs'}[
			$font->{'post'}{'STRINGS'}{"H"}
		]->read->{'yMax'}
		* 1000 / $upem
	)||0;
};	
eval {
	$opt{-xheight}=int(
		$font->{'loca'}->read->{'glyphs'}[
			$font->{'post'}{'STRINGS'}{"x"}
		]->read->{'yMax'}
		* 1000 / $upem
	)||0;
};	

	$opt{-italicangle} = $font->{'post'}->read->{'italicAngle'}||0;

	$opt{-fontbbox} = [
		int($font->{'head'}{'xMin'} * 1000 / $upem),
        	int($font->{'head'}{'yMin'} * 1000 / $upem),
        	int($font->{'head'}{'xMax'} * 1000 / $upem),
        	int($font->{'head'}{'yMax'} * 1000 / $upem)
        ];

    my $flags = 4;
    $flags = 0;
    $flags |= 1 if ($font->{'OS/2'}->read->{'bProportion'} == 9);
    $flags |= 2 unless ($font->{'OS/2'}{'bSerifStyle'} > 10 && $font->{'OS/2'}{'bSerifStyle'} < 14);
    $flags |= 32; # if ($font->{'OS/2'}{'bFamilyType'} > 3);
    $flags |= 8 if ($font->{'OS/2'}{'bFamilyType'} == 2);
    $flags |= 64 if ($font->{'OS/2'}{'bLetterform'} > 8);
    $opt{-flags} = $flags;
    
    $font->{'cmap'}->read->find_ms;

    my $issymbol = (($font->{'cmap'}{' mstable'}{'Platform'} == 3) && ($font->{'cmap'}{' mstable'}{'Encoding'} == 0));
    
    $font->{'hmtx'}->read;
    $opt{-widths}= [
    	map { 
    		int(
    			$font->{'hmtx'}{'advance'}[$font->{'cmap'}->ms_lookup($_)] / $upem * 1000)
    	}
                ($issymbol ? (0xf001 .. 0xf0ff) : @cp1252) 
    ];
    $opt{-firstchar} = 1;
    $opt{-lastchar} = 255;

	return(%opt);
}

sub opts_from_pfm {
	my ($file,%opt) = @_;

	if($psffontsloaded==0){
		$psffontsloaded=1;

		if(($^O eq 'MSWin32') && !$@) {

			my $osname= Win32::IsWinNT() ? 'Windows NT' : 'Windows';

			my $subKey= $Registry->{"HKEY_LOCAL_MACHINE/SOFTWARE/Microsoft/$osname/CurrentVersion/Type 1 Installer/Type 1 Fonts/"};
			my $fontDir=$Registry->{"HKEY_CURRENT_USER/Software/Microsoft/Windows/CurrentVersion/Explorer/Shell Folders"}->{Fonts};
			foreach my $k (keys %{$subKey}) {
				my $kk=lc($k);
				$kk=~s|^/||;
				$kk=~s/\s+//g;
				my ($t,$pfm,$pfb)=split(/\0/,$subKey->{$k});
				$pfb= $pfb=~/[\\\/]/ ? $pfb : "$fontDir/$pfb";
				$pfm= $pfm=~/[\\\/]/ ? $pfm : "$fontDir/$pfm";
				my %ff=opts_from_pfm($pfm,-pfbfile=>$pfb,-pfmfile=>$pfm);
				$winfonts{$kk}={%ff};
			#	print STDERR " $kk :: $pfb\n";
			#	print STDERR " $kk :: $pfm\n";
			}	
		} else {
		##	print STDERR " FFF \n";
		}
	}

	my $alias=lc($file);
	$alias=~s/[\s]+//g;
	 
	if(defined $winfonts{$alias}){
		my %wf=%{$winfonts{$alias}};
		my $subf;
		if((($opt{-ttopts}||0)&0x07) == 2) {
			$subf=',Bold';
		} elsif((($opt{-ttopts}||0)&0x07) == 3) {
			$subf=',Italic';
		} elsif((($opt{-ttopts}||0)&0x07) == 4) {
			$subf=',BoldItalic';
		} else {
			$subf='';
		}
		$wf{-fontname}.=$subf;
		return %wf;
	} elsif($opt{-lookup}==1) {
		return undef;
	}
	my %df;
	open(INF,$file) || return undef; 
	binmode(INF);
	read(INF,$buf,117);

	# Packing structure for PFM Header
	(	$df{Version}, 
		$df{Size}, 
		$df{Copyright}, 
		$df{Type}, 
		$df{Point}, 
		$df{VertRes}, 
		$df{HorizRes},
		$df{Ascent}, 
		$df{InternalLeading}, 
		$df{ExternalLeading}, 
		$df{Italic}, 
		$df{Underline}, 
		$df{StrikeOut},
		$df{Weight}, 
		$df{CharSet}, 
		$df{PixWidth}, 
		$df{PixHeight}, 
		$df{PitchAndFamily}, 
		$df{AvgWidth}, 
		$df{MaxWidth}, 
		$df{FirstChar}, 
		$df{LastChar}, 
		$df{DefaultChar}, 
		$df{BreakChar}, 
		$df{WidthBytes}, 
		$df{Device}, 
		$df{Face}, 
		$df{BitsPointer}, 
		$df{BitsOffset}
	) = unpack("vVa60vvvvvvvCCCvCvvCvvCCCCvVVVV",$buf); # PFM Header

	read(INF,$buf,30);

	(	$df{SizeFields}, 
		$df{ExtMetricsOffset}, 
		$df{ExtentTable},
		$df{OriginTable}, 
		$df{PairKernTable}, 
		$df{TrackKernTable},
		$df{DriverInfo},
		$df{Reserved}
	) = unpack("vVVVVVVV",$buf); # PFM Extension

	seek(INF,$df{ExtMetricsOffset},0);
	read(INF,$buf,52);
	my %etm;
	(	$etm{Size}, 
		$etm{PointSize}, 
		$etm{Orientation},
		$etm{MasterHeight}, 
		$etm{MinScale}, 
		$etm{MaxScale}, 
		$etm{CapHeight}, 
		$etm{xHeight}, 
		$etm{LowerCaseAscent}, 
		$etm{LowerCaseDescent},
		$etm{Slant}, 
		$etm{SuperScript}, 
		$etm{SubScript},
		$etm{SuperScriptSize}, 
		$etm{SubScriptSize},
		$etm{UnderlineOffset}, 
		$etm{UnderlineWidth},
		$etm{DoubleUpperUnderlineOffset},
		$etm{DoubleLowerUnderlineOffset},
		$etm{DoubleUpperUnderlineWidth},
		$etm{DoubleLowerUnderlineWidth}, 
		$etm{StrikeOutOffset}, 
		$etm{StrikeOutWidth},
		$etm{KernPairs}, 
		$etm{KernTracks}
	) = unpack("vvvvvvvvvvvvvvvvvvvvvvvvvv",$buf); # PFM Extended Text Metrics

	seek(INF,$df{Device},0);
	read(INF,$buf,250);

	($df{postScript}) = unpack("Z*",$buf);
	$buf=substr($buf,length($df{postScript})+1,250);
	($df{windowsName}) = unpack("Z*",$buf); 
	$buf=substr($buf,length($df{windowsName})+1,250);
	($df{psName}) = unpack("Z*",$buf); 

	seek(INF,$df{ExtentTable},0);

	$opt{-widths}= [];

	foreach ( $df{FirstChar} .. $df{LastChar} ) {
		read(INF,$buf,2);
		push(@{$opt{-widths}},unpack('v',$buf));
	}

	close(INF);

#foreach my $k (sort keys %df) {
#	print STDERR " $k :: $df{$k} \n";
#}
#foreach my $k (sort keys %etm) {
#	print STDERR " $k :: $etm{$k} \n";
#}
    
    	
	$opt{-type}="Type1";
	my $altname = $df{windowsName};
    	$altname =~ s/\s//og;
	my $name = $df{psName};
    	$name =~ s/[\s]+//og;
	my $subf='';
    	if($opt{-ttopts}&0x07 == 2) {
		$subf=',Bold';
    	} elsif($opt{-ttopts}&0x07 == 3) {
		$subf=',Italic';
    	} elsif($opt{-ttopts}&0x07 == 4) {
		$subf=',BoldItalic';
    	} else {
		$subf='';
    	}
	$opt{-fontname}="$name$subf";
	$opt{-altname}=$altname;
	$opt{-ascent}=$df{Ascent};
	$opt{-descent}=-200;

	$opt{-capheight}=$etm{CapHeight}||0;
	$opt{-xheight}=$etm{xHeight}||0;
	$opt{-italicangle} = $df{Italic}||0;

	$opt{-fontbbox} = [0,-200,$df{MaxWidth},$df{Ascent}];

	my $flags = 2 +32;
#	$flags |= 1 if ($font->{'OS/2'}->read->{'bProportion'} == 9);
#	$flags |= 2 unless ($font->{'OS/2'}{'bSerifStyle'} > 10 && $font->{'OS/2'}{'bSerifStyle'} < 14);
#	$flags |= 8 if ($font->{'OS/2'}{'bFamilyType'} == 2);
#	$flags |= 32; # if ($font->{'OS/2'}{'bFamilyType'} > 3);
	$flags |= 64 if ($df{Italic} != 0);
	$opt{-flags} = $flags;

	$opt{-firstchar} = $df{FirstChar};
	$opt{-lastchar} = $df{LastChar};

	return(%opt);
}


1;


__END__
