#!/usr/bin/perl -w
###########################################################################

=head1 NAME

Shout - Perl glue for libshout MP3 streaming source library

=head1 SYNOPSIS

  use Shout	qw{};

  my $conn = new Shout
	ip  		=> 'localhost',
	port		=> 8000,
	mount		=> 'testing',
	password	=> 'pa$$word!',
	icy_compat	=> 0,
	aim		=> 'myAIMid',          # ICY only
	icq		=> '000001',	       # ICY only
	irc		=> '#icecast',	       # ICY only
	dumpfile	=> undef,
	name		=> 'Wir sind nur Affen',
	url		=> 'http://apan.org/'
	genre		=> 'Monkey Music',
	description	=> 'A whole lotta monkey music.',
	bitrate		=> 64,
	ispublic	=> 0;

  # - or -

  my $conn = new Shout;

  $conn->ip('localhost');
  $conn->port(8000);
  $conn->mount('testing');
  $conn->password('pa$$word!');
  $conn->icy_compat(0);
  $conn->aim('myAIMid');  # ICY only
  $conn->icq('00001');    # ICY only
  $conn->irc('#icecast'); # ICY only
  $conn->dumpfile(undef);
  $conn->name('Test libshout-perl stream');
  $conn->url('http://www.icecast.org/');
  $conn->genre('perl');
  $conn->description('Stream with icecast at http://www.icecast.org');
  $conn->bitrate(64);
  $conn->ispublic(0);

  ### Connect to the server
  $conn->connect or die "Failed to connect: ", $conn->error;

  ### Stream some data
  my ( $buffer, $bytes ) = ( '', 0 );
  while( ($bytes = sysread( STDIN, $buffer, 4096 )) > 0 ) {
	$conn->sendData( $buffer ) && next;
	print STDERR "Error while sending: ", $conn->error, "\n";
	last;
  } continue {
	$conn->sleep;
  }

  ### Now close the connection
  $conn->disconnect;

=head1 EXPORTS

Nothing by default.

=head2 :constants

The following error constants are exported into your package if the
'C<:constants>' tag is given as an argument to the C<use> statement.

	SHOUTERR_INSANE
	SHOUTERR_NOCONNECT
	SHOUTERR_NOLOGIN
	SHOUTERR_SOCKET
	SHOUTERR_MALLOC
	SHOUTERR_METADATA

=head2 :functions

The following functions are exported into your package if the 'C<:functions>'
tag is given as an argument to the C<use> statement.

	shout_init_connection
	shout_connect
	shout_disconnect
	shout_update_metadata
	shout_send_data
	shout_sleep
	shout_set_ip
	shout_set_port
	shout_set_mount
	shout_set_password
	shout_set_icy_compat
	shout_set_aim
	shout_set_icq
	shout_set_irc
	shout_set_dumpfile
	shout_set_name
	shout_set_url
	shout_set_genre
	shout_set_description
	shout_set_bitrate
	shout_set_ispublic
	shout_get_ip
	shout_get_port
	shout_get_mount
	shout_get_password
	shout_get_icy_compat
	shout_get_aim
	shout_get_icq
	shout_get_irc
	shout_get_dumpfile
	shout_get_name
	shout_get_url
	shout_get_genre
	shout_get_description
	shout_get_bitrate
	shout_get_ispublic
	shout_error

They work almost identically to their libshout C counterparts. See the libshout
documentation for more information about how to use the function interface.

=head2 :all

All of the above symbols can be imported into your namespace by giving the
'C<:all>' tag as an argument to the C<use> statement.

=head1 DESCRIPTION

This module is an object-oriented interface to libshout, an MP3 streaming
library that allows applications to easily communicate and broadcast to an
Icecast streaming media server. It handles the socket connections, metadata
communication, and data streaming for the calling application, and lets
developers focus on feature sets instead of implementation details.

=head1 AUTHOR

Jack Moffitt <jack@icecast.org>

=cut

###########################################################################
package Shout;
use strict;

BEGIN {
	use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS $AUTOLOAD);

	$VERSION = '1.0';

	use Carp;
	use Socket	qw{inet_aton inet_ntoa};

	require Exporter;
	require DynaLoader;
	require AutoLoader;

	# Inheritance
	@ISA = qw(Exporter DynaLoader);

	### Exporter stuff
	@EXPORT = qw{};
	@EXPORT_OK = qw{
		SHOUTERR_INSANE SHOUTERR_NOCONNECT SHOUTERR_NOLOGIN
		SHOUTERR_SOCKET SHOUTERR_MALLOC SHOUTERR_METADATA
		shout_init_connection shout_connect shout_disconnect
		shout_update_metadata shout_send_data shout_sleep
		shout_set_ip shout_set_port shout_set_mount shout_set_password
		shout_set_icy_compat shout_set_dumpfile shout_set_name
		shout_set_url shout_set_genre shout_set_description
		shout_set_bitrate shout_set_ispublic shout_get_ip
		shout_get_port shout_get_mount shout_get_password
		shout_get_icy_compat shout_get_dumpfile shout_get_name
		shout_get_url shout_get_genre shout_get_description
		shout_get_bitrate shout_get_ispublic shout_error
		shout_set_aim shout_get_aim shout_set_icq shout_get_icq
		shout_set_irc shout_get_irc
	};
	%EXPORT_TAGS = (
		all			=> \@EXPORT_OK,
		constants	=> [qw{SHOUTERR_INSANE SHOUTERR_NOCONNECT SHOUTERR_NOLOGIN
						   SHOUTERR_SOCKET S<HOUTERR_MALLOC SHOUTERR_METADATA}],
		functions	=> [qw{shout_init_connection shout_connect shout_disconnect
						   shout_update_metadata shout_send_data shout_sleep
						   shout_set_ip shout_set_port shout_set_mount shout_set_password
						   shout_set_icy_compat shout_set_dumpfile shout_set_name
						   shout_set_url shout_set_genre shout_set_description
						   shout_set_bitrate shout_set_ispublic shout_get_ip
						   shout_get_port shout_get_mount shout_get_password
						   shout_get_icy_compat shout_get_dumpfile shout_get_name
						   shout_get_url shout_get_genre shout_get_description
						   shout_get_bitrate shout_get_ispublic shout_error
						   shout_set_aim shout_get_aim shout_set_icq shout_get_icq
						   shout_set_irc shout_get_irc}],
	);

}

bootstrap Shout $VERSION;

###############################################################################
###	C O N F I G U R A T I O N   G L O B A L S
###############################################################################
use vars qw{@TranslatedMethods %ErrorMessages};

@TranslatedMethods = qw{
	ip
	port
	mount
	password
	icy_compat
	aim
	icq
	irc
	dumpfile
	name
	url
	genre
	description
	bitrate
	ispublic
};

%ErrorMessages = (
	&SHOUTERR_INSANE		=> "The specified ip, password, or port was invalid, or there is already an established connection.",
	&SHOUTERR_NOCONNECT		=> "A connection to the server could not be established.",
	&SHOUTERR_NOLOGIN		=> "Login failed.",
	&SHOUTERR_SOCKET		=> "A socket error occured sending data to the server. The server may have close the connection or gone offline.",
	&SHOUTERR_MALLOC		=> "Failed to allocate new shout connection.",
	&SHOUTERR_METADATA		=> "An error occurred while sending the metadata information.",
);



###############################################################################
###	M E T H O D S
###############################################################################

### (CONSTRUCTOR) METHOD: new( %config )
###	Create and initialize a new icecast server connection. The configuration
###		hash is in the following form:
###
###		(
###			ip		=> <destination ip address>,
###			port		=> <destination port>,
###			mount		=> <stream mountpoint>,
###			password	=> <password to use when connecting>,
###			icy_compat	=> <maintain old icecast compatibility>,
###			aim		=> <aim id, icy only>
###			icq		=> <icq number, icy only>
###			irc		=> <irc channel, icy only>
###			dumpfile	=> <dumpfile for the stream>,
###			name		=> <name of the stream>,
###			url			=> <url of stream's homepage>,
###			genre		=> <genre of the stream>,
###			description	=> <stream description>,
###			bitrate		=> <bitrate of the stream>,
###			ispublic	=> <ispublic flag - list the stream in directory servers>,
###		)
###
### None of the keys are mandatory, and may be set after the connection object
###		is created. This method returns the initialized icecast server
###		connection object. Returns the undefined value on failure.
sub new {
	my $proto = shift;
	my $class = ref $proto || $proto;

	my (
		%args,					# The config pseudo-hash
		$self,					# The shout_conn_t object
	   );

	### Unwrap the pseudo-hash into a real one
	%args = @_;

	### Call our parent's constructor
	$self = $class->raw_new or return undef;
	$self->init;

	### Set each of the config hash elements by using the keys of the
	###		config pseudo-hash as the method name
	foreach my $method ( keys %args ) {

		### Allow keys to be of varying case and preceeded by an optional '-'
		$method =~ s{^-}{};
		$method = lc $method;

		### Turn off strict references so we can use a variable as a method name
	  NO_STRICT: {
			no strict 'refs';
			$self->$method( $args{$method} );
		}
	}

	return $self;
}

### METHOD: init( undef )
### Initialization method. This is just a convenience wrapper for the
###		C<shout_init_connection> method, and is already called for you if you
###		use the C<new()> constructor.
sub init {
	my $self = shift or croak "init: Method called as function";
	return $self->shout_init_connection;
}


### METHOD: ip( [$newAddress] )
### Get/set the target host for the connection. Argument can be either an
###		address or a hostname. It is a fatal error is the argument is a
###		hostname, and the numeric address cannot be resolved from it.
sub ip ($) {
	my $self = shift or croak "ip: Method called as function";

	### If we got an argument, act like a 'set' method
	if ( @_ ) {
		my $ip = shift;

		### If the ip has one or more non-number, non-dot characters in it, try to do a
		### lookup on it
		if ( $ip =~ m{[^.\d]}i ) {
			my $in_addr = inet_aton( $ip )
				or croak "Lookup of '$ip' failed";
			$ip = inet_ntoa( $in_addr )
				or croak "inet_ntoa( $in_addr ) failed";
		}

		return $self->shout_set_ip( $ip );
	}

	return $self->shout_get_ip;
}

### METHOD: connect( undef )
### Connect to the target server. Returns undef and sets the object error
###		message if the connect fails; returns a true value if the connect
###		succeeds.
sub connect {
	my $self = shift or croak "connect: Method called as function";
	return $self->shout_connect;
}

### METHOD: disconnect( undef )
### Disconnect from the target server. Returns undef and sets the object error
###		message if the disconnect fails; returns a true value if the disconnect
###		succeeds.
sub disconnect {
	my $self = shift or croak "disconnect: Method called as function";
	return $self->shout_disconnect;
}

### METHOD: error( undef )
###	Returns a human-readable error message if one has occurred in the
###		object. Returns the undefined value if no error has occurred.
sub error {
	my $self = shift or croak "error: Method called as function";

	my $err = $self->shout_error or return undef;

	return "Unknown error" unless exists $ErrorMessages{ $err };
	return $ErrorMessages{ $err };
}

### METHOD: updateMetadata( $newMetadata )
### Update the metadata for the connection. Returns a true value if the update
###		succeeds, and the undefined value if it fails.
sub updateMetadata ($$) {
	my $self = shift		or croak "updateMetadata: Method called as function";
	my $metadata = shift	or croak "updateMetadata: No metadata specified";

	return $self->shout_update_metadata( $metadata );
}

### METHOD: sendData( $data[, $length] )
### Send the specified data with the optional length to the Icecast
###		server. Returns a true value on success, and returns the undefined value
###		after setting the per-object error message on failure.
sub sendData ($$) {
	my $self = shift	or croak "sendData: Method called as function";
	my $data = shift	or croak "sendData: No data specified";
	my $len = shift || length $data;

	return $self->shout_send_data( $data, $len );
}


### METHOD: sleep( undef )
### Sleep until the connection is ready for more data. This function should be
###		used only in conjuction with C<sendData()>, in order to send data at the
###		correct speed to the icecast server.
sub sleep ($) {
	my $self = shift or croak "sleep: Method called as function";
	return $self->shout_sleep;
}



###############################################################################
###	A U T O L O A D E D   M E T H O D S
###############################################################################

###	METHOD: port( $portNumber )
###	Get/set the port to connect to on the target Icecast server.

###	METHOD: mount( $mountPointName )
###	Get/set the mountpoint to use when connecting to the server.

###	METHOD: password( $password )
###	Get/set the password to use when connecting to the Icecast server.

### METHOD: icy_compat( $boolean )
### Get/set the stream's C<icy_compat> flag. C<icy_compat> is useful if you will
###		be connecting to an older icecast server (before the 1.3.0 release) or
###		to another streaming media server like the ones at live365.com that
###		don't fully support the x-audiocast protocols. Setting this flag
###		disables both the C<dumpfile>, C<mount>, and C<description> attributes,
###		as they are not supported by servers that require C<icy_compat>.

### METHOD: aim( $aimID )
### Get/set the stream's C<aim> ID.

### METHOD: icq( $icqnum )
### Get/set the stream's C<icq> number.

### METHOD: irc( $irc )
### Get/set the stream's C<irc> channel.

### METHOD: dumpfile( $filename )
### Get/set the name of the icecast dumpfile for the stream.  The dumpfile is a
###		special feature of recent icecast servers. When dumpfile is not
###		undefined, and the x-audiocast protocol is being used, the icecast
###		server will save the stream locally to a dumpfile (a dumpfile is just a
###		raw mp3 stream dump). Using this feature will cause data to be written
###		to the drive on the icecast server, so use with caution, or you will
###		fill up your disk!

###	METHOD: name( $nameString )
###	Get/set the name of the stream.

###	METHOD: url( $urlString )
###	Get/set the url of the stream's homepage.

###	METHOD: genre( $genreString )
###	Get/set the stream's genre.

###	METHOD: description( $descriptionString )
###	Get/set the description of the stream.

###	METHOD: bitrate( $bitrate )
###	Get/set the stream's bitrate.

###	METHOD: ispublic( $boolean )
###	Get/set the connection's ispublic flag. This flag, when set to true, indicates
###		that the stream may be listed in the public directory servers.

### (PROXY) METHOD: AUTOLOAD( @args )
###	Provides a proxy for functions and methods which aren't explicitly defined.
sub AUTOLOAD {

	( my $method = $AUTOLOAD ) =~ s{.*::}{};
	croak "& not defined" if $method eq 'constant';

	### If called as a method, check to see if we're doing translation for the
	### method called. If we are, build the name of the real method and call
	### it. If not, delegate this call to Autoloadeer
	if (( ref $_[0] && UNIVERSAL::isa($_[0], __PACKAGE__) )) {
		my $self = shift;

		### If the called method is one we're translating, build the wrapper
		### method for it and jump to it
		if ( grep { $_ eq $method } @TranslatedMethods ) {

			### Turn off strict so we can do some reference trickery
		  NO_STRICT: {
				no strict 'refs';

				my $setMethod = "shout_set_$method";
				my $getMethod = "shout_get_$method";

				*$AUTOLOAD = sub ($$) {
					my $obj = shift;
					return $obj->$setMethod(@_) if @_;
					return $obj->$getMethod();
				};
			}

			### Stick the self-reference back on the stack and jump to the
			### new method
			unshift @_, $self;
			goto &$AUTOLOAD;
		}

		### If the method's not one we're translating, delegate the call to Autoloader
		else {
			$AutoLoader::AUTOLOAD = $AUTOLOAD;
			goto &AutoLoader::AUTOLOAD;

		}

	}

	### If we were called as a function, try to fetch it from the XSUB
	else {
		my $val = constant($method, @_ ? $_[0] : 0);
		croak "No such Shout constant '$method'" if $!;

		### Bootstrap a natural constant if we managed to find a value for the
		### one specified
	  NO_STRICT: {
			no strict 'refs';
			*$AUTOLOAD = sub { $val };
		}

		### Substitute a call to the new function for the current call
		goto &$AUTOLOAD;
	}

	confess "UNREACHED";
}


### Module return value indicates successful loading
1;


__END__

###	AUTOGENERATED DOCUMENTATION FOLLOWS

=head1 METHODS

=over 4

=item I<bitrate( $bitrate )>

Get/set the stream's bitrate.

=item I<connect( undef )>

Connect to the target server. Returns undef and sets the object error
message if the connect fails; returns a true value if the connect
succeeds.

=item I<description( $descriptionString )>

Get/set the description of the stream.

=item I<disconnect( undef )>

Disconnect from the target server. Returns undef and sets the object error
message if the disconnect fails; returns a true value if the disconnect
succeeds.

=item I<dumpfile( $filename )>

Get/set the name of the icecast dumpfile for the stream.  The dumpfile is a
special feature of recent icecast servers. When dumpfile is not
undefined, and the x-audiocast protocol is being used, the icecast
server will save the stream locally to a dumpfile (a dumpfile is just a
raw mp3 stream dump). Using this feature will cause data to be written
to the drive on the icecast server, so use with caution, or you will
fill up your disk!

=item I<error( undef )>

Returns a human-readable error message if one has occurred in the
object. Returns the undefined value if no error has occurred.

=item I<genre( $genreString )>

Get/set the stream's genre.

=item I<icy_compat( $boolean )>

Get/set the stream's C<icy_compat> flag. C<icy_compat> is useful if you will
be connecting to an older icecast server (before the 1.3.0 release) or
to another streaming media server like the ones at live365.com that
don't fully support the x-audiocast protocols. Setting this flag
disables both the C<dumpfile>, C<mount>, and C<description> attributes,
as they are not supported by servers that require C<icy_compat>.

=item I<aim( $aimID )>

Get/set the stream's C<aim> id.

=item I<icq( $icqnum )>

Get/set the stream's C<icq> number.

=item I<irc( $irc )>

Get/set the stream's C<irc> channel.

=item I<init( undef )>

Initialization method. This is just a convenience wrapper for the
C<shout_init_connection> method, and is already called for you if you
use the C<new()> constructor.

=item I<ip( [$newAddress] )>

Get/set the target host for the connection. Argument can be either an
address or a hostname. It is a fatal error is the argument is a
hostname, and the numeric address cannot be resolved from it.

=item I<mount( $mountPointName )>

Get/set the mountpoint to use when connecting to the server.

=item I<name( $nameString )>

Get/set the name of the stream.

=item I<password( $password )>

Get/set the password to use when connecting to the Icecast server.

=item I<port( $portNumber )>

Get/set the port to connect to on the target Icecast server.

=item I<ispublic( $boolean )>

Get/set the connection's ispublic flag. This flag, when set to true, indicates
that the stream may be listed in the public directory servers.

=item I<sendData( $data[, $length] )>

Send the specified data with the optional length to the Icecast
server. Returns a true value on success, and returns the undefined value
after setting the per-object error message on failure.

=item I<sleep( undef )>

Sleep until the connection is ready for more data. This function should be
used only in conjuction with C<sendData()>, in order to send data at the
correct speed to the icecast server.

=item I<updateMetadata( $newMetadata )>

Update the metadata for the connection. Returns a true value if the update
succeeds, and the undefined value if it fails.

=item I<url( $urlString )>

Get/set the url of the stream's homepage.

=back

=head2 Constructor Methods

=over 4

=item I<new( %config )>

Create and initialize a new icecast server connection. The configuration
hash is in the following form:

    (
        ip        => <destination ip address>,
        port        => <destination port>,
        mount       => <stream mountpoint>,
        password    => <password to use when connecting>,
        icy_compat  => <maintain old icecast compatibility>,
	aim	    => <aim id, icy only>,
	icq	    => <icq number, icy only>,
	irc	    => <irc channel, icy only>,
        dumpfile    => <dumpfile for the stream>,
        name        => <name of the stream>,
        url         => <url of stream's homepage>,
        genre       => <genre of the stream>,
        description => <stream description>,
        bitrate     => <bitrate of the stream>,
        ispublic    => <public flag - list the stream in directory servers>,
    )


None of the keys are mandatory, and may be set after the connection object
is created. This method returns the initialized icecast server
connection object. Returns the undefined value on failure.

=back

=head2 Proxy Methods

=over 4

=item I<AUTOLOAD( @args )>

Provides a proxy for functions and methods which aren't explicitly defined.

=back

=cut

