/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_file module

  Functions herein allow to open/read/write/close files.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_file

  Les fonctions presentes dans ce module assurent la gestion
des fichiers.
*/

/*-------------------------------------------------------------*/
/*en definition of a lcrzo_file. You should not use its
   fields directly. */
/*fr definition d'un lcrzo_file. Vous ne devriez pas utiliser
   directement ses champs. */
typedef struct
{ int fd;
  lcrzo_bool canclosefd;
  lcrzo_bool openforreading;
  lcrzo_bool openforwriting;
  lcrzo_bool dataend;
  lcrzo_data storeddata;
  lcrzo_int32 storeddatasize;
} lcrzo_file;

/*-------------------------------------------------------------*/
/*EN************************************************************
 * Note about file names :                                     *
 * Under Windows, '/' and '\' are recognized as directory      *
 * separator. However, under Unix, only '/' is valid.          *
 * So, if you want your code to be portable, use '/'.          *
 ***************************************************************/
/*FR************************************************************
 * Note sur les noms de fichier :                              *
 * Sous Windows, '/' et '\' sont reconnus comme separateur de  *
 * repertoire. Cependant, sous Unix, seul '/' est valide.      *
 * Donc, si vous desirez que votre code soit portable,         *
 * utilisez '/'.                                               *
 ***************************************************************/

/*-------------------------------------------------------------*/
/*EN************************************************************
 * Note about file sizes :                                     *
 * Lcrzo does not support files which have a size > 0x7FFFFFFF *
 * (2Gbytes). Indeed, lcrzo is a network library, and such big *
 * size does not make sense. If a function returns the error   *
 * LCRZO_ERR_SPFILE2G, then you cannot use lcrzo to deal with  *
 * it : just create your own function (dealing for example     *
 * with 64bits sizes).                                         *
 ***************************************************************/
/*FR************************************************************
 * Note sur les tailles de fichier :                           *
 * Lcrzo ne supporte pas les fichiers ayant une taille de plus *
 * de 2G octets (0x7FFFFFFF). En effet, lcrzo est une          *
 * bibliotheque reseau, et des tailles si grandes ne sont pas  *
 * utilisees. Lorsqu'une fonction retourne l'erreur            *
 * LCRZO_ERR_SPFILE2G, alors lcrzo ne peut pas etre employe    *
 * pour gerer le fichier : dans ce cas, creez votre propre     *
 * fonction qui gere par exemple les tailles de 64 bits).      *
 ***************************************************************/


/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_open_read
   Description :
     Open a file for reading.
   Input parameter(s) :
     filename : name of the file to open
   Input/output parameter(s) :
   Output parameter(s) :
     *pfile : lcrzo_file which is opened 
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_file_open_read
   Description :
     Ouvre un fichier en lecture.
   Parametre(s) d'entree :
     filename : nom de fichier a ouvrir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfile : lcrzo_file qui est ouvert
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_file_open_read(lcrzo_conststring filename,
			 lcrzo_file *pfile);

/*en Name : lcrzo_file_open_write
   Description :
     Open a file for writing.
   Input parameter(s) :
     filename : name of the file to open
     mustbeanexistentfile : file must already exist
     mustbeanewfile : file musn't already exist
   Input/output parameter(s) :
   Output parameter(s) :
     *pfile : lcrzo_file which is opened 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_open_write
   Description :
     Ouvre un fichier en ecriture.
   Parametre(s) d'entree :
     filename : nom de fichier a ouvrir
     mustbeanexistentfile : filename doit deja exister
     mustbeanewfile : filename ne doit pas deja exister
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfile : lcrzo_file qui est ouvert
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_open_write(lcrzo_conststring filename,
			  lcrzo_bool mustbeanexistentfile,
			  lcrzo_bool mustbeanewfile,
			  lcrzo_file *pfile);

/*en Name : lcrzo_file_open_append
   Description :
     Open a file for appending.
   Input parameter(s) :
     filename : name of the file to open
     mustbeanexistentfile : file must already exist
   Input/output parameter(s) :
   Output parameter(s) :
     *pfile : lcrzo_file which is opened 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_open_append
   Description :
     Ouvre un fichier en concatenation.
   Parametre(s) d'entree :
     filename : nom de fichier a ouvrir
     mustbeanexistentfile : filename doit deja exister
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfile : lcrzo_file qui est ouvert
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_open_append(lcrzo_conststring filename,
			 lcrzo_bool mustbeanexistentfile,
			 lcrzo_file *pfile);

/*en Name : lcrzo_xxx_close
   Description :
     Close a file.
   Input parameter(s) :
     pfile : lcrzo_file to close
     fd : file descriptor to close
     *pstream : FILE to close
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_close
   Description :
     Ferme un fichier.
   Parametre(s) d'entree :
     pfile : lcrzo_file a fermer
     fd : descripteur de fichier a fermer
     *pstream : FILE qui est ferme
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_close(lcrzo_file *pfile);
int lcrzo_fd_close(int fd);
int lcrzo_stream_close(LCRZOFILE *pstream);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_init_fd
   Description :
     Associate a lcrzo_file to a file descriptor.
     After this function, user has to close the fd and pfile.
   Input parameter(s) :
     fd : file descriptor
   Input/output parameter(s) :
   Output parameter(s) :
     *pfile : lcrzo_file which is associated to fd
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_init_fd
   Description :
     Associe un lcrzo_file a un descripteur de fichier.
     Apres cette fonction, l'utilisateur devra fermer fd et
     pfile.
   Parametre(s) d'entree :
     fd : descripteur de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfile : lcrzo_file qui est associe a fd
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_init_fd(int fd,
                       lcrzo_file *pfile);

/*en Name : lcrzo_file_init_stream
   Description :
     Associate a lcrzo_file to a FILE stream.
     After this function, user has to close the fd and pstream.
   Input parameter(s) :
     pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *pfile : lcrzo_file which is associated to pstream
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_init_stream
   Description :
     Associe un lcrzo_file a un FILE.
     Apres cette fonction, l'utilisateur devra fermer fd et
     pstream.
   Parametre(s) d'entree :
     pstream : FILE 
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfile : lcrzo_file qui est associe a pstream
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_init_stream(LCRZOFILE *pstream,
                           lcrzo_file *pfile);

/*en Name : lcrzo_fd_init_file
   Description :
     Associate a file descriptor to a lcrzo_file.
     After this function, user has to close the fd and pfile.
   Input parameter(s) :
     *pfile : lcrzo_file
   Input/output parameter(s) :
   Output parameter(s) :
     *pfd : file descriptor associated to pfile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_fd_init_file
   Description :
     Associe un descripteur de fichier a un lcrzo_file.
     Apres cette fonction, l'utilisateur devra fermer fd et
     pfile.
   Parametre(s) d'entree :
     *pfile : lcrzo_file
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfd : descripteur de fichier associe a pfile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_init_file(lcrzo_file *pfile, 
                       int *pfd);

/*en Name : lcrzo_stream_init_file
   Description :
     Associate a FILE to a lcrzo_file.
     After this function, user has to close the FILE and pfile.
   Input parameter(s) :
     *pfile : lcrzo_file
   Input/output parameter(s) :
   Output parameter(s) :
     *ppstream : FILE associated to pfile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stream_init_file
   Description :
     Associe un FILE a un lcrzo_file.
     Apres cette fonction, l'utilisateur devra fermer le FILE et
     pfile.
   Parametre(s) d'entree :
     *pfile : lcrzo_file
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ppstream : FILE associe a pfile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_stream_init_file(lcrzo_file *pfile, 
                           LCRZOFILE **ppstream);


/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_write
   Description :
     Write data in a file.
     If datasize is too long, data will be written in several
     chunks (use lcrzo_file_write_msg if you want data to be written
     in one step).
   Input parameter(s) :
     *pfile : lcrzo_file where to write
     data : data to write
     datasize : size of data to write
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_write
   Description :
     Ecrit des donnees dans un fichier.
     Si datasize est trop long, les donnees sont ecrites en plusieurs
     fois (employez lcrzo_file_write_msg si vous desirez que les
     donnees soient ecrites en une seule fois).
   Parametre(s) d'entree :
     *pfile : lcrzo_file ou ecrire
     data : donnees a ecrire
     datasize : taille des donnees a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_write(lcrzo_file *pfile,
		     lcrzo_constdata data,
		     lcrzo_int32 datasize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_write_msg
   Description :
     Write data in a file.
   Input parameter(s) :
     *pfile : lcrzo_file where to write
     data : data to write
     datasize : size of data to write
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_write_msg
   Description :
     Ecrit des donnees dans un fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file ou ecrire
     data : donnees a ecrire
     datasize : taille des donnees a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_write_msg(lcrzo_file *pfile,
			 lcrzo_constdata data,
			 lcrzo_int32 datasize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_readm
   Description :
     Allocate a memory array, and initialize it with
     data read from a file.
   Input parameter(s) :
     *pfile : lcrzo_file where to read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
			       exited without reading data
*/
/*fr Nom : lcrzo_file_readm
   Description :
     Alloue une zone de memoire et l'initialise avec
     des donnees lues depuis un fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file ou lire
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
			LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_file_readm(lcrzo_file *pfile,
		     lcrzo_time *pmaxwaitingtime, 
		     lcrzo_data *pdataout,
		     lcrzo_int32 *pdataoutsize);

/*en Name : lcrzo_file_readm_fixed
   Description :
     Allocate a memory array, and initialize it with
     data of fixed size read from a file.
   Input parameter(s) :
     *pfile : lcrzo_file where to read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
     datasizetoread : size to read
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
			       exited without reading data
*/
/*fr Nom : lcrzo_file_readm_fixed
   Description :
     Alloue une zone de memoire et l'initialise avec
     des donnees de taille constante lues depuis un fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file ou lire
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
			LCRZO_WAIT_INFINITE)
     datasizetoread : taille des donnees a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_file_readm_fixed(lcrzo_file *pfile,
			   lcrzo_time *pmaxwaitingtime, 
			   lcrzo_int32 datasizetoread,
			   lcrzo_data *pdataout);

/*en Name : lcrzo_file_readm_line
   Description :
     Allocate a memory array, and initialize it with
     data read from a file. We stop at first '\r' or '\n' (which
     is not stored in data).
   Input parameter(s) :
     *pfile : lcrzo_file where to read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
			       exited without reading data
*/
/*fr Nom : lcrzo_file_readm_line
   Description :
     Alloue une zone de memoire et l'initialise avec
     des donnees lues depuis un fichier. On s'arrete de lire
     au premier '\r' ou '\n' (qui n'est pas stocke dans
     *pdataout).
   Parametre(s) d'entree :
     *pfile : lcrzo_file ou lire
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
			LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_file_readm_line(lcrzo_file *pfile,
			  lcrzo_time *pmaxwaitingtime, 
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*en file types */
/*fr types de fichiers */
/*en regular file *//*fr fichier normal */
#define LCRZO_FILE_TYPE_REG   1
/*en directory *//*fr repertoire */
#define LCRZO_FILE_TYPE_DIR   2
/*en unix link *//*fr lien unix */
#define LCRZO_FILE_TYPE_LINK  3
/*en unix socket *//*fr socket unix */
#define LCRZO_FILE_TYPE_SOCK  4
/*en other type *//*fr autre type */
#define LCRZO_FILE_TYPE_OTHER 5

/*en Name : lcrzo_xxx_type
   Description :
     Get type of a file.
   Input parameter(s) :
     *filename : filename
     *pfile : lcrzo_file
   Input/output parameter(s) :
   Output parameter(s) :
     *ptype : type of the file
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_xxx_type
   Description :
     Obtient le type du fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     *pfile : lcrzo_file
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ptype : type du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_filename_type(lcrzo_conststring filename, 
			lcrzo_int16 *ptype);
int lcrzo_file_type(lcrzo_file *pfile, 
		    lcrzo_int16 *ptype);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_flush
   Description :
     Flush a file.
   Input parameter(s) :
     *pfile : lcrzo_file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_flush
   Description :
     Force l'ecriture dans un fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_flush(lcrzo_file *pfile);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_offset_get
   Description :
     Get offset of a file.
   Input parameter(s) :
     *pfile : lcrzo_file
   Input/output parameter(s) :
   Output parameter(s) :
     *poffset : offset of the file
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_offset_get
   Description :
     Obtient l'offset d'un fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *poffset : offset du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_offset_get(lcrzo_file *pfile, 
			  lcrzo_int32 *poffset);

/*en Name : lcrzo_file_offset_set
   Description :
     Set offset of a file.
   Input parameter(s) :
     *pfile : lcrzo_file
     offset : offset of the file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_offset_set
   Description :
     Affecte l'offset d'un fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file
     offset : offset du fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_offset_set(lcrzo_file *pfile, 
			  lcrzo_int32 offset);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_xxx_size
   Description :
     Get size of a file.
   Input parameter(s) :
     *filename : filename
     *pfile : lcrzo_file
   Input/output parameter(s) :
   Output parameter(s) :
     *psize : size of the file
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_xxx_size
   Description :
     Obtient la taille d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     *pfile : lcrzo_file
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psize : taille du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_filename_size(lcrzo_conststring filename, 
			lcrzo_int32 *psize);
int lcrzo_file_size(lcrzo_file *pfile, 
		    lcrzo_int32 *psize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_filename_exist
   Description :
     Check if a file exists.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
     *pyes : if file exists
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_filename_exist
   Description :
     Verifie si un fichier existe.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pyes : si le fichier existe
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_filename_exist(lcrzo_conststring filename,
			 lcrzo_bool *pyes);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_filename_create
   Description :
     Create an empty file.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_filename_create
   Description :
     Cree un fichier vide.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_filename_create(lcrzo_conststring filename);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_filename_remove
   Description :
     Remove a file.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_filename_remove
   Description :
     Efface un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_filename_remove(lcrzo_conststring filename);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_filename_rename
   Description :
     Rename a file.
   Input parameter(s) :
     *oldfilename : old filename
     *newfilename : new filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_filename_rename
   Description :
     Efface un fichier.
   Parametre(s) d'entree :
     *oldfilename : ancien nom de fichier
     *newfilename : nouveau nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_filename_rename(lcrzo_conststring oldfilename, 
			  lcrzo_conststring newfilename);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_filename_copy
   Description :
     Copy a file.
   Input parameter(s) :
     *oldfilename : old filename
     *newfilename : new filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_filename_copy
   Description :
     Copie un fichier.
   Parametre(s) d'entree :
     *oldfilename : ancien nom de fichier
     *newfilename : nouveau nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_filename_copy(lcrzo_conststring oldfilename, 
			lcrzo_conststring newfilename);

/*en Name : lcrzo_file_copy
   Description :
     Copy a file.
   Input parameter(s) :
     *pfile : source file to copy from
     *newfilename : new filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_file_copy
   Description :
     Copie un fichier.
   Parametre(s) d'entree :
     *pfile : fichier source a partir duquel copier
     *newfilename : nouveau nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_file_copy(lcrzo_file *pfile, 
		    lcrzo_conststring newfilename);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_file_truncate
   Description :
     Truncate a file.
   Input parameter(s) :
     *filename : filename
     *pfile : lcrzo_file
     size : size to truncate
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_file_truncate
   Description :
     Tronque un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     *pfile : lcrzo_file
     size : taille apres troncature
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_filename_truncate(lcrzo_conststring filename,
			    lcrzo_int32 size);
int lcrzo_file_truncate(lcrzo_file *pfile, 
			lcrzo_int32 size);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_filename_read
   Description :
     Read all data from a file.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
     *pdata : pointer which will be malloced (so, the
              memory will have to be freed by the
              user with 'free(*pdata)').
              The allocated memory will be set with data
	      from the file
     *pdatasize : reached size of *pdata
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : file was not found
*/
/*fr Nom : lcrzo_filename_read
   Description :
     Lit toutes les donnees d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdata : pointeur qui sera alloue (la memoire
              devra etre liberee par l'utilisateur 
              avec 'free(*pdata)').
              Cette memoire allouee recevra les donnees 
              du fichier
     *pdatasize : taille atteinte par *pdata
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le fichier n'a pas ete trouve
*/
int lcrzo_filename_read(lcrzo_conststring filename,
			lcrzo_data *pdata,
			lcrzo_int32 *pdatasize);

/*en Name : lcrzo_filename_write
   Description :
     Write data in a file.
   Input parameter(s) :
     *filename : filename
     mustbeanexistentfile : file must already exist
     mustbeanewfile : file musn't already exist
     data : data to write
     datasize : size of data to write
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_filename_write
   Description :
     Ecrit des donnees dans un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     mustbeanexistentfile : filename doit deja exister
     mustbeanewfile : filename ne doit pas deja exister
     data : donnees a ecrire
     datasize : taille a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_filename_write(lcrzo_conststring filename,
		     lcrzo_bool mustbeanexistentfile,
		     lcrzo_bool mustbeanewfile,
		     lcrzo_constdata data,
		     lcrzo_int32 datasize);

/*en Name : lcrzo_filename_append
   Description :
     Append data in a file.
   Input parameter(s) :
     *filename : filename
     mustbeanexistentfile : file must already exist
     data : data to append
     datasize : size of data to append
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_filename_append
   Description :
     Concatene des donnees dans un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     mustbeanexistentfile : filename doit deja exister
     data : donnees a ecrire
     datasize : taille a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_filename_append(lcrzo_conststring filename,
		      lcrzo_bool mustbeanexistentfile,
		      lcrzo_constdata data,
		      lcrzo_int32 datasize);


/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*en Name : lcrzo_dir_exist
   Description :
     Check if a directory exist.
   Input parameter(s) :
     *dirname : name of the directory
   Input/output parameter(s) :
   Output parameter(s) :
     *pyes : true if directory exists
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_dir_exist
   Description :
     Verifie si un repertoire existe.
   Parametre(s) d'entree :
     *dirname : nom de repertoire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pyes : vrai si le repertoire existe
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_dir_exist(lcrzo_conststring dirname,
		    lcrzo_bool *pyes);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_dir_create
   Description :
     Create a directory.
     Warning : this function is not secured against
               symlink races
   Input parameter(s) :
     *dirname : name of the directory
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_dir_create
   Description :
     Cree un repertoire.
     Attention : cette fonction n'est pas sure face aux
                 attaques par symlink
   Parametre(s) d'entree :
     *dirname : nom de repertoire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_dir_create(lcrzo_conststring dirname);

/*en Name : lcrzo_dir_remove
   Description :
     Remove a directory.
     Warning : this function is not secured against
               symlink races
   Input parameter(s) :
     *dirname : name of the directory
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_dir_remove
   Description :
     Efface un repertoire.
     Attention : cette fonction n'est pas sure face aux
                 attaques par symlink
   Parametre(s) d'entree :
     *dirname : nom de repertoire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_dir_remove(lcrzo_conststring dirname);

/*en Name : lcrzo_dir_rename
   Description :
     Rename a directory.
     Warning : this function is not secured against
               symlink races
   Input parameter(s) :
     *olddirname : previous name of the directory
     *newdirname : new name for the directory
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_dir_rename
   Description :
     Renomme un repertoire.
     Attention : cette fonction n'est pas sure face aux
                 attaques par symlink
   Parametre(s) d'entree :
     *olddirname : ancien nom de repertoire
     *newdirname : nouveau nom pour le repertoire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_dir_rename(lcrzo_conststring olddirname, 
		     lcrzo_conststring newdirname);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_dir_count
   Description :
     Count the number of files in a directory.
   Input parameter(s) :
     *dirname : name of the directory
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of files
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : directory was not found
*/
/*fr Nom : lcrzo_dir_count
   Description :
     Obtient le nombre de fichiers d'un repertoire
   Parametre(s) d'entree :
     *dirname : nom de repertoire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nom de fichiers du repertoire
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le repertoire n'a pas ete trouve
*/
int lcrzo_dir_count(lcrzo_conststring dirname,
		    lcrzo_int32 *pcount);

/*EN*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/
/*en Name : lcrzo_dir_value_pos
   Description :
     Get the name of the file situated at a given position.
   Input parameter(s) :
     *dirname : name of the directory
     pos : wanted position
   Input/output parameter(s) :
   Output parameter(s) :
     *pfilename : mallocated string containing the file
                  name at position pos
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : directory was not found
*/
/*fr Nom : lcrzo_dir_value_pos
   Description :
     Obtient le nom du fichier situe a une position donnee. 
   Parametre(s) d'entree :
     *dirname : nom de repertoire
      pos : position du nom de fichier voulu
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfilename : chaine alloue contenant le nom du fichier
                  situe a la position pos
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKFILENOTFOUND : le repertoire n'a pas ete trouve
*/
int lcrzo_dir_value_pos(lcrzo_conststring dirname,
			lcrzo_int32 pos,
			lcrzo_string *pfilename);


/*-------------------------------------------------------------*/
/*en Name : lcrzo_filename_clean
   Description :
     Clean a filename ("//"->"/", "/./"->"/", "/aa/../"->"/", etc.).
   Input parameter(s) :
     filename : filename (ex : "/tmp/dir/..//./file")
   Input/output parameter(s) :
   Output parameter(s) :
     *pcleanedfilename : cleaned filename (ex : "/tmp/file")
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_filename_clean
   Description :
     Nettoie le nom d'un fichier ("//"->"/", "/./"->"/",
     "/aa/../"->"/", etc.).
   Parametre(s) d'entree :
     *filename : nom de fichier (ex : "/tmp/dir/..//./fich")
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcleanedfilename : nom de fichier nettoye (ex : "/tmp/fich")
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_filename_clean(lcrzo_conststring filename,
			 lcrzo_string *pcleanedfilename);

/*en Name : lcrzo_filename_parentdir
   Description :
     Set the directory name of a path.
   Input parameter(s) :
     filename : filename (ex : "/tmp/file")
   Input/output parameter(s) :
   Output parameter(s) :
     *pparentdir : directory name (ex : "/tmp")
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_filename_parentdir
   Description :
     Obtient le nom de repertoire d'un fichier
   Parametre(s) d'entree :
     *filename : nom de fichier (ex : "/tmp/fich")
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pparentdir : nom de repertoire (ex : "/tmp")
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_filename_parentdir(lcrzo_conststring filename,
			     lcrzo_string *pparentdir);

/*en Name : lcrzo_filename_lastfile
   Description :
     Set the file name of a path.
   Input parameter(s) :
     filename : filename (ex : "/tmp/file")
   Input/output parameter(s) :
   Output parameter(s) :
     *plastfile : file name (ex : "file")
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_filename_lastfile
   Description :
     Obtient le nom d'un fichier
   Parametre(s) d'entree :
     *filename : nom de fichier (ex : "/tmp/fich")
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *plastfile : nom de fichier (ex : "fich")
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_filename_lastfile(lcrzo_conststring filename,
			    lcrzo_string *plastfile);

