/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_ipopt module

  Functions herein allow to create and decode IP options.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_ipopt

  Les fonctions presentes dans ce module permettent de creer des
options IP.
*/

/*-------------------------------------------------------------*/
/*en define an IP option type */
/*fr definit un type pour les options IP */
#define LCRZO_IPOPT_MAXBYTES 40
typedef lcrzo_uint8 lcrzo_ipopt[LCRZO_IPOPT_MAXBYTES];
typedef const lcrzo_ipopt lcrzo_constipopt;

/*-------------------------------------------------------------*/
/*en this define permits to verify that there is no buffer overflow
   before setting a lcrzo_ipopt */
/*fr ce define permet de verifier qu'il n'y ait pas
   de buffer overflow dans un lcrzo_ipopt */
#define lcrzo_ipopt_verifbof(nboctoptip) if(nboctoptip>LCRZO_IPOPT_MAXBYTES)return(LCRZO_ERR_PAOVERFLOW)

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_init_noop
   Description :
     Initialize a lcrzo_ipopt with NOOP option (NO OPeration).
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_noop
   Description :
     Initialise un lcrzo_ipopt avec une option NOOP (NO OPeration).
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_noop(lcrzo_ipopt ipopt,
			  lcrzo_uint8 *pipoptsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_init_lsrr
   Description :
     Initialize a lcrzo_ipopt with LSRR option
     (loose source routing and record route).
     The output lcrzo_ipopt is ended with NOOP padding
     to have a 4 bytes alignment
   Input parameter(s) :
     ipl1, ipl2, etc. : routers to put in the option
     numberofsetipl : number of variables iplx to put in
                      ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_lsrr
   Description :
     Initialise un lcrzo_ipopt avec une option LSRR
     (loose source routing and record route).
     Le lcrzo_ipopt est termine par des padding "noop" (01)
     afin qu'il soit aligne sur 4 octets.
   Parametre(s) d'entree :
     ipl1, ipl2, etc. : routeurs a mettre dans l'option
     numberofsetipl : nombre d'adresses iplx a mettre dans
                      ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_lsrr(lcrzo_ipl ipl1,
			  lcrzo_ipl ipl2,
			  lcrzo_ipl ipl3,
			  lcrzo_ipl ipl4,
			  lcrzo_ipl ipl5,
			  lcrzo_ipl ipl6,
			  lcrzo_ipl ipl7,
			  lcrzo_ipl ipl8,
			  lcrzo_ipl ipl9,
			  lcrzo_uint8 numberofsetipl,
			  lcrzo_ipopt ipopt,
			  lcrzo_uint8 *pipoptsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_init_ssrr
   Description :
     Initialize a lcrzo_ipopt with SSRR option
     (strict source routing and record route).
     The output lcrzo_ipopt is ended with NOOP padding
     to have a 4 bytes alignment
   Input parameter(s) :
     ipl1, ipl2, etc. : routers to put in the option
     numberofsetipl : number of variables iplx to put in
                      ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_ssrr
   Description :
     Initialise un lcrzo_ipopt avec une option SSRR
     (strict source routing and record route).
     Le lcrzo_ipopt est termine par des padding "noop" (01)
     afin qu'il soit aligne sur 4 octets.
   Parametre(s) d'entree :
     ipl1, ipl2, etc. : routeurs a mettre dans l'option
     numberofsetipl : nombre d'adresses iplx a mettre dans
                      ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_ssrr(lcrzo_ipl ipl1,
			  lcrzo_ipl ipl2,
			  lcrzo_ipl ipl3,
			  lcrzo_ipl ipl4,
			  lcrzo_ipl ipl5,
			  lcrzo_ipl ipl6,
			  lcrzo_ipl ipl7,
			  lcrzo_ipl ipl8,
			  lcrzo_ipl ipl9,
			  lcrzo_uint8 numberofsetipl,
			  lcrzo_ipopt ipopt,
			  lcrzo_uint8 *pipoptsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_init_rr
   Description :
     Initialize a lcrzo_ipopt with RR option
     (record route).
     The output lcrzo_ipopt is ended with NOOP padding
     to have a 4 bytes alignment
   Input parameter(s) :
     numberofroutestosave : number of route that can be set in the
                            lcrzo_ipopt created
			    The max value is 9
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_rr
   Description :
     Initialise un lcrzo_ipopt avec une option RR
     (record route).
     Le lcrzo_ipopt est termine par des padding "noop" (01)
     afin qu'il soit aligne sur 4 octets.
   Parametre(s) d'entree :
     numberofroutestosave : nombre de routes que l'on pourra 
                            mettre dans ipopt
			    The max value is 9
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_rr(lcrzo_uint8 numberofroutestosave,
			lcrzo_ipopt ipopt,
			lcrzo_uint8 *pipoptsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_init_time
   Description :
     Initialize a lcrzo_ipopt with timestamp option.
     The output lcrzo_ipopt is ended with NOOP padding
     to have a 4 bytes alignment
   Input parameter(s) :
     numberoftimestamptosave : number of timestamp that can
                               be set in the lcrzo_ipopt created
			       The max value is 9
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_time
   Description :
     Initialise un lcrzo_ipopt avec une option timestamp.
     Le lcrzo_ipopt est termine par des padding "noop" (01)
     afin qu'il soit aligne sur 4 octets.
   Parametre(s) d'entree :
     numberoftimestamptosave : nombre de timestamp que l'on pourra 
                               mettre dans ipopt
			       The max value is 9
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_time(lcrzo_uint8 numberoftimestamptosave,
			  lcrzo_ipopt ipopt,
			  lcrzo_uint8 *pipoptsize);

/*en Name : lcrzo_ipopt_init_timeip
   Description :
     Initialize a lcrzo_ipopt with timestamp option.
     The output lcrzo_ipopt is ended with NOOP padding
     to have a 4 bytes alignment
   Input parameter(s) :
     numberofiptimestamptosave : number of {ip,timestamp} that can
                                 be set in the lcrzo_ipopt created
			         The max value is 4
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_timeip
   Description :
     Initialise un lcrzo_ipopt avec une option timestamp.
     Le lcrzo_ipopt est termine par des padding "noop" (01)
     afin qu'il soit aligne sur 4 octets.
   Parametre(s) d'entree :
     numberofiptimestamptosave : nombre de {ip,timestamp} que
                                 l'on pourra mettre dans ipopt
			         The max value is 4
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_timeip(lcrzo_uint8 numberofiptimestamptosave,
			    lcrzo_ipopt ipopt,
			    lcrzo_uint8 *pipoptsize);

/*en Name : lcrzo_ipopt_init_timeipp
   Description :
     Initialize a lcrzo_ipopt with timestamp option.
     The output lcrzo_ipopt is ended with NOOP padding
     to have a 4 bytes alignment
   Input parameter(s) :
     ipl1, ipl2, etc. : routers to put in the option
     numberofsetipl : number of variables iplx to put in
                      ipopt. The max value is 4
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_timeipp
   Description :
     Initialise un lcrzo_ipopt avec une option timestamp.
     Le lcrzo_ipopt est termine par des padding "noop" (01)
     afin qu'il soit aligne sur 4 octets.
   Parametre(s) d'entree :
     ipl1, ipl2, etc. : routeurs a mettre dans l'option
     numberofsetipl : nombre d'adresses iplx a mettre dans
                      ipopt 
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_timeipp(lcrzo_ipl ipl1,
			     lcrzo_ipl ipl2,
			     lcrzo_ipl ipl3,
			     lcrzo_ipl ipl4,
			     lcrzo_uint8 numberofsetipl,
			     lcrzo_ipopt ipopt,
			     lcrzo_uint8 *pipoptsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_init_mixed
   Description :
     Initialize an IP option, with data presented 
     in mixed form
   Input parameter(s) :
     mixed : mixed string to use
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : mixed is too long to be put
                                 in ipopt

*/
/*fr Nom : lcrzo_ipopt_init_mixed
   Description :
     Initialise un lcrzo_ipopt par des donnees mixtes
   Parametre(s) d'entree :
     mixed: chaine mixte a utiliser
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : mixed est trop long pour
                                 etre stocke dans ipopt
*/
int lcrzo_ipopt_init_mixed(lcrzo_constmixed mixed,
			   lcrzo_ipopt ipopt,
			   lcrzo_uint8 *pipoptsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_init_stdin
   Description :
     Initialize an IP option, with data presented 
     in mixed form, and entered by the user
   Input parameter(s) :
     message : message to print before asking user
     defaulmixed : mixed string to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     ipopt : lcrzo_ipopt set with the option
     *pipoptsize : size reached by ipopt
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipopt_init_stdin
   Description :
     Initialise un lcrzo_ipopt par des donnees mixtes
     entrees par stdin
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer du mixte
     defaultmixed: chaine mixte a utiliser si l'utilisateur 
                   n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipopt : lcrzo_ipopt affecte avec l'option
     *pipoptsize : taille atteinte par ipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipopt_init_stdin(lcrzo_conststring message,
		      lcrzo_constmixed defaultmixed,
		      lcrzo_ipopt ipopt,
		      lcrzo_uint8 *pipoptsize);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_1null
   Description :
     Test if the first option in ipopt is a NULL option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ipopt starts with a NULL
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't start with a NULL
*/
/*fr Nom : lcrzo_ipopt_decode_1null
   Description :
     Teste si la premiere option de ipopt est une option NULL.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt commence par un NULL
     LCRZO_ERR_OKNOTDECODED : ipopt ne commence pas par un NULL
*/
int lcrzo_ipopt_decode_1null(lcrzo_constipopt ipopt,
			     lcrzo_uint8 ipoptsize);

/*en Name : lcrzo_ipopt_decode_1noop
   Description :
     Test if the first option in ipopt is a NOOP option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ipopt starts with a NOOP
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't start with a NOOP
*/
/*fr Nom : lcrzo_ipopt_decode_1noop
   Description :
     Teste si la premiere option de ipopt est une option NOOP.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt commence par un NOOP
     LCRZO_ERR_OKNOTDECODED : ipopt ne commence pas par un NOOP
*/
int lcrzo_ipopt_decode_1noop(lcrzo_constipopt ipopt,
			     lcrzo_uint8 ipoptsize);

/*en Name : lcrzo_ipopt_decode_1lsrr
   Description :
     Test if the first option in ipopt is a LSRR option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsetipl : number of routers set in the IP option
     *ppointer : value of the LSRR pointer
     *piplx : IP address set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt starts with a LSRR
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't start with a LSRR
*/
/*fr Nom : lcrzo_ipopt_decode_1lsrr
   Description :
     Teste si la premiere option de ipopt est une option LSRR.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsetipl : nombre de routeurs stockes dans l'option
     *ppointer : valeur du pointeur LSRR
     *piplx : adresses IP stockees dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt commence par un LSRR
     LCRZO_ERR_OKNOTDECODED : ipopt ne commence pas par un LSRR
*/
int lcrzo_ipopt_decode_1lsrr(lcrzo_constipopt ipopt,
			     lcrzo_uint8 ipoptsize,
			     lcrzo_uint8 *pnumberofsetipl,
			     lcrzo_uint8 *ppointer,
			     lcrzo_ipl *pipl1,
			     lcrzo_ipl *pipl2,
			     lcrzo_ipl *pipl3,
			     lcrzo_ipl *pipl4,
			     lcrzo_ipl *pipl5,
			     lcrzo_ipl *pipl6,
			     lcrzo_ipl *pipl7,
			     lcrzo_ipl *pipl8,
			     lcrzo_ipl *pipl9);

/*en Name : lcrzo_ipopt_decode_1ssrr
   Description :
     Test if the first option in ipopt is a SSRR option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsetipl : number of routers set in the IP option
     *ppointer : value of the SSRR pointer
     *piplx : IP address set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt starts with a SSRR
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't start with a SSRR
*/
/*fr Nom : lcrzo_ipopt_decode_1ssrr
   Description :
     Teste si la premiere option de ipopt est une option SSRR.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsetipl : nombre de routeurs stockes dans l'option
     *ppointer : valeur du pointeur SSRR
     *piplx : adresses IP stockees dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt commence par un SSRR
     LCRZO_ERR_OKNOTDECODED : ipopt ne commence pas par un SSRR
*/
int lcrzo_ipopt_decode_1ssrr(lcrzo_constipopt ipopt,
			     lcrzo_uint8 ipoptsize,
			     lcrzo_uint8 *pnumberofsetipl,
			     lcrzo_uint8 *ppointer,
			     lcrzo_ipl *pipl1,
			     lcrzo_ipl *pipl2,
			     lcrzo_ipl *pipl3,
			     lcrzo_ipl *pipl4,
			     lcrzo_ipl *pipl5,
			     lcrzo_ipl *pipl6,
			     lcrzo_ipl *pipl7,
			     lcrzo_ipl *pipl8,
			     lcrzo_ipl *pipl9);

/*en Name : lcrzo_ipopt_decode_1rr
   Description :
     Test if the first option in ipopt is a RR option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsetipl : number of routers set in the IP option
     *ppointer : value of the RR pointer
     *piplx : IP address set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt starts with a RR
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't start with a RR
*/
/*fr Nom : lcrzo_ipopt_decode_1rr
   Description :
     Teste si la premiere option de ipopt est une option RR.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsetipl : nombre de routeurs stockes dans l'option
     *ppointer : valeur du pointeur RR
     *piplx : adresses IP stockees dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt commence par un RR
     LCRZO_ERR_OKNOTDECODED : ipopt ne commence pas par un RR
*/
int lcrzo_ipopt_decode_1rr(lcrzo_constipopt ipopt,
			   lcrzo_uint8 ipoptsize,
			   lcrzo_uint8 *pnumberofsetipl,
			   lcrzo_uint8 *ppointer,
			   lcrzo_ipl *pipl1,
			   lcrzo_ipl *pipl2,
			   lcrzo_ipl *pipl3,
			   lcrzo_ipl *pipl4,
			   lcrzo_ipl *pipl5,
			   lcrzo_ipl *pipl6,
			   lcrzo_ipl *pipl7,
			   lcrzo_ipl *pipl8,
			   lcrzo_ipl *pipl9);

/*en Name : lcrzo_ipopt_decode_1time
   Description :
     Test if the first option in ipopt is a timestamp option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsettimestamp : number of timestamps set in the IP option
     *ppointer : value of the pointer field in the timestamp
     *poverflow : value of the overflow field in the timestamp
     *pflag : value of the flag field in the timestamp
     *ptimestampx : timestamps set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt starts with a timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't start with a timestamp
*/
/*fr Nom : lcrzo_ipopt_decode_1time
   Description :
     Teste si la premiere option de ipopt est une option timestamp.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsettimestamp : nombre de timestamp stockes dans l'option
     *ppointer : valeur du champ 'pointer' du timestamp
     *poverflow : valeur du champ 'overflow' du timestamp
     *pflag : valeur du champ 'flag' du timestamp
     *ptimestampx : timestamp stockes dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt commence par un timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt ne commence pas par un timestamp
*/
int lcrzo_ipopt_decode_1time(lcrzo_constipopt ipopt,
			     lcrzo_uint8 ipoptsize,
			     lcrzo_uint8 *pnumberofsettimestamp,
			     lcrzo_uint8 *ppointer,
			     lcrzo_uint8 *poverflow,
			     lcrzo_uint8 *pflag,
			     lcrzo_uint32 *ptimestamp1,
			     lcrzo_uint32 *ptimestamp2,
			     lcrzo_uint32 *ptimestamp3,
			     lcrzo_uint32 *ptimestamp4,
			     lcrzo_uint32 *ptimestamp5,
			     lcrzo_uint32 *ptimestamp6,
			     lcrzo_uint32 *ptimestamp7,
			     lcrzo_uint32 *ptimestamp8,
			     lcrzo_uint32 *ptimestamp9);

/*en Name : lcrzo_ipopt_decode_1timeip
   Description :
     Test if the first option in ipopt is a timestamp option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsetiptimestamp : number of timestamps set in the IP option
     *ppointer : value of the pointer field in the timestamp
     *poverflow : value of the overflow field in the timestamp
     *pflag : value of the flag field in the timestamp
     *piplx, *ptimestampx : IP addresses and timestamps
   Normal return values :
     LCRZO_ERR_OK : ipopt starts with a timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't start with a timestamp
*/
/*fr Nom : lcrzo_ipopt_decode_1timeip
   Description :
     Teste si la premiere option de ipopt est une option timestamp.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsetiptimestamp : nombre de timestamp stockes dans l'option
     *ppointer : valeur du champ 'pointer' du timestamp
     *poverflow : valeur du champ 'overflow' du timestamp
     *pflag : valeur du champ 'flag' du timestamp
     piplx, *ptimestampx : adresses IP et timestamps
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt commence par un timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt ne commence pas par un timestamp
*/
int lcrzo_ipopt_decode_1timeip(lcrzo_constipopt ipopt,
			       lcrzo_uint8 ipoptsize,
			       lcrzo_uint8  *pnumberofsetiptimestamp,
			       lcrzo_uint8  *ppointer,
			       lcrzo_uint8  *poverflow,
			       lcrzo_uint8  *pflag,
			       lcrzo_ipl    *pipl1,
			       lcrzo_uint32 *ptimestamp1,
			       lcrzo_ipl    *pipl2,
			       lcrzo_uint32 *ptimestamp2,
			       lcrzo_ipl    *pipl3,
			       lcrzo_uint32 *ptimestamp3,
			       lcrzo_ipl    *pipl4,
			       lcrzo_uint32 *ptimestamp4);


/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_noop
   Description :
     Test if ipopt contains a NOOP option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnbbytesnoop : number of NOOP bytes
   Normal return values :
     LCRZO_ERR_OK : ipopt contains a NOOP
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't contain a NOOP
*/
/*fr Nom : lcrzo_ipopt_decode_noop
   Description :
     Teste si ipopt contient une option NOOP.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnbbytesnoop : nombre d'octets NOOP
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt contient un NOOP
     LCRZO_ERR_OKNOTDECODED : ipopt ne contient pas un NOOP
*/
int lcrzo_ipopt_decode_noop(lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_uint8 *pnbbytesnoop);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_lsrr
   Description :
     Test if ipopt contains a LSRR option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsetipl : number of routers set in the IP option
     *ppointer : value of the LSRR pointer
     *piplx : IP address set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt contains a LSRR
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't contain a LSRR
*/
/*fr Nom : lcrzo_ipopt_decode_lsrr
   Description :
     Teste si ipopt contient une option LSRR.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsetipl : nombre de routeurs stockes dans l'option
     *ppointer : valeur du pointeur LSRR
     *piplx : adresses IP stockees dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt contient un LSRR
     LCRZO_ERR_OKNOTDECODED : ipopt ne contient pas un LSRR
*/
int lcrzo_ipopt_decode_lsrr(lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_uint8 *pnumberofsetipl,
			    lcrzo_uint8 *ppointer,
			    lcrzo_ipl *pipl1,
			    lcrzo_ipl *pipl2,
			    lcrzo_ipl *pipl3,
			    lcrzo_ipl *pipl4,
			    lcrzo_ipl *pipl5,
			    lcrzo_ipl *pipl6,
			    lcrzo_ipl *pipl7,
			    lcrzo_ipl *pipl8,
			    lcrzo_ipl *pipl9);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_ssrr
   Description :
     Test if ipopt contains a SSRR option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsetipl : number of routers set in the IP option
     *ppointer : value of the SSRR pointer
     *piplx : IP address set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt contains a SSRR
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't contain a SSRR
*/
/*fr Nom : lcrzo_ipopt_decode_ssrr
   Description :
     Teste si ipopt contient une option SSRR.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsetipl : nombre de routeurs stockes dans l'option
     *ppointer : valeur du pointeur SSRR
     *piplx : adresses IP stockees dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt contient un SSRR
     LCRZO_ERR_OKNOTDECODED : ipopt ne contient pas un SSRR
*/
int lcrzo_ipopt_decode_ssrr(lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_uint8 *pnumberofsetipl,
			    lcrzo_uint8 *ppointer,
			    lcrzo_ipl *pipl1,
			    lcrzo_ipl *pipl2,
			    lcrzo_ipl *pipl3,
			    lcrzo_ipl *pipl4,
			    lcrzo_ipl *pipl5,
			    lcrzo_ipl *pipl6,
			    lcrzo_ipl *pipl7,
			    lcrzo_ipl *pipl8,
			    lcrzo_ipl *pipl9);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_rr
   Description :
     Test if ipopt contains a RR option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsetipl : number of routers set in the IP option
     *ppointer : value of the RR pointer
     *piplx : IP address set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt contains a RR
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't contain a RR
*/
/*fr Nom : lcrzo_ipopt_decode_rr
   Description :
     Teste si ipopt contient une option RR.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsetipl : nombre de routeurs stockes dans l'option
     *ppointer : valeur du pointeur RR
     *piplx : adresses IP stockees dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt contient un RR
     LCRZO_ERR_OKNOTDECODED : ipopt ne contient pas un RR
*/
int lcrzo_ipopt_decode_rr(lcrzo_constipopt ipopt,
			  lcrzo_uint8 ipoptsize,
			  lcrzo_uint8 *pnumberofsetipl,
			  lcrzo_uint8 *ppointer,
			  lcrzo_ipl *pipl1,
			  lcrzo_ipl *pipl2,
			  lcrzo_ipl *pipl3,
			  lcrzo_ipl *pipl4,
			  lcrzo_ipl *pipl5,
			  lcrzo_ipl *pipl6,
			  lcrzo_ipl *pipl7,
			  lcrzo_ipl *pipl8,
			  lcrzo_ipl *pipl9);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_time
   Description :
     Test if ipopt contains a timestamp option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsettimestamp : number of timestamps set in the IP option
     *ppointer : value of the pointer field in the timestamp
     *poverflow : value of the overflow field in the timestamp
     *pflag : value of the flag field in the timestamp
     *ptimestampx : timestamps set with the values in the IP option
   Normal return values :
     LCRZO_ERR_OK : ipopt contains a timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't contain a timestamp
*/
/*fr Nom : lcrzo_ipopt_decode_time
   Description :
     Teste si ipopt contient une option timestamp.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsettimestamp : nombre de timestamp stockes dans l'option
     *ppointer : valeur du champ 'pointer' du timestamp
     *poverflow : valeur du champ 'overflow' du timestamp
     *pflag : valeur du champ 'flag' du timestamp
     *ptimestampx : timestamp stockes dans l'option
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt contient un timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt ne contient pas un timestamp
*/
int lcrzo_ipopt_decode_time(lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_uint8 *pnumberofsettimestamp,
			    lcrzo_uint8 *ppointer,
			    lcrzo_uint8 *poverflow,
			    lcrzo_uint8 *pflag,
			    lcrzo_uint32 *ptimestamp1,
			    lcrzo_uint32 *ptimestamp2,
			    lcrzo_uint32 *ptimestamp3,
			    lcrzo_uint32 *ptimestamp4,
			    lcrzo_uint32 *ptimestamp5,
			    lcrzo_uint32 *ptimestamp6,
			    lcrzo_uint32 *ptimestamp7,
			    lcrzo_uint32 *ptimestamp8,
			    lcrzo_uint32 *ptimestamp9);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_timeip
   Description :
     Test if ipopt contains a timestamp option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofsettimestamp : number of timestamps set in the IP option
     *ppointer : value of the pointer field in the timestamp
     *poverflow : value of the overflow field in the timestamp
     *pflag : value of the flag field in the timestamp
     *piplx, *ptimestampx : IP addresses and timestamps
   Normal return values :
     LCRZO_ERR_OK : ipopt contains a timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't contain a timestamp
*/
/*fr Nom : lcrzo_ipopt_decode_timeip
   Description :
     Teste si ipopt contient une option timestamp.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofsettimestamp : nombre de timestamp stockes dans l'option
     *ppointer : valeur du champ 'pointer' du timestamp
     *poverflow : valeur du champ 'overflow' du timestamp
     *pflag : valeur du champ 'flag' du timestamp
     piplx, *ptimestampx : adresses IP et timestamps
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt contient un timestamp
     LCRZO_ERR_OKNOTDECODED : ipopt ne contient pas un timestamp
*/
int lcrzo_ipopt_decode_timeip(lcrzo_constipopt ipopt,
			      lcrzo_uint8 ipoptsize,
			      lcrzo_uint8  *pnumberofsetiptimestamp,
			      lcrzo_uint8  *ppointer,
			      lcrzo_uint8  *poverflow,
			      lcrzo_uint8  *pflag,
			      lcrzo_ipl    *pipl1,
			      lcrzo_uint32 *ptimestamp1,
			      lcrzo_ipl    *pipl2,
			      lcrzo_uint32 *ptimestamp2,
			      lcrzo_ipl    *pipl3,
			      lcrzo_uint32 *ptimestamp3,
			      lcrzo_ipl    *pipl4,
			      lcrzo_uint32 *ptimestamp4);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipopt_decode_unknown
   Description :
     Test if ipopt contains an unknown option.
   Input parameter(s) :
     ipopt : lcrzo_ipopt to check
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     unknownipopt : lcrzo_ipopt set with the unknown options
                    of ipopt
     *punknownipoptsize : size of unknownipopt
   Normal return values :
     LCRZO_ERR_OK : ipopt contains an unknown IP option
     LCRZO_ERR_OKNOTDECODED : ipopt doesn't contain an
                              unknown IP option
*/
/*fr Nom : lcrzo_ipopt_decode_unknown
   Description :
     Teste si ipopt contient une option non reconnue.
   Parametre(s) d'entree :
     ipopt : lcrzo_ipopt a verifier
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     unknownipopt : lcrzo_ipopt affecte avec les options 
                    inconnues de ipopt
     *punknownipoptsize : taille de unknownipopt
   Valeurs de retour normales :
     LCRZO_ERR_OK : ipopt contient une option inconnue
     LCRZO_ERR_OKNOTDECODED : ipopt ne contient pas
                              d'option inconnue
*/
int lcrzo_ipopt_decode_unknown(lcrzo_constipopt ipopt,
			       lcrzo_uint8 ipoptsize,
			       lcrzo_ipopt unknownipopt,
			       lcrzo_uint8 *punknownipoptsize);

 
