""" Ciphers -- Wrapped ciphers defined in SSLeay

    (c) Marc-Andre Lemburg; All Rights Reserved; mailto: mal@lemburg.com
    See the documentation for further copyright information or contact
    the author.
"""

# Load C interface

from Crypto import mxCrypto

# Constants

ECB = mxCrypto.ECB
CBC = mxCrypto.CBC
CFB = mxCrypto.CFB
BLOCK_SIZE = mxCrypto.BLOCK_SIZE
DEFAULT_IV = '\0'*BLOCK_SIZE
DEFAULT_MODE = ECB

# Helpers
def _NOP(*args):
    pass

def _NotSupported(*args):
    raise TypeError,'method not supported'

class Cipher:

    """ Abstract base class of the ciphers.
    """

    # Underlying C object
    instance = None

    # C interface
    C_new = _NOP
    C_delete = _NOP
    C_encrypt = _NotSupported
    C_decrypt = _NotSupported
    C_getmode = _NotSupported

    def encrypt(self,data):

        return self.C_encrypt(self.instance,data)

    def decrypt(self,data):

        return self.C_decrypt(self.instance,data)

    def __del__(self):
        
        if self.instance:
            self.C_delete(self.instance)

    def __getattr__(self,name):

        if name == "mode":
            return self.C_getmode(self.instance)
        raise AttributeError,name

class StreamCipher(Cipher):

    """ Stream cipher. Allows encryption of arbitrary length data.
    """
    blocksize = mxCrypto.StreamCipher_blocksize
    keysize = mxCrypto.StreamCipher_keysize

    C_delete = mxCrypto.delete_StreamCipher

    def __init__(self,key,mode=DEFAULT_MODE):

        self.instance = self.C_new(key,mode)

class BlockCipher(StreamCipher):

    """ Block cipher. Encrypts in blocks of fixed size.
    """
    blocksize = mxCrypto.BlockCipher_blocksize
    C_reinit = mxCrypto.BlockCipher_reinit

    def __init__(self,key,mode=DEFAULT_MODE,IV=DEFAULT_IV):

        self.instance = self.C_new(key,mode,IV)
        self.IV = IV

    def __setattr__(self,name,value):

        if name == 'IV':
            self.C_reinit(self.instance,value)
        self.__dict__[name] = value

class RC2(BlockCipher):

    C_encrypt = mxCrypto.RC2Cipher_encrypt
    C_decrypt = mxCrypto.RC2Cipher_decrypt
    C_getmode = mxCrypto.RC2Cipher_getmode
    C_new = mxCrypto.new_RC2Cipher
    
class RC4(StreamCipher):

    C_encrypt = mxCrypto.RC4Cipher_encrypt
    C_decrypt = mxCrypto.RC4Cipher_decrypt
    C_getmode = mxCrypto.RC4Cipher_getmode
    C_new = mxCrypto.new_RC4Cipher

class RC5(BlockCipher):

    version = 0x10
    wordsize = 32

    C_encrypt = mxCrypto.RC5Cipher_encrypt
    C_decrypt = mxCrypto.RC5Cipher_decrypt
    C_getmode = mxCrypto.RC5Cipher_getmode
    C_new = mxCrypto.new_RC5Cipher

    def __init__(self,key,mode=DEFAULT_MODE,IV=DEFAULT_IV,rounds=16,
                 version=0x10,wordsize=32):

        self.instance = self.C_new(key,mode,IV,rounds)
        self.IV = IV
        self.rounds = rounds

class Blowfish(BlockCipher):

    C_encrypt = mxCrypto.BlowfishCipher_encrypt
    C_decrypt = mxCrypto.BlowfishCipher_decrypt
    C_getmode = mxCrypto.BlowfishCipher_getmode
    C_new = mxCrypto.new_BlowfishCipher

class IDEA(BlockCipher):

    keysize = mxCrypto.IDEACipher_keysize

    C_encrypt = mxCrypto.IDEACipher_encrypt
    C_decrypt = mxCrypto.IDEACipher_decrypt
    C_getmode = mxCrypto.IDEACipher_getmode
    C_new = mxCrypto.new_IDEACipher

class DES(BlockCipher):
    
    keysize = mxCrypto.DESCipher_keysize
    
    C_encrypt = mxCrypto.DESCipher_encrypt
    C_decrypt = mxCrypto.DESCipher_decrypt
    C_getmode = mxCrypto.DESCipher_getmode
    C_new = mxCrypto.new_DESCipher
    C_delete = mxCrypto.delete_DESCipher
    C_isWeak = mxCrypto.DESCipher_isWeak

    def isWeak(self):

        return self.C_isWeak(self.instance)

class DES3(BlockCipher):

    keysize = mxCrypto.DES3Cipher_keysize

    C_encrypt = mxCrypto.DES3Cipher_encrypt
    C_decrypt = mxCrypto.DES3Cipher_decrypt
    C_getmode = mxCrypto.DES3Cipher_getmode
    C_new = mxCrypto.new_DES3Cipher
    C_delete = mxCrypto.delete_DES3Cipher
    C_isWeak = mxCrypto.DES3Cipher_isWeak

    def isWeak(self):

        return self.C_isWeak(self.instance)

class CAST(BlockCipher):

    C_encrypt = mxCrypto.CASTCipher_encrypt
    C_decrypt = mxCrypto.CASTCipher_decrypt
    C_getmode = mxCrypto.CASTCipher_getmode
    C_new = mxCrypto.new_CASTCipher

