package FreeBSD::Ports;

use strict;
use vars qw($VERSION $Index_File);

use Carp qw(carp);
use IO::File qw(O_RDONLY);
use FreeBSD::Ports::Port ();

$VERSION	= '0.03';
$Index_File	= '/usr/ports/INDEX';


sub TIEHASH {
    my $class = shift;
    my $ports;

    if (ref $class) {
	$ports = $class->_clone;
    } else {
	$ports = bless { }, $class;
	my $filename = (shift or $Index_File);

	my $index = IO::File->new( $filename, O_RDONLY ) or carp "Cannot read file $filename";
	while (my $index_line = $index->getline) {
	    my $port = FreeBSD::Ports::Port->new($index_line);
	    $ports->{'PORTS'}{ $port->{'DISTRIBUTION_NAME'} } = $port;
	}
    }
    $ports;
}

sub FETCH {
    my($self, $port_name) = @_;
    $self->{'PORTS'}->{$port_name};
}

sub STORE { carp 'Method STORE not implemented' }
sub DELETE { carp 'Method DELETE not implemented' }
sub CLEAR { carp 'Method CLEAR not implemented' }

sub EXISTS {
    my($self, $port_name) = @_;
    exists $self->{'PORTS'}{$port_name};
}

sub FIRSTKEY {
    my $self = shift;
    return (each %{ $self->{'PORTS'} }) unless exists $self->{'SORTED'};
    $self->{'KEY_OFFSET'} = 0;
    ${ $self->{'SORTED'} }[0];
}

sub NEXTKEY {
    my $self = shift;
    return (each %{ $self->{'PORTS'} }) unless exists $self->{'KEY_OFFSET'};
    ${ $self->{'SORTED'} }[ ++($self->{'KEY_OFFSET'}) ];
}

sub _clone {
    my $self = shift;
    my $clone = bless {
	'PORTS'		=> { %{$self->{'PORTS'}} },
    }, ref $self;
    $clone->{'SORTED'} = [ @{$self->{'SORTED'}} ] if exists $self->{'SORTED'};
    $clone;
}

sub port {
    my($self, $port_name) = @_;
    $self->{'PORTS'}{$port_name};
}

sub maintainer {
    my($self, $email) = @_;
    $self->_select_string( $email, 'maintainer' );
}

sub primary_category {
    my($self, $category_name) = @_;
    $self->_select_string( $category_name, 'primary_category' );
}

sub _select_string {
    my($self, $match, $property) = @_;
    $match = lc($match);
    while ( my($port_name, $port) = each %{ $self->{'PORTS'} } ) {
	my $value = $self->{'PORTS'}{$port_name}->$property();
	unless (lc($value) eq $match) {
	    delete $self->{'PORTS'}{$port_name};
	}
    }
    delete $self->{'SORTED'};
    $self;
}

sub category {
    my($self, $category_name) = @_;
    while ( my($port_name, $port) = each %{ $self->{'PORTS'} } ) {
	unless ($self->{'PORTS'}{$port_name}->primary_category eq $category_name) {
	    unless (exists ${ $self->{'PORTS'}{$port_name}->categories }{$category_name}) {
		delete $self->{'PORTS'}{$port_name};
	    }
	}
    }
    delete $self->{'SORTED'};
    $self;
}

sub run_depends {
    my($self, $port) = @_;
    $self->_depends( $port, 'run_depends' );
}

sub build_depends {
    my $self = shift;
    $self->_depends( shift, 'build_depends' );
}

sub depends {
    my $self = shift;
    $self->_depends( shift, 'all_depends' );
}

sub _depends {
    my($self, $depends_proto, $depends_type) = @_;
    my $depends_name = ref $depends_proto ? $depends_proto->name : $depends_proto;

    while ( my($port_name, $port) = each %{ $self->{'PORTS'} } ) {
	unless (exists ${ $self->{'PORTS'}{$port_name}->$depends_type() }{$depends_name}) {
	    delete $self->{'PORTS'}{$port_name};
	}
    }
    delete $self->{'SORTED'};
    $self;
}

sub sort {
    my($self, $method, $field) = @_;
    carp 'Sorting method not defined' unless defined $method;
    $field = 'DISTRIBUTION_NAME' unless defined $field;

    my @sorted;
    my $ports = $self->{'PORTS'};
    if ($method eq 'alpha') {
	@sorted = sort { $ports->{$a}{$field} cmp $ports->{$b}{$field} } keys %{ $self->{'PORTS'} };
    } elsif ($method eq 'rev_alpha') {
	@sorted = sort { $ports->{$b}{$field} cmp $ports->{$a}{$field} } keys %{ $self->{'PORTS'} };
    } else {
        carp "Invalid sorting method: $method";
    }

    $self->{'SORTED'} = \@sorted;
    $self;
}

sub match {
    my($self, $term, $field, $insensitive) = @_;
    carp 'Search term not defined' unless defined $term;
    $field = 'COMMENT' unless defined $field;

    my $match_sub = (defined $insensitive and $insensitive == 1) ?
	sub { $_[0]->{$_[1]} =~ m!$_[2]!io } :
	sub { $_[0]->{$_[1]} =~ m!$_[2]!o };

    while ( my($port_name, $port) = each %{ $self->{'PORTS'} } ) {
	unless ( &$match_sub($port, $field, $term) ) {
	    delete $self->{'PORTS'}{$port_name};
	}
    }

    delete $self->{'SORTED'};
    $self;
}

1;

__END__


=head1 NAME

FreeBSD::Ports - Class for parsing FreeBSD's Ports INDEX

=head1 WARNING!

This is a very early version of this module.  The interface to the class
may be changed in the future.  The documentation needs improving.

Consequently, suggestions, comments and patches are extremely welcome!
I believe the correct place to discuss this module is the freebsd-doc
mailing list at freebsd-doc@freebsd.org.

=head1 SYNOPSIS

  # Describe ports maintained by tom@FreeBSD.org, sorted alphabetically
  use FreeBSD::Ports;
  my $ports = tie my %port, 'FreeBSD::Ports', '/usr/ports/INDEX';
  $ports->maintainer('tom@FreeBSD.org');
  $ports->sort('alpha');
  foreach my $p (values %port) {
      print $p->as_ascii,"\n";
  }

  # How many ports are there currently?
  use FreeBSD::Ports;
  my $ports = tie my %port, 'FreeBSD::Ports', '/usr/ports/INDEX';
  my $count = scalar keys %port;
  print "There are $count ports\n";

  # List ports containing 'MPEG' in their comment and present in the
  # 'audio' category?
  use FreeBSD::Ports;
  my $ports = tie my %port, 'FreeBSD::Ports', '/usr/ports/INDEX';
  $ports->category('audio');
  $ports->match('mpeg', 'COMMENT', 1);
  foreach my $name (keys %port) {
      print "$name\n";
  }

  # Any ports which are under the 'www' or 'net' category
  use FreeBSD::Ports;
  my $all_ports = tie my %port, 'FreeBSD::Ports';
  my $www_ports = tie my(%www_port), $all_ports;
  $www_ports->category('www');
  my $net_ports = tie my(%net_port), $all_ports;
  $net_ports->category('net');
  my %www_or_net_port;
  while ( my($key, $value) = (each(%www_port), each(%net_port)) ) {
      $www_or_net_port{$key} = $value;
  }


=head1 DESCRIPTION

C<FreeBSD::Ports> is a simple interface to the INDEX file used in
FreeBSD's ports collection.

This class uses Perl's I<tie> interface.  See L<perltie> for more
information.

For further information, or to obtain the latest version of this module,
see <URL:http://people.FreeBSD.org/~tom/portpm/>.

=head1 METHODS

=over 4

=item $ports = tie my %port, $class, $filename

Read and parse an INDEX file.  C<$filename> is the name of the file
containing the index.  If undefined, F</usr/ports/INDEX> is used.

C<$class> should be the name of the class (C<FreeBSD::Ports>) or another
C<FreeBSD::Ports> object to be cloned.

C<%port> is a hash whose keys are the names of the ports within the
index.  The value of each key is a
L<FreeBSD::Ports::Port|FreeBSD::Ports::Port> object.

C<$ports> is an object which can be accessed using the methods within
this class.

=item $port = $ports->port($port_name)

Returns the L<FreeBSD::Ports::Port|FreeBSD::Ports::Port> object whose
name is C<$port_name> if that object exists.

=item $ports->maintainer($email)

Selects only those ports whose maintainer's e-mail address is C<$email>.
Addresses are matched case insensitively.

=item $ports->primary_category($category_name)

Selects only those ports whose primary category is C<$category_name>.
The primary category is the first category in which a port is listed.
This represents the directory under F</usr/ports> in which a port is
stored.  Category names are matched case insensitively.

=item $ports->category($category_name)

Selects only those ports which are present in the category named
C<$category_name>.  In this case, category names are matched case
sensitively.

=item $ports->run_depends($port)

Selects only those ports which have a run dependency on the port
represented by C<$port>.

C<$port> is either the name of a port or a
L<FreeBSD::Ports::Port|FreeBSD::Ports::Port> object.

=item $ports->build_depends($port)

Selects only those ports which have a build dependency on C<$port>.

=item $ports->depends($port)

Selects those ports which have any type of dependency on C<$port>.

=item $ports->sort($method, $field)

Sorts the selected ports. C<$method> specifies how the ports should be
sorted.  I<alpha> means the ports should be sorted alphabetically, from
A to Z.  I<rev_alpha> means the ports should be sorted alphabetically in
reverse, from Z to A.  Sorting is performed case sensitively.

C<$field> specifies which property of the ports should be used to sort
by.  I<DISTRIBUTION_NAME> is used if this value is undefined.  One of
the following properties is usually selected:

=over 8

=item I<COMMENT>: Comment used to describe the port

=item I<DISTRIBUTION_NAME>: Port's name

=item I<MAINTAINER>: Maintainer's e-mail address

=item I<PRIMARY_CATEGORY>: Port's primary category

=back

=item $ports->match($term, $field, $insensitive)

Selects only those ports where C<$field> matches C<$term>.  The values
which can be used for C<$field> are the same as those used for the
C<sort> method.  However, for this method C<COMMENT> is used if
C<$field> is undefined.

C<$term> is evaluated as a regular expression.  If C<$insensitive> is
defined, the expression is evaluated case insensitively.  If undefined,
case sensitive evalutaion is used.

=head1 CREDITS

This module is written by Tom Hukins E<lt>tom@FreeBSD.orgE<gt>.

Thanks to Nik Clayton for encouragement and assistance.

=head1 COPYRIGHT

This module is distributed under the same license as FreeBSD
<http://www.FreeBSD.org/copyright/freebsd-license.html>.

=cut
