# Name:: FormatR
# Description:: Perl like formats for ruby
# Author:: Paul Rubel (rubel@crhc.uiuc.edu)
# Release:: 1.05
# Homepage:: http://www.crhc.uiuc.edu/~rubel/FormatR.html
# Date:: 17 April 2002
# License:: You can redistribute it and/or modify it under the same term as Ruby.
#           Copyright (c) 2002  Paul Rubel
#
#     THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
#     IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
#     WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
#     PURPOSE.
# = To Test this code:
# Try test_format.rb with no arguments. If nothing is amiss you should
# see OK (??/?? tests ?? asserts). This tests the format output
# against perl output (which is in the test directory if you don't
# have perl). If you would like to see the format output try
# test_format.rb --keep which will place the test's output in the file
# format_testfile{1-10}
#
# = Usage                                
# Class FormatR::Format in module FormatR provides perl like formats for ruby. 
# For a summary of the methods you're likely to need please see FormatR::Format.
# Formats are used to create output with a similar format but with changing
# values.                                  
#                                          
# For example:                             
#     require "format.rb"                  
#     include FormatR                      
#                                          
#     top_ex = <<DOT                       
#        Piggy Locations for @<< @#, @### 
#                          month, day, year
#                                          
#     Number: location              toe size
#     -------------------------------------------
#     DOT                                  
#                                          
#     ex = <<TOD                           
#     @)      @<<<<<<<<<<<<<<<<       @#.##
#     num,    location,             toe_size
#     TOD                                  
#                                          
#     body_fmt = Format.new (top_ex, ex)           
#                                          
#     body_fmt.setPageLength(10)           
#     num = 1                              
#                                          
#     month = "Sep"                        
#     day = 18                             
#     year = 2001                          
#     ["Market", "Home", "Eating Roast Beef", "Having None", "On the way home"].each {|location|
#         toe_size = (num * 3.5)           
#         body_fmt.printFormat(binding)    
#         num += 1                         
#     }                                    
#                                          
#                                          
# When run, the above code produces the following output:  
#        Piggy Locations for Sep 18, 2001  
#                                          
#     Number: location              toe size
#     -------------------------------------------
#     1)      Market                   3.50
#     2)      Home                     7.00
#     3)      Eating Roast Beef       10.50
#     4)      Having None             14.00
#     5)      On the way home         17.50
#                                          
#                                          
# More examples are found in test_format.rb
#
# = Supported Format Fields
#
# ===Standard perl formats 
# These are explained at http://www.perldoc.com/perl5.6.1/pod/perlform.html and include:
# * left justified text, @<<<
# * right justified text, @>>
# * centered text @||| all of whose length is the number of characters in the
#   field. 
#
# * It also supports fields that start with a ^ which signifies that
#   the input is a large string and after being printed the variable
#   should have the printed portion removed from its value.
#
# * Numeric formats of the form @##.## which let you decide where you
#   want a decimal point. It will add extra zeroes to the fractional part
#   but if the whole portion is too big will write it out regardless
#   of your specification (regarding the whole as more important than the
#   fraction). 
#
# * A line that starts with ~ will be suppressed if it will be blank
#
# * A line that starts with ~~ will repeat until it is blank, be sure
#   to use this feature with at least one field starting with a ^.
#
# === Scientific formats of the form @.#G##, @.#g##, @.#E##, and @.#e##
# *  The use of G, g, E, and e is consistent to their use in printf. 
#
# * If a G or g is specified the number of characters before the
#   exponent, excluding the decimal point, will give the number of
#   significant figures to be used in the output. For example:
#   @.##G### with the value 1.234e-14 will print 1.23E-14 which has 3
#   significant figures. This format @##.###g### with the value
#   123.4567E200 produces 1.23457e+202, with 6 significant figures. 
#   The capitalization of G effects whether the e is lower- or upper-case.
#
# * If a E or e is used the number of hashes between the decimal
#   point and the E or e tells how many digits to print after the decimal
#   point. The number of hashes after the precision argument just adds to the
#   number of spaces available, I can't see how to reasonably adjust
#   that given the other constraints.  For example the format
#   @##.#E### with the value 123.4567E200 produces 1.2E+202 since
#   there is only one hash after the decimal point. 
#
# * More examples of using the scientific formats can be found in test_format.rb
#
#
# = Changes:
# 
# ==1.05
# * Hugh Sasse sent in a patch to clean up warnings. I was sloppy with my  
#   spacing but hopefully have learned better. Thanks Hugh!
#
# * Fixed a bug in repeating lines using ~~ when the last line wouldn't get
#   placed correctly unless it ended with a ' '
#
# * Fixed a bug where a line that started with a <,>, or | would loose
#   this character if there wasn't a @ or ^ before it. 
#   The parsing of the non-picture parts of a picture line is greatly 
#   improved.
#
# ==1.04
# * Added a scientific notation formatter so you can use @#.###E##,
#   @##.##e##, @#.###G##, or @##.##g##.  The use of G and E is
#   consistent to their use in printf. If a G or g is specified the
#   number of characters before the exponent excluding the decimal
#   point will give the number of significant figures to be used in the
#   output. If a E or e is used the number of hashes between the decimal
#   point and the E tells how many digits to print after the decimal
#   point. The number of hashes after the E just adds to the
#   number of spaces available, I can't see how to reasonably adjust
#   that given the other constraints.  
#
# ==1.03
# * If perl isn't there use cached output to test against.
#
# * better packaging, new versions won't write over the older ones when 
#   you unpack
#
# * Changed the Format.new call. In the past you could pass in an IO
#   object as a second parameter. You now need to use the Format.io=
#   method as the "signature" of Format.new has changed as shown
#   below. None of the examples used the second parameter so hopefully
#   it's safe to change
#
# * Added optional arguments to Format.new so you can set top, body, and middle
#   all at once like so Format.new(top, middle, bottom) or even Format.new(top, middle).
#   If you want a bottom without a top you'll either need to call setBottom or pass nil
#   or an empty format for top like so Format.new (nil, middle, bottom)
#
# * Made the testing script clean up after itself unless you pass the -keep flag
#
# * Modified setTop and setBottom so you can pass in a string or an array of strings
#   that can be used to specify a format instead of having to create one yourself.
#   Thanks again to Hugh Sasse for not settling for a second rate interface.
# 
# * Move test_format.rb over to runit. 
#
# * Added functionality so that if you pass in a format string, or array of strings
#   to setTop or setBottom it does the right thing. This way you don't need to make the 
#   extra formats just to pass them in. 
#
#
# ==1.02
# * Allow formats to be passed in as arrays of strings as well as just long strings
#
# * Added functionality so that if the first format on a page is too
#   long to fit on that page it will be printed partially with a
#   bottom. Perl seems to just print the whole thing and ignore the page
#   size in this case.
#
# * Fixed a bug where if your number didn't have a fractional part it
#   would crash if you used a format that need a fractional portion like @##.##
#
# * On the recommendation of Hugh Sasse added
#   finishPageWithoutFF(aBinding, io=@io) and 
#   finishPageWithFF(aBinding, io=@io) which will print out blank
#   lines until the end of the page and then print the bottom, with
#   and without a ^L. Only works on fixed sized bottoms.
#
# ==1.01
# * Moved to rdoc for generating documentation.
#
# ==1.00
# * Bottoms work iff you have a fixed size format and print out a
#   top afterwords. This means that you will only get a bottom if you
#   will print a top right after it so the last format page you print
#   won't have a bottom. It's impossible to figure out if you are
#   done with the format and therefore need to print the
#   bottom. Perhaps in a future release we can just take fixed sized
#   bottoms off the available size and get them to work that way.
# * Added support for Format.pageNumber()
# * Support ~ to be a space
# * Support ~ to suppress lines when the variables are empty
# * Support ~~ to repeat until the variables are empty
# * Support comments. If the first character in a line is a # the
#   line is a comment
# * Testing now compares against perl, it's a bit easier than
#   writing the tests manually.
# ==0.93
# * Added support for the ^ character to start a format
# 
# == 0.92
# * Added end of page characters and introduced line counts. 
#
# * Added the ability to manipulate the line count in case you write
#   to the file handle yourself
#
# * Added format sizes. They just give the number of lines in the
#   current format. They don't try to iterate and get some total
#   count including tops and bottoms.
#
#
# = Incompatibilities/Issues
#
# * The first format printed on a page won't see any changes done on 
#   account of printing the top. This needs to be fixed.
#
# * If you use bottom be sure to check that you're happy with the
#   output. It doesn't currently work with variable sized bottoms. You
#   can use the finishPageWith{out}FF(...) methods to print out a
#   bottom if you're done printing but haven't finished a page.
#
# * Watch out for @#@??? as formats, see [ruby-talk:27782] and
#   [ruby-talk:27734]. This should be fixed in a future version of
#   ruby. The basic problem is that the here documents are equivalent
#   to "" and not '', they will evaluate variables in them. If this is
#   a problem be sure to just make a long string with '' and pass that
#   in. The next version of FormatR should support arrays as
#   constructor arguments.
#
# * Rounding seems to be broken in perl, if you try to print the following
#   format with 123.355 you won't get the same answer, you'll get 123.35 and
#   123.36. FormatR rounds "correctly" and plans to unless there is a
#   convincing reason not to.
#    format TEST_FORMAT =
#      ^#.### ^##.##
#    $num,  $num
#
#
# =To Do/Think about:
# * blank out undefined @##.# values with ~
# 
# *  use rbconfig to install?
#
# *  Is there a better name than resetPage?
#
# *  Hugh Sasse: The only other thing I wanted from Perl formats, which was not there,
#    was a means to set the maximum width, and create picture lines
#    computationally, so I could decide I wanted this and that on the left,
#    such and such on the right, and *the rest* (the middle) filled out with
#    some data without having to bang away on the < key for ages, hoping
#    I got the width right.
#
# *  Fix variable sized bottoms better. I'm not sure if this is
#    possible. You could try computing it first but this would cause
#    trouble if it depends upon the body format. I'm currently planning
#    to just live with fixed sized bottoms.
#
# * Page numbering is messy and doesn't play well with buffering lines
#   that go on the next page. Perl gets away from this by writing more
#   lines than will fit on a page when a format overflows. Because we cache
#   them for later their values may look "old" if they go after a top that 
#   was evaluated later but printed earlier. This needs help!
#
# =Thanks go to
# Hugh Sasse for his enlightening comments and suggestions. He has been incredibly
# helpful in making this package usable.

module FormatR

  # an exception that we can throw
  class FormatException < Exception
  end

  # This class holds a single block of text, either something
  # unchanging or a picture element of some format.
  class FormatEntry
    attr_accessor :val

    def initialize (val, space)
      @space = space
      @val = val
      if (!space)
        s = val.size - 1
        
        if (val =~ /[@^][<]{#{s},#{s}}/)
          @formatter = LeftFormatter.new(val)
        elsif (val =~ /[@^][>]{#{s},#{s}}/)     
          @formatter = RightFormatter.new(val)
        elsif (val =~ /[@^][\|]{#{s},#{s}}/)
          @formatter = CenterFormatter.new(val)
        elsif (val =~ /[@^](#*)([\.]{0,1})(#*)([eEgG])(#+)/)
          @formatter = ScientificNotationFormatter.new($1, $2, $3, $4, $5)
        elsif (val =~ /[@^](#*)([\.]{0,1})(#*)/)
          @formatter = NumberFormatter.new($1, $2, $3)
        else
          raise FormatException.new(), "Malformed format entry \"#{@val}\""
        end
      end
    end
    
    # is this just unchanging characters
    def isUnchanging ()
      return @space
    end

    # give back the string passed through the appropriate formatter
    def formatString (string, var_name=nil, aBinding=nil)
      result = @formatter.formatString(string, var_name, aBinding)
      return result
    end
  end

  # This is the base class for all the formats, <,>,|, and # of the
  # @ or ^ persuasion. It keeps track of filled variables and the length
  # string should have.
  class Formatter
    def initialize (val)
      @len = val.size()
      @filled = false
      if (val =~ /\^.*/)
        @filled = true
      end
    end

    #if it's a filled field chop the displayed stuff off in the context given
    def changeVarValue (var_value, var_name, aBinding)
      result = var_value[0, @len]
      max_space = var_value[0,@len + 1].rindex(' ')
      if (var_value.length <= @len)
        result = var_value
        max_space = @len
      end
      if (max_space != nil)
        result = var_value[0,max_space]
      end
      s = "#{var_name} = #{var_name}[#{result.size()},#{var_value.size()}].gsub(/^\\s*/,'');" +
        "if #{var_name} == nil then #{var_name} = ''; end"
      eval(s, aBinding)
      return result
    end
    
    # return a formatted string of the correct length
    def formatString (var_value, var_name, binding)
      result = var_value[0,@len]
      if (! @filled)
        return result
      end
      return changeVarValue(var_value, var_name, binding)
    end


  end

  # this format doesn't care if it's a @ or an ^, it acts the same and doesn't chop things
  # used for @##.## formats
  class NumberFormatter < Formatter
    def initialize (wholeString, radix, fractionString)
      @whole = wholeString.size + 1 # for the '@'
      @fraction = fractionString.size
      @radix = radix.size #should always be 1
      @len = @whole + @fraction + @radix
    end
    
    # given a string that's a number spit it back with the right number of digits
    # and rounded the correct amount.
    def formatString (s, unused_var_name=nil, unused_aBinding=nil)
      num = s.split('.') # should this take into account internationalization?
      if (s.size == 1)
        return formatInt(s)
      end

      res = num[0]
      spaceLeft = @fraction + @radix
      if (res.size > @whole)
        spaceLeft = @len - res.size()
      end
      if (spaceLeft > 0)
        res += '.'  
        spaceLeft -= 1
      end
      res += getFract(num, spaceLeft) if (spaceLeft > 0)

      max = @len
      if (res.size > max)
        res = res[0,max]
      end
      res.rjust(max)
    end

    def formatInt (s)
      s.to_s.ljust(@len)
    end

    # what portion of the number is after the decimal point and should be printed
    def getFract (num, spaceLeft)
      num[1] = "" if (num[1].nil?)
      @fraction.times {num[1] += '0'}
      fract = num[1][0,spaceLeft + 1]
      if (fract.size() >= spaceLeft + 1)
        if ((fract[spaceLeft,1].to_i) >= 5 )
          fract[spaceLeft - 1, 1] = ((fract[spaceLeft - 1, 1].to_i) + 1).to_s
        end
      end
      return fract[0,spaceLeft] 
    end
  end

  # make a formatter that will spit out scientific notation
  class ScientificNotationFormatter < Formatter
    # Make a new formatter that will print out in scientific notation
    def initialize (whole, radix, fraction, precision_g_e, exponent)
      @total_size = ("@" + whole + radix + fraction + precision_g_e + exponent).size 
      @fraction = fraction.length
      @sig_figs = ("@" + whole + fraction).length
      @g_e = precision_g_e
    end

    def formatString (s, unused_var_name=nil, unused_aBinding=nil)
      #might want to put a %0 to pad w/ 0's
      precision = ((@g_e =~ /[Ee]/) ? @fraction : @sig_figs)
      result = sprintf("%#{@total_size}.#{precision}#{@g_e}", s)
      result
    end
  end

  ## Format things that go to the left, ala <
  class LeftFormatter < Formatter
    def initialize (val)
      super
    end

    def formatString (s, var_name, binding)
      s = super 
      s.ljust(@len)
    end
  end

  ## Format things that go to the right, ala >
  class RightFormatter < Formatter
    def initialize (val)
      super
      @len = val.size()
    end

    def formatString (s, var_name, binding)
      s = super 
      s.rjust(@len)
    end
  end

  ## Format things that go to the center, ala |
  class CenterFormatter < Formatter
    def initialize (val)
      super
      @len = val.size()
    end

    def formatString (s, var_name, binding)
      s = super
      s.center(@len)
    end
  end


  # The class that exports the functionality that a user is interested in.
  class Format 
    public 
    # Set the IO that the format will be printed to, from stdout to a
    # file for example.
    attr_accessor  :io

    # Print out the specified format. You need to pass in a Binding
    # object for the variables that will be used in the format. This is
    # usually just a call to Kernel.binding. The next argument gives a
    # file handler to print to. This is useful so that a top or
    # bottom's output get written to the same place as the main format
    # is going even if their formats have a different io when they're
    # not attached.
    def printFormat (aBinding, io = @io)
      @binding = aBinding
      return printBodyFormat (io)
    end

    # When you don't want anymore on this page just fill it with blank
    # lines and print the bottom if it's there, print a ^L also. This
    # is good if you want to finish off the page but print more later
    # to the same file.
    def finishPageWithFF (aBinding, io = @io)
      finishPage(aBinding, false, io)
    end

    # When you don't want anymore on this page just fill it with blank
    # lines and print the bottom if it's there. Don't print a ^L at
    # the end. This is good if this will be the last page.
    def finishPageWithoutFF (aBinding, io = @io)
      finishPage(aBinding, true, io)
    end

    # Return how many times the top has been printed. You can use this
    # to number pages. An empty top can be used if you need the page
    # number but don't want to print a header
    def pageNumber ()
      return @page_number
    end

    # How big is the format? May be useful if you want to try a bottom
    # with a variable length format
    def getSize ()
      @format_length
    end
    
    # If you want something to show up before the regular text of a
    # format you can specify it here. It will be printed once above
    # the format it is being set within. You can pass in either a
    # format or the specification of a format and it will make one for you.

    def setTop (format)
      top_format = format
      if (format.class != Format)
        top_format = Format.new(format)
      end
      raise FormatException.new(), "recursive format not allowed" if (top_format == self)
      @top = top_format 

    end

    # Set a format to print at the end of a page. This is tricky and
    # you should be careful using it. It currently has problems on
    # short pages (at least). In order for a bottom to show up you
    # need to finish off a page. This means that formats less than a
    # page will need to be finished off with a call to one of the
    # finishPageWith[out]FF methods.
    def setBottom (format)
      bottom_format = format
      if (format.class != Format)
        bottom_format = Format.new(format);
      end
      raise FormatException, "recursive format not allowed" if (bottom_format == self)
      @bottom = bottom_format
    end

    # Sets the number of lines on a page. If you don't want page breaks
    # set this to some large number that you hope you won't offset or
    # liberally use resetPage. The default is 60.
    def setPageLength (len)
      @page_length = len
      resetPage()
    end

    # Sets the variable that says how many lines may be printed to the
    # maximum for the page which can be set using setPageLength (anInt).
    # Defaults to 60.
    def resetPage ()
      @lines_left = @page_length
    end

    # If you're writing to the file handle in another way than by
    # calling printFormat you can keep the pagination working using
    # this call to correctly keep track of lines.
    def addToLineCount (line_change)
      @lines_left += line_change
    end

    
    # Create a new format with the given top, bottom, and middle
    # formats. One argument will default to a top while two will give
    # you a top and a middle. If you want a bottom and no top you'll
    # need to pass an empty format in as the first argument or a
    # nil. The output defaults to standard out but can be changed with
    # the Format.io= method. 
    #  The format is a string in the style of a perl format or an array
    # of strings each of which is a line of a perl format. The passed
    # in format contains multiple lines, picture lines and argument
    # lines. A picture line can contain any text but if it contains an
    # at field (@ followed by any number of <,>,| or a group of #'s of
    # the format #*.#* or #*) it must be followed by an argument
    # line. The arguments in the argument line are inserted in place
    # of the at fields in the picture line. Perl documentation for
    # formats can be found here:
    # http://www.cpan.org/doc/manual/html/pod/perlform.html
    # An example of a format is
    #  format = <<DOT
    #  Name: @<<<       @<<<<<
    #        first_name last_name
    #  DOT
    # 
    # This line specifies that when requested one line should be
    # printed and that it will say "Name: #{first_name} #{last_name}\n" but
    # that if either of those variables is longer than the length its
    # format the result will be truncated to the length of the format.
    # 
    # An at field specified as @<* specifies that the variable should
    # be left justified within the space allocated. @>* is right
    # justified, and @| is centered. #'s are used to print numbers and
    # can be used to set the number of digits after the decimal
    # point. However the whole number portion of an argument will
    # always be printed in its entirety even if it takes space set for
    # the fractional portion or even more space. If the fractional
    # portion is not long enough to fill the described space it will be
    # padded with 0s.
    def initialize (mid_or_top, mid_format=nil, bottom_format=nil)
      if (mid_or_top.nil?)
        raise FormatException.new(),
          " You need to pass in at least one non-nil argument"
      end
      @io = $stdout
      @picture_lines = []
      @vars = []
      @top = @bottom = nil
      @page_length = 60
      @lines_left = @page_length
      @format_length = 0

      @buffered_lines = []
      @print_top = true
      @printed_a_body = false
      @print_bottom = false

      @page_number = 1
      
      lines = ((mid_format.nil?) ? mid_or_top : mid_format)
      if (lines.class == String)
        lines = lines.split( '\n' )
      end

      expected_vars = 0
      lines.each {|line|
        if (line =~ /^#.*/)
          #don't do anything, it's a comment
        elsif (0 != expected_vars)
          expected_vars = getVarLine(line, expected_vars)
        else
          expected_vars = getPictureLine(line, expected_vars)
          @format_length += 1
        end
      }

      setTop(mid_or_top) if mid_format
      setBottom(bottom_format) if bottom_format
    end


    # Things you shouldn't have to deal with.
    private

    # When you don't want anymore on this page just fill it with blank
    # lines and print the bottom if it's there
    def finishPage (aBinding, suppressFF, io)
      @binding = aBinding
      tryOutputFormat(io)
      bottom_size = 0
      (bottom_size = @bottom.getSize()) if (@bottom)
      (@lines_left - bottom_size).times { io.puts("")}
      @print_bottom = true
      tryPrintBottom(io, suppressFF)
    end
        
    # pull out the formatting
    def getPictureLine (line, expected_vars)
      num_vars = line.count('@') + line.count('^')
      if (num_vars != 0)
        expected_vars = num_vars
      else #the next line is also a picture line, so no vars this time
        @vars.push([])
      end
      nonFormats = getNonFormats(line)
      formats = getFormats(line)
      a = FormatHolder.new()
      a.repeat = (line =~ /^~~.*/) ? true :false
      a.suppress = (line =~ /^~.*/) ? true : false
      nonFormats.each_index {|i|
        puts "adding non #{nonFormats[i]} " if (nonFormats[i] && nonFormats[i] =~ /.*oo/)
        a.push( FormatEntry.new( nonFormats[i], true )) if ( nonFormats[i] )
        a.push( FormatEntry.new( formats[i], false )) if ( formats[i] )
      }
      @picture_lines.push(a)
      return expected_vars
    end

    # what variables should be put into the picture line above
    def getVarLine (line, expected_vars)
      vars = line.split(',')
      if (vars.size != expected_vars)
        raise FormatException.new(),"malformed format, not enough variables provided.\n" +
          "Be sure to separate using commas:" +
          "Expected #{expected_vars} but received '#{line}'" 
      end
      @vars.push(vars)
      expected_vars = 0
      return expected_vars
    end

    # pull out each individual format from a line and return a list of
    # them
    def getFormats (line)
      last_found = line.size()
      output = []
      var_count = line.count('@') + line.count('^')
      var_count.times {|i|
        last_found = findFormatBefore(last_found, line, output)
      }
      output
    end

    # find a format before the position given in last_found and shove
    # it on the output
    def findFormatBefore (last_found, line, output)
      first_hat = line.rindex('^',last_found)
      first_at = line.rindex('@',last_found)
      first_hat = -1 if !first_hat 
      first_at  = -1 if !first_at
      first_index = (first_hat > first_at) ? first_hat : first_at
      first_char = (first_hat > first_at) ? '^' : '@'

      line_section = line[(first_index + 1),(last_found - first_index)]
      # all the formats that we could have, blech this is ugly
      #num_re = 0
      [ /^(>+)[^>]*/,          # 1
        /^(\|+)[^\|]*/,        # 2
        /^(#*\.{0,1}#*[EeGg]#+).*/, # 3 for scientific notation 
        /^(#+\.{0,1}#*).*/,    # 5 notice that *+  for ones without a fraction
        /^(#*\.{0,1}#+).*/,    # 6            +*  or a whole
        /^(<*)[^<]*/           # 7
      ].each {|re|
        #num_re += 1
        if (line_section =~ re)
          #puts "line #{line_section} matches re #{num_re}"
          output.unshift(first_char + $1)
          last_found = (first_index - 1)
          return last_found
        end
      }
    end

    # split the string into groupings that start with an @ or a ^
    def splitByAtOr (picture_line)
      return [picture_line] unless picture_line.index(/[@^]/)
      ats = []
      chars = picture_line.split('')
      index = 0
      chars.each {|c|
        if (c =~ /[@^]/)
          ats.push(index)
        end
        index += 1
      }
      ats2 = []
      if (ats[0] == 0)
        ats2.push([0,0])
      else
        ats2.push( [0, ats[0]]) unless (ats[0] == 0)
      end
      ((ats.length) - 1).times { |i|
        ats2.push( [ats[i],ats[i+1] ] )
      }
      ats2.push( [ats[ats.length-1], chars.length] )
      result = []
      ats2.each {|i|
        result.push( picture_line[i[0]...i[1]])
      }
      result
    end

    # pull out from a picture line the components of the line that aren't formats
    def getNonFormats (picture_line) 
      #puts "splitting #{picture_line}"
      lines = splitByAtOr( picture_line)
      output = []
      lines = lines.each {|element|
        element.gsub!(/^[@^]#*\.{0,1}#*[EeGg]#+/, '')
        element.gsub!(/^[@^]#+\.{0,1}#*/, '')
        element.gsub!(/^[@^]#*\.{0,1}#+/, '')
        element.gsub!(/^[@^]>+/, '')
        element.gsub!(/^[@^]\|+/, '')
        element.gsub!(/^[@^]<*/, '')
        element.gsub!(/^[@^]/, '')
        element.gsub!(/~/, ' ')
        output.push(element)
      }
      return output
    end

    ## print related functions

    # Try to save a line for outputting and perhaps a top and bottom.
    # we need the whole format to be able to print so buffer until we
    # get it
    def printLine (line, io, suppress = false)
      if (!suppress)
        line.gsub!(/\s*$/, "")
        @buffered_lines.push("#{line}\n")
        tryPrintPartialPage(io) 
        tryPrintBottom(io)
        tryPrintTop(io) 
        # There is a bug with the top. When we buffer lines if the bottom
        # won't fit we first buffer them and then print the top and bottom.
        # the buffered lines have values before the top and bottom are printed but
        # appear after them, ugh, it's a big mess. I can see why perl doesn't bother
        # with bottoms!
      end
    end

    # if the page is too short to hold even one format just print what
    # we can
    def tryPrintPartialPage (io)
      if (!@printed_a_body)
        bottom_size = 0
        bottom_size = @bottom.getSize() if (@bottom)
        if ((@buffered_lines.size() + bottom_size) == @lines_left)
          printBufferedLines(io)
          @print_bottom = true
        end
      end
    end

    # When we have a whole format try to print it, if there isn't
    # enough room we have to save it for later.
    def tryOutputFormat (io) 
      bottom_size = 0
      bottom_size = @bottom.getSize() if (@bottom)
      #puts "buffered size #{@buffered_lines.size} bottom #{bottom_size} left: #{@lines_left}"
      if ((@buffered_lines.size() + bottom_size) <= @lines_left)
        printBufferedLines(io)
      else
        @print_bottom = true
      end 
    end
    
    #print the buffered lines
    def printBufferedLines (io)
      io.puts(@buffered_lines)
      @lines_left -= (@buffered_lines.size())
      @buffered_lines = []
      @printed_a_body = true
    end

    # see if a top is the right thing to print
    def tryPrintTop (io)
      if (@print_top)
        @printed_a_body = false
      end
      if (@print_top && @top)
        @print_top = false
        @top.resetPage()
        lines = @top.printFormat(@binding, io)
        @lines_left = (@page_length - lines)
      end
    end

    #we have a bottom, even if it's only ^L, try to get this working!
    def tryPrintBottom (io, suppressLF = false)
      bottom_size = 0
      bottom_size = @bottom.getSize() if (@bottom)

      #this bottom is a mess if we have repeating lines
      if (@print_bottom)
        @print_bottom = false
        if (@bottom)
          @bottom.printFormat(@binding, io) 
          @bottom.resetPage() 
        end
        @lines_left = @page_length
        io.print "\f" unless suppressLF
        @page_number += 1
        @print_top = true
        return true
      end
      return false
    end

    # The workhorse of printFormat, buffer the format contained and
    #  then try to print it out
    def printBodyFormat (io)
      @picture_lines.each_index {|i| # There isn't a 1-1 correspondence here anymore!
        line = @picture_lines[i]
        suppress = line.suppress()
        repeat = true
        while (repeat) 
          outputLine = ""
          vars = @vars[i].dup if @vars[i]
          line.each {|item|
            if (item.isUnchanging()) 
              outputLine << "#{item.val}"
            else
              begin
                to_eval = vars.shift
                raise FormatException.new(), 
                  "Not enough variables supplied to match format" if (to_eval.nil?)
                s = eval(to_eval, @binding)
              rescue NameError
                raise NameError.new(), "cannot find variable '#{to_eval}'"
              end
              suppress = false if (("" != s) && suppress) 
              res = item.formatString(s.to_s, to_eval, @binding)
              outputLine << "#{res}"
            end
          }
          printLine(outputLine, io, suppress)
          if ((false == suppress) && line.repeat)
            suppress = line.suppress()
          else
            repeat = false
          end
        end #while
      }
      tryOutputFormat(io)
      return (@page_length - @lines_left)
    end

  end # class Format


  # a subclass of array that knows about ~ and ~~
  class FormatHolder < Array
    attr_accessor :suppress, :repeat
  end

end
