/* FileViewer.java
 *
 * created: Thu Nov 19 1998
 *
 * This file is part of Artemis
 *
 * Copyright (C) 1998,1999,2000  Genome Research Limited
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/FileViewer.java,v 1.29 2000/11/01 14:58:55 kmr Exp $
 */

package diana.components;

import diana.*;

import java.awt.*;
import java.awt.event.*;
import java.io.Reader;
import java.io.BufferedReader;
import java.io.IOException;

/**
 *  This class implements a simple file viewer.  In fact any Reader object can
 *  be viewed.
 *
 *  @author Kim Rutherford
 *  @version $Id: FileViewer.java,v 1.29 2000/11/01 14:58:55 kmr Exp $
 *
 **/

public class FileViewer
  extends Frame {
  /**
   *  Create a new FileViewer component and make it visible.
   *  @param label The name to attach to the new Frame.
   **/
  public FileViewer (final String label) {
    this (label, true);
  }

  /**
   *  Create a new FileViewer component.
   *  @param label The name to attach to the new Frame.
   *  @param visible The new FileViewer will be made visible if and only if
   *    this argument is true.
   **/
  public FileViewer (final String label, final boolean visible) {
    super (label);

    setLayout(new BorderLayout ());

    final Font font = Options.getOptions ().getFont ();

    setFont (font);

    text_area = new TextArea ("", 18, 90, TextArea.SCROLLBARS_VERTICAL_ONLY);
    text_area.setEditable (false);
    text_area.setFont (font);
    add (text_area, "Center");

    button_panel = new Panel ();
    add (button_panel, "South");

    close_button = new Button ("Close");
    close_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent e) {
        dispose ();
      }
    });

    button_panel.add (close_button);

    addWindowListener (new WindowAdapter () {
      public void windowClosing (WindowEvent event) {
        dispose ();
      }
    });

    addComponentListener (new ComponentAdapter () {
      public void componentResized (ComponentEvent e) {
        saved_size = FileViewer.this.getSize ();
        saved_position = FileViewer.this.getLocation ();
      }
      public void componentMoved (ComponentEvent e) {
        saved_size = FileViewer.this.getSize ();
        saved_position = FileViewer.this.getLocation ();
      }
    });

    pack ();

    setSize (original_size);

    final Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();

    if (saved_position == null ||
        saved_position.x < 10 || saved_position.y < 10 ||
        saved_position.x + saved_size.width > screen.width ||
        saved_position.y + saved_size.height > screen.height) {
      saved_position = new Point ((screen.width - getSize ().width) / 2,
                                  (screen.height - getSize ().height) / 2);
    }

    setLocation (saved_position);

    setVisible (visible);
  }

  /**
   *  Clear the viewer window.
   **/
  public void clear () {
    text_area.setText ("");
  }

  /**
   *  Read from the given Reader and append the text to this FileViewer.
   *  @param read_stream The stream to read the contents of the viewer from.
   **/
  public void appendFile (Reader read_stream)
      throws IOException {
    final BufferedReader buffered_reader = new BufferedReader (read_stream);

    String line;

    while ((line = buffered_reader.readLine ()) != null) {
      appendString (line + "\n");
      Thread.yield ();
    }

    buffered_reader.close ();
  }

  /**
   *  Clear the viewer window and display the lines from the given Reader.
   *  @param read_stream The stream to read the contents of the viewer from.
   **/
  public void readFile (Reader read_stream)
      throws IOException {
    final BufferedReader buffered_reader = new BufferedReader (read_stream);

    String line;

    final StringBuffer line_buffer = new StringBuffer ();

    while ((line = buffered_reader.readLine ()) != null) {
      line_buffer.append (line).append ('\n');
    }

    buffered_reader.close ();

    final String new_text = line_buffer.toString ();

    text_area.setText (new_text);
  }

  /**
   *  Clear the viewer window and display the lines from the given String.
   *  @param read_string The string to read the contents of the viewer from.
   **/
  public void setText (String read_string) {
    if (!read_string.equals (text_area.getText ())) {
      text_area.setText (read_string);
    }
  }


  /**
   *  Clear the viewer window and display the lines from the given String.
   *  @param read_string The string to read the contents of the viewer from.
   **/
  public void appendString (String read_string) {
    text_area.append (read_string);
  }

  /**
   *  Return a String containing the text that this component is displaying.
   **/
  public String getText () {
    return getTextArea ().getText ();
  }

  /**
   *  Destroy this component.
   **/
  public void dispose () {
    setVisible (false);
    super.dispose ();
  }

  /**
   *  return the TextArea component from this FileViewer.
   **/
  public TextArea getTextArea () {
    return text_area;
  }

  /**
   *  Return the Panel containing the close button of this FileViewer.
   **/
  protected Panel getButtonPanel () {
    return button_panel;
  }

  /**
   *  Pressing this button will distroy the Frame.
   **/
  private Button close_button;

  /**
   *  A Panel to hold the close button.
   **/
  private Panel button_panel;

  /**
   *  The main component we use for displaying the file.
   **/
  private TextArea text_area = null;

  /**
   *  This is the original value of saved_size.
   **/
  private static Dimension original_size = new Dimension (850, 550);

  /**
   *  The size of the last FileViewer Frame to be resized.  When a new
   *  FileViewer component is created it is given the current value of this
   *  variable.  This is updated when any FileViewer frame is resized.
   **/
  private static Dimension saved_size = original_size;

  /**
   *  The position of the last FileViewer Frame to be moved.  When a new
   *  FileViewer component is created it is given the current value of this
   *  variable.  This is updated when any FileViewer frame is moved.
   **/
  private static Point saved_position = null;
}
