/* EnsemblCorbaEntry.java
 *
 * created: Thu Oct 14 1999
 *
 * This file is part of Artemis
 * 
 * Copyright (C) 1999  Genome Research Limited
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/uk/ac/sanger/pathogens/embl/EnsemblCorbaEntry.java,v 1.5 2000/07/24 11:12:33 kmr Exp $
 */

package uk.ac.sanger.pathogens.embl;

import uk.ac.sanger.pathogens.*;

import collections.*;

//import Ensembl.artemis.Entry;
//import Ensembl.artemis.Feature;

import java.util.Vector;
import java.io.IOException;
import java.util.NoSuchElementException;

/**
 *  This class implements the Entry interface with the data for the entry
 *  coming from an EnsEMBL Corba server.
 *
 *  @author Kim Rutherford
 *  @version $Id: EnsemblCorbaEntry.java,v 1.5 2000/07/24 11:12:33 kmr Exp $
 **/

public class EnsemblCorbaEntry extends ReadOnlyEntry
    implements Entry {
  /**
   *  Create a new EnsemblCorbaEntry object from the given handle.
   *  @param data This is the corba object that we will read from.
   **/
  public EnsemblCorbaEntry (final Ensembl.artemis.Entry corba_handle)
      throws LocationParseException, InvalidKeyException {
    this.corba_handle = corba_handle;
    this.sequence = new EnsemblCorbaSequence (corba_handle.getSequence ());

    grabFeatures ();
    this.entry_name = corba_handle.getName ();
  }

  /**
   *  Return the text of the EMBL header of this Entry or null if there is no
   *  header.
   **/
  public String getHeaderText () {
    return null;
  }

  /**
   *  Return the name of this Entry.
   **/
  public String getName () {
    return entry_name;
  }

  /**
   *  Return a count of the number of Feature objects in this Entry.
   **/
  public int getFeatureCount () {
    return features.size ();
  }

  /**
   *  Return the ith Feature from this Entry.  This Features are returned in a
   *  consistent order, sorted by the first base of each Feature.
   **/
  public Feature getFeatureAtIndex (int arg_index) {
    final CollectionEnumeration enumerator = features.elements ();

    int i = 0;

    while (enumerator.hasMoreElements ()) {
      final Feature this_feature = (Feature) enumerator.nextElement ();
      
      if (i == arg_index) {
        return this_feature;
      }

      ++i;
    }

    return null;
  }

  /**
   *  Return the index of the given Feature.  This does the reverse of
   *  getFeatureAtIndex ().
   **/
  public int indexOf (Feature feature) {
    final CollectionEnumeration enumerator = features.elements ();

    int i = 0;

    while (enumerator.hasMoreElements ()) {
      final Feature this_feature = (Feature) enumerator.nextElement ();
      
      if (this_feature == feature) {
        return i;
      }
      ++i;
    }

    return -1;
  }

  /**
   *  Returns true if and only if this Entry contains the given feature.
   **/
  public boolean contains (Feature feature) {
    return features.includes (feature);
  }

  /**
   *  Returns an enumeration of the Feature objects in this Entry.  The
   *  returned Enumeration object will generate all features in this object in
   *  turn. The first item generated is the item at index 0, then the item at
   *  index 1, and so on.
   **/
  public FeatureEnumeration features () {
    return features.features ();
  }

  /**
   *  Return a vector containing the references of the Feature objects within
   *  the given range.
   *  @param range Return features that overlap this range - ie the start of
   *    the feature is less than or equal to the end of the range and the end
   *    of the feature is greater than or equal to the start of the range.
   *  @return The non-source key features of this feature table the are within
   *    the given range.  The returned object is a copy - changes will not
   *    effect the FeatureTable object itself.
   **/
  public FeatureVector getFeaturesInRange (Range range) {
    return features.getFeaturesInRange (range);
  }

  /**
   *  Return a vector containing the references of all the Feature objects in
   *  this Entry.
   *  @return The non-source key features of this Entry.  The returned object
   *    is a copy - changes will not effect the Entry object itself.
   **/
  public FeatureVector getAllFeatures () {
    final FeatureVector return_features = new FeatureVector ();

    final CollectionEnumeration enumerator = features.elements ();

    while (enumerator.hasMoreElements ()) {
      final Feature this_feature = (Feature) enumerator.nextElement ();
      
      return_features.add (this_feature);
    }

    return return_features;
  }

  /**
   *  Return the Sequence object from this entry or null if it does not
   *  contain one.
   *  @return a Sequence object for this Entry.  the returned object is
   *    not a copy - changes to it will change the Entry object itself
   **/
  public Sequence getSequence () {
    return sequence;
  }

  // XXX
  public static EntryInformation default_entry_information =
    new SimpleEntryInformation ();

  /**
   *  Return the EntryInformation object for this Entry.
   **/
  public EntryInformation getEntryInformation () {
    return default_entry_information; 
  }

  /**
   *  Read the features from the corba_handle and put them in a FeatureTree.
   **/
  private void grabFeatures ()
      throws LocationParseException, InvalidKeyException {
    final Ensembl.artemis.Feature [] feature_handles =
      corba_handle.getAllFeatures ();

    for (int i = 0 ; i < feature_handles.length ; ++i) {
      try {
        final Feature new_feature =
          new EnsemblCorbaFeature (feature_handles [i]);

//        System.out.println ("adding new feature: " + new_feature.getKey () +
//                            " " + new_feature.getLocation ());

        if (! new_feature.getKey ().equals ("source")) {
          features.add (new_feature);
        }
      } catch (InvalidRelationException e) {
        System.out.println ("exception while reading: " + e);
      }
    }
  }

  /**
   *  The EmblSeq object that was passed to the constructor.
   **/
  private Ensembl.artemis.Entry corba_handle;

  /**
   *  This is created in the constructor 
   **/
  private EnsemblCorbaSequence sequence = null;

  /**
   *  The FeatureTree containing the features of this Entry.
   **/
  private FeatureTree features = new FeatureTree (new FeatureComparator ());

  /**
   *  The name of the entry as read from the corba_handle.
   **/
  final String entry_name;
}
