/*************************************************
* Public Key Interface Header File               *
* (C) 1999-2002 The Botan Project                *
*************************************************/

#ifndef BOTAN_PUBKEY_H__
#define BOTAN_PUBKEY_H__

#include <botan/base.h>
#include <botan/pk_keys.h>
#include <botan/pk_misc.h>

namespace Botan {

/*************************************************
* Public Key Encryptor                           *
*************************************************/
class PK_Encryptor
   {
   public:
      SecureVector<byte> encrypt(const byte[], u32bit) const;
      SecureVector<byte> encrypt(const SecureVector<byte>&) const;
      u32bit maximum_input_size() const;
      PK_Encryptor(const PK_Encrypting_Key&, EME* = 0);
      PK_Encryptor(const PK_Encrypting_Key&, const std::string&);
      ~PK_Encryptor() { delete encoder; }
   private:
      const PK_Encrypting_Key& key;
      const EME* encoder;
   };

/*************************************************
* Public Key Decryptor                           *
*************************************************/
class PK_Decryptor
   {
   public:
      SecureVector<byte> decrypt(const byte[], u32bit) const;
      SecureVector<byte> decrypt(const SecureVector<byte>&) const;
      PK_Decryptor(const PK_Decrypting_Key&, EME* = 0);
      PK_Decryptor(const PK_Decrypting_Key&, const std::string&);
      ~PK_Decryptor() { delete encoder; }
   private:
      const PK_Decrypting_Key& key;
      const EME* encoder;
   };

/*************************************************
* Public Key Signer                              *
*************************************************/
class PK_Signer
   {
   public:
      SecureVector<byte> sign_message(const byte[], u32bit);
      void update(const byte[], u32bit);
      SecureVector<byte> signature();
      PK_Signer(const PK_Signing_Key&, EMSA*);
      PK_Signer(const PK_Signing_Key&, const std::string&);
      ~PK_Signer() { delete emsa; }
   private:
      const PK_Signing_Key& key;
      EMSA* emsa;
   };

/*************************************************
* Public Key Verifier Base Class                 *
*************************************************/
class PK_Verifier
   {
   public:
      bool verify_message(const byte[], u32bit, const byte[], u32bit);
      void update(const byte[], u32bit);
      bool valid_signature(const byte[], u32bit);
      PK_Verifier(EMSA*);
      PK_Verifier(const std::string&);
      virtual ~PK_Verifier() { delete emsa; }
   protected:
      virtual bool validate_signature(const SecureVector<byte>&,
                                      const byte[], u32bit) = 0;
      EMSA* emsa;
   };

/*************************************************
* PK_Verifier_with_MR                            *
*************************************************/
class PK_Verifier_with_MR : public PK_Verifier
   {
   public:
      PK_Verifier_with_MR(const PK_Verifying_with_MR_Key&, EMSA*);
      PK_Verifier_with_MR(const PK_Verifying_with_MR_Key&, const std::string&);
   private:
      bool validate_signature(const SecureVector<byte>&, const byte[], u32bit);
      const PK_Verifying_with_MR_Key& key;
   };

/*************************************************
* PK_Verifier_wo_MR                              *
*************************************************/
class PK_Verifier_wo_MR : public PK_Verifier
   {
   public:
      PK_Verifier_wo_MR(const PK_Verifying_wo_MR_Key&, EMSA*);
      PK_Verifier_wo_MR(const PK_Verifying_wo_MR_Key&, const std::string&);
   private:
      bool validate_signature(const SecureVector<byte>&, const byte[], u32bit);
      const PK_Verifying_wo_MR_Key& key;
   };

/*************************************************
* PK_Key_Agreement_Scheme                        *
*************************************************/
class PK_Key_Agreement_Scheme
   {
   public:
      SymmetricKey derive_key(const byte[], u32bit, u32bit) const;
      SymmetricKey derive_key(const SecureVector<byte>&, u32bit) const;
      PK_Key_Agreement_Scheme(const PK_Key_Agreement_Key&, KDF* = 0);
      PK_Key_Agreement_Scheme(const PK_Key_Agreement_Key&, const std::string&);
      ~PK_Key_Agreement_Scheme() { delete kdf; }
   private:
      const PK_Key_Agreement_Key& key;
      KDF* kdf;
   };

typedef PK_Key_Agreement_Scheme PK_Key_Agreement;

}

#endif
