#!/usr/bin/perl
#  common.pl - common routines shared by many files
#
#  Written by Curtis Olson.  Started August 22, 1994.
#
#  Copyright (C) 1994 - 1999  Curtis L. Olson  - curt@me.umn.edu
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

# $Id: common.pl,v 1.2 2000/01/02 19:08:02 curt Exp $


package CBB;

use strict;    # don't take no guff


# We need a version number
$CBB::version = "Version 0.8.1";
($CBB::junk, $CBB::version_num, $CBB::junk) = split(/ +/, $CBB::version);


# Contributed by Christopher Browne, Oct. 24/94
sub pad { 
    return sprintf("%02d", $_[0]); 
}


# return the directory of a file name 
sub file_dirname {
    my($file) = @_;
    my($pos);

    $pos = rindex($file, "/");
    if ( $pos >= 0 ) {
	return substr($file, 0, ($pos + 1));
    } else {
	return "./";
    }
}


# return the base file name
sub file_basename {
    my($file) = @_;
    my($pos);

    $pos = rindex($file, "/");
    return substr($file, ($pos + 1));
}


# return the file name root (ending at last ".")
sub file_root {
    my($file) = @_;
    my($pos);

    $pos = rindex($file, ".");
    return substr($file, 0, $pos);
}


# return the file name extension (starting at first ".")
sub file_extension {
    my($file) = @_;
    my($pos);

    $pos = rindex($file, ".");
    return substr($file, ($pos + 1));
}


# return current date in a nice format
sub nice_date {
    my($date_fmt) = @_;

    my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = 
        localtime(time);
    $year += 1900;

    if ( $date_fmt == 1 ) {
        return &pad($mon+1) . "/" . &pad($mday) . "/" . &pad($year);
    } else {
        return &pad($mday) . "." . &pad($mon+1) . "." . &pad($year);
    }
}


# return current date in a raw format
sub raw_date {
    my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = 
        localtime(time);
    $year += 1900;

    return &pad($year) . &pad($mon+1) . &pad($mday);
}

# start date: return date in raw format, takes argument of those types:
# -[num]m months (eg. "-0m" means only current month, "-1m" means current and last)
# -[num]d days (eg. "-10m" means 10 days)
# dd.mm.yy[yy] : "international" format
# mm/dd/yy[yy] : "us" format
# yyyymmdd     : "raw" format
#
# This can get a bit complicated, thank god we don't have to care whether
# we return invalid days

sub start_date {
    my($idate) = @_;
    my($odate, $value);
    my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = 
        localtime(time);
    $year += 1900;
    $mon = $mon + 1;

    if ( $idate =~ /^\d{8}$/ ) {	 # "raw" format
	$odate = $idate;
    } elsif ($idate =~ /^-\d{1,2}m$/ ) {	# "month" format

	$value = substr($idate, 1, 3);	# a maximum of 99 months !
	if ($value >= $mon) {
		$year = $year - 1 - int( ($value - $mon) / 12 );
		$value = ($value % 12 );
	}
	$mon = $mon - $value;
	if ($mon < 1) {
		$value = $value + 12; 
	}
	$odate = &pad($year) . &pad($mon) . &pad(1);

    } elsif ($idate =~ /^-\d{1,3}d$/ ) {	# "day" format

	$value = substr($idate, 1, 4);	# a maximum of 999 days !
	if ($value >= $yday) {
		$year = $year - 1 - int( ($value - $yday) / 360 );
		$value = ( $value % 360 );
	}
	if ($value >= $mday) {
		$mon = $mon - 1 - int( ($value - $mday) / 30 );
		if ($mon < 1) {
			$mon = $mon + 12;
		}
		$value = ( $value % 30 );
	}
	$mday = $mday - $value;
	if ($mday < 1) {
		$mday = $mday + 30;
	}
	$odate = &pad($year) . &pad($mon) . &pad($mday);

    } elsif ( $idate =~ /^\d{1,2}\/\d{1,2}\/\d{2,4}$/ ) { 	# "us" format

	($mon, $mday, $year) = split(/\//, $idate);
	if ($year < 100) {
		$year += &century() * 100;
	}
	$odate = &pad($year) . &pad($mon) . &pad($mday);

    } elsif ( $idate =~ /^\d{1,2}\.\d{1,2}\.\d{2,4}$/ ) { 	# "int" format

	($mday, $mon, $year) = split(/\./, $idate);
	if ($year < 100) {
		$year += &century() * 100;
	}
	$odate = &pad($year) . &pad($mon) . &pad($mday);

    } else {	# nonsense, give them everything since 1900
	$odate = "19000101";
    }

    return ($odate);
}

# return the current century in the form 19, 20, 21, etc.
# requires the Unix "date" command to be in the path
sub century {
   my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = 
       localtime(time);

   my($century) = sprintf("%d", (1900 + $year) / 100);

   return ($century);
}


1;				# need to return a true value
