// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of ccaudio.
// 
// The exception is that, if you link the ccaudio library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccaudio library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccaudio.  If you copy code from other releases into a copy of
// ccaudio, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccaudio, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#ifndef	__CCXX_AUDIO_H__
#define	__CCXX_AUDIO_H__

#ifndef	__EXPORT
#define __EXPORT
#endif

typedef enum
{
	SAMPLE_RATE_UNKNOWN,
	SAMPLE_RATE_6KHZ = 6000,
	SAMPLE_RATE_8KHZ = 8000,
	SAMPLE_RATE_44KHZ = 44100
}	samplerate_t;

typedef enum
{
	UNKNOWN_AUDIO_ENCODING = 0,
	G721_ADPCM_ENCODING,
	G722_AUDIO_ENCODING,
	G722_7BIT_ENCODING,
	G722_6BIT_ENCODING,
	G723_3BIT_ENCODING,
	G723_5BIT_ENCODING,
	GSM_VOICE_ENCODING,
	MULAW_AUDIO_ENCODING,
	ALAW_AUDIO_ENCODING,
	OKI_ADPCM_ENCODING,
	DIALOGIC_ADPCM_ENCODING,
	CDA_STEREO_ENCODING,
	CDA_MONO_ENCODING,
	PCM8_STEREO_ENCODING,
	PCM8_AUDIO_ENCODING,
	PCM16_STEREO_ENCODING,
	PCM16_AUDIO_ENCODING,
	PCM32_STEREO_ENCODING,
	PCM32_AUDIO_ENCODING
}	audioencoding_t;

typedef enum
{
	AUDIO_FORMAT_RAW,
	AUDIO_FORMAT_SUN,
	AUDIO_FORMAT_RIFF,
	AUDIO_FORMAT_WAVE
} audioformat_t;

typedef enum
{
	AUDIO_SUCCESS = 0,
	AUDIO_READ_LASTFRAME,
	AUDIO_NOT_OPENED,
	AUDIO_END_OF_FILE,
	AUDIO_START_OF_FILE,
	AUDIO_RATE_UNSUPPORTED,
	AUDIO_ENCODING_UNSUPPORTED,
	AUDIO_READ_INTERRUPTED,
	AUDIO_WRITE_INTERRUPTED,
 	AUDIO_READ_FAILURE,
	AUDIO_WRITE_FAILURE,
	AUDIO_READ_INCOMPLETE,
	AUDIO_WRITE_INCOMPLETE,
	AUDIO_REQUEST_INVALID,
	AUDIO_TOC_FAILED,
	AUDIO_STAT_FAILED,
	AUDIO_INVALID_TRACK,
	AUDIO_PLAYBACK_FAILED,
	AUDIO_NOT_PLAYING
} audioerror_t;

typedef struct
{
	audioformat_t format;
	audioencoding_t encoding;
	unsigned rate;
	unsigned order;
	char *annotation;
}	audioinfo_t;

bool ismono(audioencoding_t encoding);
bool issterio(audioencoding_t encoding);
samplerate_t samplerate(audioencoding_t encoding);
int sampleframe(audioencoding_t encoding, int samples = 0);
int samplecount(audioencoding_t);
unsigned long tosamples(audioencoding_t encoding, size_t bytes);
unsigned long tobytes(audioencoding_t encoding, unsigned long samples);
void samplefill(unsigned char *addr, int samples, audioencoding_t encoding);

class __EXPORT AudioFile
{
private:
	union
	{
		int fd;
		void *handle;
	}	file;

	char *pathname;
	audioerror_t error;
	audioinfo_t info;
	unsigned long header;		// offset to start of audio
	unsigned long minimum;		// minimum sample size required

	void Initialize(void);
	void getWaveFormat(int size);

protected:
	virtual char *getContinuation(void)
		{return NULL;};

	audioerror_t setError(audioerror_t err);

	unsigned short getaushort(unsigned char *data);
	void setaushort(unsigned char *data, unsigned short value);
	unsigned long getaulong(unsigned char *data);
	void setaulong(unsigned char *data, unsigned long value);

public:
	AudioFile(const char *fname, unsigned long samples = 0);
	AudioFile(const char *fname, audioinfo_t *info, unsigned long min = 0);

	AudioFile()
		{Initialize();};

	~AudioFile()
		{Close();};

	void Open(const char *fname);
	void Create(const char *fname, audioinfo_t *info);
	void Close(void);
	audioerror_t getSamples(void *addr, unsigned samples);
	audioerror_t putSamples(void *addr, unsigned samples);
	audioerror_t Skip(long samples);
	audioerror_t setPosition(unsigned long samples = ~0l);
	audioerror_t getInfo(audioinfo_t *info);
	audioerror_t setMinimum(unsigned long samples);
	unsigned long getPosition(void);
	bool isOpen(void);

	inline audioencoding_t getEncoding(void)
		{return info.encoding;};

	inline audioformat_t getFormat(void)
		{return info.format;};

	inline unsigned getSampleRate(void)
		{return info.rate;};

	inline char *getAnnotation(void)
		{return info.annotation;};

	inline audioerror_t getError(void)
		{return error;};

	inline bool operator!(void)
		{return !isOpen();};
};

class __EXPORT CDAudio
{
private:
	union
	{
		int fd;
	} file;
	unsigned char v0, v1;
#ifdef	__WIN32__
	CRITICAL_SECTION crit;
	bool paused;
	bool opened;
	char position[20];
	char endmark[24];
	char ret[256];
	DWORD command(char *fmt, ...);
#endif
	audioerror_t err;

public:
	CDAudio(int devnbr = 0);
	~CDAudio();

	audioerror_t Play(int start, int end = 0);	
	audioerror_t Stop(void);
	audioerror_t Pause(void);
	audioerror_t Resume(void);
	audioerror_t Eject(void);
	audioerror_t Reload(void);
	int getFirst(void);
	int getLast(void);
	bool isPaused(void);
	bool isAudio(int track);
	bool isOpen(void);

	unsigned char getVolume(int speaker);
	void setVolume(unsigned char left, unsigned char right);

	inline unsigned char getVolumeLeft(void)
		{return getVolume(0);};

	inline unsigned char getVolumeRight(void)
		{return getVolume(1);};

	inline audioerror_t getError(void)
		{return err;};

	inline bool operator!(void)
		{return !isOpen();};
};

class AudioSample
{
private:
	audioencoding_t encoding;
	unsigned rate;
	unsigned count;
	unsigned char *samples;

public:
	AudioSample(unsigned size, audioencoding_t coding = PCM16_AUDIO_ENCODING, unsigned rate = 8000);
	~AudioSample();

	inline unsigned getCount(void)
		{return count;};

	inline unsigned getRate(void)
		{return rate;};

	inline audioencoding_t getEncoding(void)
		{return encoding;};

	inline unsigned char *getSamples(void)
		{return samples;};
};	

class AudioTone : public AudioSample
{
private:
	unsigned f1, f2;

public:
	AudioTone(unsigned size, unsigned f1, unsigned rate = 8000);
	AudioTone(unsigned size, unsigned f1, unsigned f2, unsigned rate  = 8000);

	inline unsigned getFreq1(void)
		{return f1;};

	inline unsigned getFreq2(void)
		{return f2;};
};
#endif

