/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* diamultiline.h
 * --------------
 * A multiline is a line that can add handles to itself if someone tries to
 * drag a line segment. A multiline is created like a normal line, only its
 * behaviour after creation is different
 */

#ifndef __DIA_MULTI_LINE_H__
#define __DIA_MULTI_LINE_H__

#include <diacanvas/diadynline.h>

DIA_OPEN

#define DIA_MULTI_LINE(x) ((DiaMultiLine*) x)
/* The minimal angle between two lines. if the angle is smaller, the middle
 * handle will be removed. */
#define DIA_MULTI_LINE_SIGMA 0.1

typedef struct _DiaMultiLine DiaMultiLine;

typedef enum
{
  /* bit 0-3 and are used by DIA_OBJECT_STATE_*.
   * bit 4-6 are used by DiaBaseLine. */
  DIA_MULTI_LINE_MOUSE_ON_LINE = 1 << 8,
} DiaMultiLineFlags;

struct _DiaMultiLine
{
  DiaDynLine   line;
  DiaHandle   *handle_on_line;
};

DiaObjectType* dia_multi_line_get_type ();

void           dia_multi_line_init (DiaMultiLine *line);

/* Returns the handle that starts the line segment POINT lays on. Point will
 * be changed so it lays perfectly on the line. */
DiaHandle*     dia_multi_line_point_on_line (DiaMultiLine *line, Point *point);

/* Add a handle to the graph at POINT, this will cause a new segment to
 * be created. PREV_HANDLE is the first handle of the segment that is about
 * to be split.
 * The CP's will also be updated.
 * NOTE: The point has to be placed between the first and the last point, the
 *       line can not be extended. 
 */
DiaHandle*  dia_multi_line_add_line_segment (DiaMultiLine *line, Point *point,
					     DiaHandle *prev_handle);

/* Remove handles that lay on one line. If handles were removed the BB
 * is recalculated and the REDRAW request is set.
 */
gboolean       dia_multi_line_update_handles (DiaMultiLine *line);


DIA_CLOSE

#endif /* __DIA_MULTI_LINE_H__ */



