/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* diagraph.h
 * ----------
 * A graph is a line that connects two objects (elements) together. Most of
 * the technical drawing methods are based on the graph theory. The difference
 * between graphs and lines is that graphs can't exist with only one (or zero)
 * line ends connected.
 *
 * Graphs are only allowed to connect with elements that are inherited from
 * DiaElement.
 */

#ifndef __DIA_GRAPH_H__
#define __DIA_GRAPH_H__

#include <diacanvas/diamultiline.h>

DIA_OPEN

#define DIA_GRAPH(x) ((DiaGraph*) x)

typedef struct _DiaGraph DiaGraph;

#include <diacanvas/diaelement.h>

typedef enum
{
  /* Bit 0-3 and are used by DIA_OBJECT_STATE_*;
   * bit 4-6 are used by DiaBaseLine; 
   * bit 8 is used by DiaMultiLine. */
  DIA_GRAPH_IS_NEW = 1 << 10, /* set this if the graph is new
			       * (DIA_BASE_LINE_NEW will be reset as
			       * soon as a move occurs. */
} DiaGraphFlags;

struct _DiaGraph
{
  DiaMultiLine   line;

  /* a centric line will always point its ends to the center of the
   * element it is connected to. */
  gint           centric: 1;
  
  /* temporaly values (no need to copy): */
  DiaLineStyle   stored_style;
  DiaElement    *elem;
};

DiaObjectType* dia_graph_get_type ();

void           dia_graph_init (DiaGraph *line);

/* create a graph marked by the positions in **POS. The head of the graph is
* connected to ELEM1 and the other end to ELEM2. */
DiaObject*     dia_graph_new (Point **pos, DiaElement *elem1,
			      DiaElement *elem2);

/* find an element to connect to. */
DiaElement* dia_graph_find_dia_element (DiaGraph *graph, DiaLayer *layer,
					Point *point);

/* If the handle is the first handle, it returns the next;
 * if a handle is the last handle it returns the previous, otherwise
 * it returns NULL.
 */
DiaHandle*     dia_graph_get_next_handle (DiaGraph *graph, DiaHandle *handle);

DIA_CLOSE

#endif /* __DIA_GRAPH_H__ */



