/* JDBC Connection Pool
 * Copyright (C) 1998 James Cooper
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * For more information about this software, visit:
 * http://www.bitmechanic.com/projects/
 */

package com.bitmechanic.sql;

import java.sql.*;
import java.util.*;
import java.io.InputStream;
import java.math.BigDecimal;
import java.sql.Date;
import com.bitmechanic.util.*;

/**
 * Individual pool associated with a JDBC datasource and username.  Each pool
 * is identified by an alias.  Use the alias name to acquire a ref to a
 * Connection from the pool.  See the ConnectionPoolManager for further
 * details on how to do this.
 *
 * @see ConnectionPoolManager
 * @author James Cooper
 * @version $Id: ConnectionPool.java,v 1.11 1998/12/16 01:15:22 pixel Exp $
 */
public class ConnectionPool {

  private Vector connVector;
  private Queue  waitingThreads;
  private String url, username, password, alias;
  private int maxConn, timeoutSeconds, checkoutSeconds, numCheckoutTimeout;
  private int numRequests, numWaits, maxCheckout;

  private boolean trace;

  /**
   * Creates a Connection pool with no maxCheckout parameter.
   *
   * @param alias Name of the pool
   * @param url JDBC URL to connect to
   * @param username JDBC username to connect as
   * @param password username's password in the database
   * @param maxConn Maximum number of connections to open; When this limit
   *                is reached, threads requesting a connection are queued
   *                until a connection becomes available
   * @param timeoutSeconds Maximum number of seconds a Connection can go
   *                       unused before it is closed
   * @param checkoutSeconds Maximum number of seconds a Thread can checkout a
   *                        Connection before it is closed and returned to the
   *                        pool.  This is a protection against the Thread 
   *                        dying and leaving the Connection checked out
   *                        indefinately
   */
  public ConnectionPool(String alias,
			String url, String username, String password, 
			int maxConn, int timeoutSeconds, int checkoutSeconds) {
    this(alias, url, username, password, maxConn, timeoutSeconds,
	 checkoutSeconds, 0);
  }

  /**
   * Creates a Connection pool
   *
   * @param alias Name of the pool
   * @param url JDBC URL to connect to
   * @param username JDBC username to connect as
   * @param password username's password in the database
   * @param maxConn Maximum number of connections to open; When this limit
   *                is reached, threads requesting a connection are queued
   *                until a connection becomes available
   * @param timeoutSeconds Maximum number of seconds a Connection can go
   *                       unused before it is closed
   * @param checkoutSeconds Maximum number of seconds a Thread can checkout a
   *                        Connection before it is closed and returned to the
   *                        pool.  This is a protection against the Thread 
   *                        dying and leaving the Connection checked out
   *                        indefinately
   * @param maxCheckout If this is greater than 0, the number of times a
   *                    Connection may be checked out before it is closed. 
   *                    Used as a safeguard against cursor leak, which occurs
   *                    if you don't call ResultSet.close() and
   *                    Statement.close() 
   */
  public ConnectionPool(String alias,
			String url, String username, String password, 
			int maxConn, int timeoutSeconds, int checkoutSeconds,
			int maxCheckout) {

    this.timeoutSeconds = timeoutSeconds;
    this.checkoutSeconds = checkoutSeconds;

    this.alias = alias;
    this.url = url;
    this.username = username;
    this.password = password;
    this.maxConn = maxConn;
    this.maxCheckout = maxCheckout;

    this.numRequests = 0;
    this.numWaits = 0;
    this.numCheckoutTimeout = 0;

    connVector = new Vector(maxConn);
    waitingThreads = new Queue();

    trace = false;
  }

  /**
   * Turns tracing on or off.  If turned on, verbose messages about the pool
   * will be printed to STDERR
   */
  public void setTracing(boolean on) {
    trace = on;
  }

  /**
   * Returns the alias for this pool.  This name is defined by the user in
   * the constructor
   */
  public String getAlias() {
    return alias;
  }

  /**
   * Returns the number of times a Connection has been checked out from the
   * pool
   */
  public int getNumRequests() {
    return numRequests;
  }

  /**
   * Returns the number of times a thread has had to block on wait() as a 
   * result of all PooledConnections being in use.  Useful diagnostic tool to
   * see if your pool needs more nodes (which could require a database 
   * license upgrade if you're running Oracle for instance)
   */
  public int getNumWaits() {
    return numWaits;
  }

  /**
   * Returns the number of times a Connection has been closed by the
   * reapIdleConnections() method due to being checked out for longer than
   * the checkoutSeconds interval.  
   * <p>
   * If this is greater than 0 it means that
   * you either have queries that take longer to execute than the 
   * checkoutSeconds interval allows, or it means that you are forgetting to
   * call conn.close() somewhere in your application.  Both conditions are
   * undesirable, but they have different solutions.  In the latter case either
   * tune your query to execute more quickly, or increase the checkoutSeconds
   * parameter to the pool.  In the former case you simply need to find the
   * code that calls DriverManager.getConnection() but not conn.close()
   */
  public int getNumCheckoutTimeouts() {
    return numCheckoutTimeout;
  }

  /**
   * Returns the maximum number of connections this pool can open
   */
  public int getMaxConn() {
    return maxConn;
  }

  /**
   * Returns the current number of Connections in the pool.
   */
  public int size() {
    return connVector.size();
  }

  /**
   * Check  all connections to make sure they haven't:
   *   1) gone idle for too long<br>
   *   2) been checked out by a thread for too long (cursor leak)
   */
  public synchronized void reapIdleConnections() {
    if(trace) System.err.println("ConnectionPool: " + alias + 
				 " reapIdleConnections() starting");
    long now = System.currentTimeMillis();
    long idleTimeout = now - (timeoutSeconds * 1000);
    long checkoutTimeout = now - (checkoutSeconds * 1000);
    
    for(Enumeration e = connVector.elements(); e.hasMoreElements();) {
      PooledConnection conn = (PooledConnection)e.nextElement();
      if(conn.isLocked() && (conn.getLastAccess() < checkoutTimeout)) {
	numCheckoutTimeout++;
	removeConnection(conn);
      }
      else if(conn.getLastAccess() < idleTimeout) {
	if(conn.getLock()) {
	  removeConnection(conn);
	  conn.releaseLock();
	}
      }
    }

    if(trace) System.err.println("ConnectionPool: " + alias + 
				 " reapIdleConnections() finished");
  }

  /**
   * Removes all connections from the pool and calls close() on them.  It
   * squelches any SQLExceptions that might results from close().  That's
   * probably not ideal.
   */
  public synchronized void removeAllConnections() {
    // FIXME: what do we do when a connection is in progress.  Maybe we
    // pass a boolean to this method that tells us whether to shutdown
    // immediately, or to close connections as they are returned..
    // for now we'll just close 'em
    if(trace) System.err.println("ConnectionPool: " + alias + 
				 " removeAllConnections() called");

    for(Enumeration e = connVector.elements(); e.hasMoreElements();) {
      PooledConnection conn = (PooledConnection)e.nextElement();
      removeConnection(conn);
    }
  }

  private synchronized void removeConnection(PooledConnection conn) {
      //
      // I've had problems with Oracle drivers that resulted in a deadlock
      // when trying to close a connection.
      // 
      // So we'll set a timeout of 10 seconds on this operation.  if the
      // close doesn't complete within this interval, we'll return without
      // removing the connection.  
      //
      // This will hopefully still allow us to trap cases where buggy 
      // application code fails to return the Connection back to the pool,
      // but won't stop a query that's in the middle of running on some
      // drivers (Oracle).  With some MySQL drivers, this will interrupt
      // a query in progress.  
      //
      try {
	  JavaAlarm alarm = new JavaAlarm(conn, 10000);

	  // remove the connection, since we completed successfully
	  connVector.removeElement(conn);
      }
      catch(TimeoutException e) {
	  if(trace) e.printStackTrace();
      }
  }


  /**
   * Returns a connection for the pool.  There are three ways this can happen:
   *   <ol>
   *     <li>If a connection in the pool is free, it is locked and returned
   *         immediately
   *     <li>If no connections are free, but the pool is not full, a new
   *         Connection is opened and returned
   *     <li>If no connections are free, and the pool is full, the thread
   *         calls wait() and blocks until a connection is returned
   *   </ol>
   */
  public synchronized Connection getConnection() throws SQLException {
    if(trace) System.err.println("ConnectionPool: " + alias + 
				 " getConnection() called");
    numRequests++;
    while(true) {
      // Check if connection is free in pool
      PooledConnection p;
      for(int i = 0; i < connVector.size(); i++) {
        p = (PooledConnection)connVector.elementAt(i);
        if(p.getLock()) return p;
      }

      if(trace) System.err.println("ConnectionPool: " + alias +
                                   " all connections locked.  calling" +
                                   " createConnection()");

      if(connVector.size() < maxConn) {
        if(trace) System.err.println("ConnectionPool: " + alias +
                                     " opening new connection to database");

        Connection conn = DriverManager.getConnection(url, username,
                                                      password);

        if(trace) System.err.println("ConnectionPool: " + alias +
                                     " finished opening new connection");

        p = new PooledConnection(conn, this);
        p.getLock();
        connVector.addElement(p);

        return p;
      }

      // Wait until a Connection is returned
      try {
        if(trace) System.err.println("ConnectionPool: " + alias +
                                     " pool is full.  calling wait()");

        numWaits++;
        wait();    
      }
      catch(InterruptedException e) { }

      if(trace) System.err.println("ConnectionPool: " + alias +
                                   " awoken from wait().  trying to grab" +
                                   " an available connection");
    }
  } 

  public synchronized void returnConnection(PooledConnection conn) {
    // remove connection from pool (close it) if we've used this connection
    // too many times
    if(maxCheckout > 0 && conn.getCheckoutCount() >= maxCheckout) {
      if(trace) System.err.println("ConnectionPool: " + alias + 
				   " connection checked out max # of times." +
				   " closing it.");

      removeConnection(conn);
    }

    // Wake up any waiting threads
    if(trace) System.err.println("ConnectionPool: " + alias + 
				 " releasing lock and calling notifyAll()");

    conn.releaseLock();
    notifyAll();
  }

}

class PooledConnection implements Connection, Runnable {

  private ConnectionPool pool;
  private Connection conn;
  private boolean locked;
  private long lastAccess;
  private int checkoutCount;

  private Hashtable prepStmts, callStmts;

  public PooledConnection(Connection conn, ConnectionPool pool) {
    this.conn = conn;
    this.pool = pool;
    this.locked = false;
    this.lastAccess = 0;
    this.checkoutCount = 0;

    prepStmts = new Hashtable();
    callStmts = new Hashtable();
  }

    /**
     * This is used by ConnectionPool.removeConnection().  It closes all
     * statements and closes the actual JDBC Connection.  We implement this
     * in a runnable method so that we can run this in a separate Thread and
     * use a timeout mechanism to avoid deadlock (which was occurring when
     * trying to close a Connection that had a query in progress)
     */
    public void run() {
      try {
	  closeStatements();
      }
      catch(SQLException e) { }
      
      try {
	  getConnection().close();
      }
      catch(SQLException e) { }
    }

  /**
   * Closes all PreparedStatments and CallableStatements
   */
  public void closeStatements() throws SQLException {
    for(Enumeration e = prepStmts.elements(); e.hasMoreElements();) {
      PooledPreparedStatement stmt = (PooledPreparedStatement)e.nextElement();
      stmt.getStatement().close();
    }

    for(Enumeration e = callStmts.elements(); e.hasMoreElements();) {
      PooledCallableStatement stmt = (PooledCallableStatement)e.nextElement();
      stmt.getStatement().close();
    }
  }

  /**
   * Returns true if this Connection is not in use.  Returns false if it
   * is in use
   */
  public synchronized boolean getLock() {
    if(locked) return false;
    else {
      locked = true;
      checkoutCount++;

      // Touch our access time so that we don't get immediately reaped
      // by accident, which could happen if the idle timeout is close to the
      // max duration of the query
      lastAccess = System.currentTimeMillis();
      return true;
    }
  }

  public boolean isLocked() {
    return locked;
  }

  /**
   * Returns number of times this node has been checked out from the pool
   */
  public int getCheckoutCount() {
    return checkoutCount;
  }

  /**
   * Returns when this connection was last checked out; 0 if it has never
   * been used.
   */
  public long getLastAccess() {
    return lastAccess;
  }

  /**
   * Doesn't actually close the connection, but instead returns itself
   * back to the pool to be re-used.  However, if you specified maxCheckouts
   * in the constructor, then this *will* close the JDBC Connection and re-open
   * it if the number of checkouts has been exceeded.
   */
  public void close() throws SQLException {
    lastAccess = System.currentTimeMillis();
    pool.returnConnection(this);
  }

  /**
   * called by ConnectionPool.returnConnection() right before it wakes up
   * the threads 
   */
  protected void releaseLock() {
    locked = false;
  }

  protected Connection getConnection() {
    return conn;
  }

  /*
   * Uncomment these methods if you want to play
   * around with cached PreparedStatements and CallableStatements.
   * 
   * I didn't have a chance to test this stuff much, and I needed to release
   * to fix the deadlock bug.. so I've left this stuff in, but commented out
   *
   * If you do use this stuff, you'll need to comment out the "proxy"
   * version of these methods below.
   *
  public PreparedStatement prepareStatement(String sql) throws SQLException {
    PooledPreparedStatement stmt = (PooledPreparedStatement)prepStmts.get(sql);
    if(stmt == null) {
      stmt = new PooledPreparedStatement(conn.prepareStatement(sql));
      prepStmts.put(sql, stmt);
    }
    return stmt;
  }

  public CallableStatement prepareCall(String sql) throws SQLException {
    PooledCallableStatement stmt = (PooledCallableStatement)callStmts.get(sql);
    if(stmt == null) {
      stmt = new PooledCallableStatement(conn.prepareCall(sql));
      callStmts.put(sql, stmt);
    }
    return stmt;
  }
  */

  /**************************************************************************
   *
   * Proxy all other JDBC calls to actual Connection object
   *
   **************************************************************************/

  public PreparedStatement prepareStatement(String sql) throws SQLException {
    return conn.prepareStatement(sql);
  }

  public CallableStatement prepareCall(String sql) throws SQLException {
    return conn.prepareCall(sql);
  }

  public Statement createStatement() throws SQLException {
    return conn.createStatement();
  }

  public String nativeSQL(String sql) throws SQLException {
    return conn.nativeSQL(sql);
  }

  public void setAutoCommit(boolean autoCommit) throws SQLException {
    conn.setAutoCommit(autoCommit);
  }

  public boolean getAutoCommit() throws SQLException {
    return conn.getAutoCommit();
  }

  public void commit() throws SQLException {
    conn.commit();
  }

  public void rollback() throws SQLException {
    conn.rollback();
  }

  public boolean isClosed() throws SQLException {
    return conn.isClosed();
  }

  public DatabaseMetaData getMetaData() throws SQLException {
    return conn.getMetaData();
  }

  public void setReadOnly(boolean readOnly) throws SQLException {
    conn.setReadOnly(readOnly);
  }

  public boolean isReadOnly() throws SQLException {
    return conn.isReadOnly();
  }

  public void setCatalog(String catalog) throws SQLException {
    conn.setCatalog(catalog);
  }

  public String getCatalog() throws SQLException {
    return conn.getCatalog();
  }

  public void setTransactionIsolation(int level) throws SQLException {
    conn.setTransactionIsolation(level);
  }

  public int getTransactionIsolation() throws SQLException {
    return conn.getTransactionIsolation();
  }

  public SQLWarning getWarnings() throws SQLException {
    return conn.getWarnings();
  }

  public void clearWarnings() throws SQLException {
    conn.clearWarnings();
  }
}

class PooledPreparedStatement implements PreparedStatement {

  private PreparedStatement stmt;

  public PooledPreparedStatement(PreparedStatement stmt) {
    this.stmt = stmt;
  }

  public void close() throws SQLException {  }

  public PreparedStatement getStatement() {
    return stmt;
  }

  /**************************************************************************
   *
   * Proxy all other JDBC calls to actual PreparedStatement object
   *
   **************************************************************************/

  public ResultSet executeQuery() throws SQLException {
    return stmt.executeQuery();
  }

  public int executeUpdate() throws SQLException {
    return stmt.executeUpdate();
  }

  public void setNull(int index, int sqlType) throws SQLException {
    stmt.setNull(index, sqlType);
  }

  public void setBoolean(int index, boolean x) throws SQLException {
    stmt.setBoolean(index, x);
  }

  public void setByte(int index, byte x) throws SQLException {
    stmt.setByte(index, x);
  }

  public void setShort(int index, short x) throws SQLException {
    stmt.setShort(index, x);
  }

  public void setInt(int index, int x) throws SQLException {
    stmt.setInt(index, x);
  }

  public void setLong(int index, long x) throws SQLException {
    stmt.setLong(index, x);
  }

  public void setFloat(int index, float x) throws SQLException {
    stmt.setFloat(index, x);
  }

  public void setDouble(int index, double x) throws SQLException {
    stmt.setDouble(index, x);
  }

  public void setBigDecimal(int index, BigDecimal x) throws SQLException {
    stmt.setBigDecimal(index, x);
  }

  public void setString(int index, String x) throws SQLException {
    stmt.setString(index, x);
  }

  public void setBytes(int index, byte x[]) throws SQLException {
    stmt.setBytes(index, x);
  }

  public void setDate(int index, Date x) throws SQLException {
    stmt.setDate(index, x);
  }

  public void setTime(int index, Time x) throws SQLException {
    stmt.setTime(index, x);
  }

  public void setTimestamp(int index, Timestamp x) throws SQLException {
    stmt.setTimestamp(index, x);
  }

  public void setAsciiStream(int index, InputStream x, int length) 
    throws SQLException {
    stmt.setAsciiStream(index, x, length);
  }

  public void setUnicodeStream(int index, InputStream x, int length) 
    throws SQLException {
    stmt.setUnicodeStream(index, x, length);
  }

  public void setBinaryStream(int index, InputStream x, int length)
    throws SQLException {
    stmt.setBinaryStream(index, x, length);
  }

  public void clearParameters() throws SQLException {
    stmt.clearParameters();
  }

  public void setObject(int index, Object x, int target, int scale) 
    throws SQLException {
    stmt.setObject(index, x, target, scale);
  }

  public void setObject(int index, Object x, int target) 
    throws SQLException {
    stmt.setObject(index, x, target);
  }

  public void setObject(int index, Object x) throws SQLException {
    stmt.setObject(index, x);
  }

  public boolean execute() throws SQLException {
    return stmt.execute();
  }

  //
  //  Methods in Statement
  //

  public ResultSet executeQuery(String sql) throws SQLException {
    return stmt.executeQuery(sql);
  }

  public int executeUpdate(String sql) throws SQLException {
    return stmt.executeUpdate(sql);
  }

  public int getMaxFieldSize() throws SQLException {
    return stmt.getMaxFieldSize();
  }

  public void setMaxFieldSize(int max) throws SQLException {
    stmt.setMaxFieldSize(max);
  }

  public int getMaxRows() throws SQLException {
    return stmt.getMaxRows();
  }

  public void setMaxRows(int max) throws SQLException {
    stmt.setMaxRows(max);
  }

  public void setEscapeProcessing(boolean enable) throws SQLException {
    stmt.setEscapeProcessing(enable);
  }

  public int getQueryTimeout() throws SQLException {
    return stmt.getQueryTimeout();
  }

  public void setQueryTimeout(int seconds) throws SQLException {
    stmt.setQueryTimeout(seconds);
  }

  public void cancel() throws SQLException {
    stmt.cancel();
  }

  public SQLWarning getWarnings() throws SQLException {
    return stmt.getWarnings();
  }

  public void clearWarnings() throws SQLException {
    stmt.clearWarnings();
  }

  public void setCursorName(String name) throws SQLException {
    stmt.setCursorName(name);
  }

  public boolean execute(String sql) throws SQLException {
    return stmt.execute(sql);
  }

  public ResultSet getResultSet() throws SQLException {
    return stmt.getResultSet();
  }

  public int getUpdateCount() throws SQLException {
    return stmt.getUpdateCount();
  }

  public boolean getMoreResults() throws SQLException {
    return stmt.getMoreResults();
  }
}

class PooledCallableStatement implements CallableStatement {

  private CallableStatement stmt;

  public PooledCallableStatement(CallableStatement stmt) {
    this.stmt = stmt;
  }

  public void close() throws SQLException {  }

  public CallableStatement getStatement() {
    return stmt;
  }

  /**************************************************************************
   *
   * CallableStatment methods
   *
   **************************************************************************/

  public void registerOutParameter(int index, int type) throws SQLException {
    stmt.registerOutParameter(index, type);
  }

  public void registerOutParameter(int index, int type, int scale) throws SQLException {
    stmt.registerOutParameter(index, type, scale);
  }

  public boolean wasNull() throws SQLException {
    return stmt.wasNull();
  }

  public BigDecimal getBigDecimal(int index, int scale) throws SQLException {
    return stmt.getBigDecimal(index, scale);
  }

  public boolean getBoolean(int index) throws SQLException {
    return stmt.getBoolean(index);
  }

  public byte getByte(int index) throws SQLException {
    return stmt.getByte(index);
  }

  public byte[] getBytes(int index) throws SQLException {
    return stmt.getBytes(index);
  }

  public Date getDate(int index) throws SQLException {
    return stmt.getDate(index);
  }

  public double getDouble(int index) throws SQLException {
    return stmt.getDouble(index);
  }

  public float getFloat(int index) throws SQLException {
    return stmt.getFloat(index);
  }

  public int getInt(int index) throws SQLException {
    return stmt.getInt(index);
  }

  public long getLong(int index) throws SQLException {
    return stmt.getLong(index);
  }

  public Object getObject(int index) throws SQLException {
    return stmt.getObject(index);
  }

  public short getShort(int index) throws SQLException {
    return stmt.getShort(index);
  }

  public String getString(int index) throws SQLException {
    return stmt.getString(index);
  }

  public Time getTime(int index) throws SQLException {
    return stmt.getTime(index);
  }

  public Timestamp getTimestamp(int index) throws SQLException {
    return stmt.getTimestamp(index);
  }

  /**************************************************************************
   *
   * PreparedStatment methods
   *
   **************************************************************************/

  public ResultSet executeQuery() throws SQLException {
    return stmt.executeQuery();
  }

  public int executeUpdate() throws SQLException {
    return stmt.executeUpdate();
  }

  public void setNull(int index, int sqlType) throws SQLException {
    stmt.setNull(index, sqlType);
  }

  public void setBoolean(int index, boolean x) throws SQLException {
    stmt.setBoolean(index, x);
  }

  public void setByte(int index, byte x) throws SQLException {
    stmt.setByte(index, x);
  }

  public void setShort(int index, short x) throws SQLException {
    stmt.setShort(index, x);
  }

  public void setInt(int index, int x) throws SQLException {
    stmt.setInt(index, x);
  }

  public void setLong(int index, long x) throws SQLException {
    stmt.setLong(index, x);
  }

  public void setFloat(int index, float x) throws SQLException {
    stmt.setFloat(index, x);
  }

  public void setDouble(int index, double x) throws SQLException {
    stmt.setDouble(index, x);
  }

  public void setBigDecimal(int index, BigDecimal x) throws SQLException {
    stmt.setBigDecimal(index, x);
  }

  public void setString(int index, String x) throws SQLException {
    stmt.setString(index, x);
  }

  public void setBytes(int index, byte x[]) throws SQLException {
    stmt.setBytes(index, x);
  }

  public void setDate(int index, Date x) throws SQLException {
    stmt.setDate(index, x);
  }

  public void setTime(int index, Time x) throws SQLException {
    stmt.setTime(index, x);
  }

  public void setTimestamp(int index, Timestamp x) throws SQLException {
    stmt.setTimestamp(index, x);
  }

  public void setAsciiStream(int index, InputStream x, int length) 
    throws SQLException {
    stmt.setAsciiStream(index, x, length);
  }

  public void setUnicodeStream(int index, InputStream x, int length) 
    throws SQLException {
    stmt.setUnicodeStream(index, x, length);
  }

  public void setBinaryStream(int index, InputStream x, int length)
    throws SQLException {
    stmt.setBinaryStream(index, x, length);
  }

  public void clearParameters() throws SQLException {
    stmt.clearParameters();
  }

  public void setObject(int index, Object x, int target, int scale) 
    throws SQLException {
    stmt.setObject(index, x, target, scale);
  }

  public void setObject(int index, Object x, int target) 
    throws SQLException {
    stmt.setObject(index, x, target);
  }

  public void setObject(int index, Object x) throws SQLException {
    stmt.setObject(index, x);
  }

  public boolean execute() throws SQLException {
    return stmt.execute();
  }

  //
  //  Methods in Statement
  //

  public ResultSet executeQuery(String sql) throws SQLException {
    return stmt.executeQuery(sql);
  }

  public int executeUpdate(String sql) throws SQLException {
    return stmt.executeUpdate(sql);
  }

  public int getMaxFieldSize() throws SQLException {
    return stmt.getMaxFieldSize();
  }

  public void setMaxFieldSize(int max) throws SQLException {
    stmt.setMaxFieldSize(max);
  }

  public int getMaxRows() throws SQLException {
    return stmt.getMaxRows();
  }

  public void setMaxRows(int max) throws SQLException {
    stmt.setMaxRows(max);
  }

  public void setEscapeProcessing(boolean enable) throws SQLException {
    stmt.setEscapeProcessing(enable);
  }

  public int getQueryTimeout() throws SQLException {
    return stmt.getQueryTimeout();
  }

  public void setQueryTimeout(int seconds) throws SQLException {
    stmt.setQueryTimeout(seconds);
  }

  public void cancel() throws SQLException {
    stmt.cancel();
  }

  public SQLWarning getWarnings() throws SQLException {
    return stmt.getWarnings();
  }

  public void clearWarnings() throws SQLException {
    stmt.clearWarnings();
  }

  public void setCursorName(String name) throws SQLException {
    stmt.setCursorName(name);
  }

  public boolean execute(String sql) throws SQLException {
    return stmt.execute(sql);
  }

  public ResultSet getResultSet() throws SQLException {
    return stmt.getResultSet();
  }

  public int getUpdateCount() throws SQLException {
    return stmt.getUpdateCount();
  }

  public boolean getMoreResults() throws SQLException {
    return stmt.getMoreResults();
  }
}
