/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_record module

  Functions herein allow to save and read data from files.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_record

  Les fonctions presentes dans ce module permettent de lire et
  d'ecrire des donnees dans des fichiers.
*/

/*-------------------------------------------------------------*/
/*en definition of a lcrzo_record. You should not use its
   fields directly. */
/*fr definition d'un lcrzo_record. Vous ne devriez pas utiliser
   directement ses champs. */
typedef struct
{ LCRZOFILE *pf;
  lcrzo_int16 opentype;
  lcrzo_int32 justbeforeposition;
  lcrzo_uint32 row;
  lcrzo_uint32 column;
} lcrzo_record;

/*-------------------------------------------------------------*/
/*en a record can be open in read/write/append mode */
/*fr un record peut etre ouvert en lecture/ecriture/append */
#define LCRZO_RECORD_OPEN_READ   1
#define LCRZO_RECORD_OPEN_WRITE  2
#define LCRZO_RECORD_OPEN_APPEND 3

/*-------------------------------------------------------------*/
/*en Name : lcrzo_record_open
   Description :
     Open a lcrzo_record.
   Input parameter(s) :
     filepath : path for the file for this record
                For example "/home/user/rec1.dat".
		Symlinks are forbidden
     openingtype : opening type (LCRZO_RECORD_OPEN_READ, 
                   WRITE, APPEND)
   Input/output parameter(s) :
   Output parameter(s) :
     *precord : lcrzo_record opened
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_open
   Description :
     Ouvre un lcrzo_record
   Parametre(s) d'entree :
     filepath : chemin d'acces au fichier a associer a ce record
                Par exemple "/home/utilisateur/rec1.dat".
     openingtype : type d'ouverture (LCRZO_RECORD_OPEN_READ, 
                    WRITE, APPEND)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *precord : lcrzo_record ouvert
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_open(lcrzo_record *precord,
		      lcrzo_conststring filepath,
		      lcrzo_int16 openingtype);

/*en Name : lcrzo_record_close
   Description :
     Close a lcrzo_record.
   Input parameter(s) :
   Input/output parameter(s) :
     *precord : lcrzo_record to close
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_close
   Description :
     Ferme un lcrzo_record
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a fermer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_close(lcrzo_record *precord);

/*-------------------------------------------------------------*/
/*EN*********************************************************
 * Note about a record :                                    *
 * A record is a file where we can save and read data.      *
 *                                                          *
 * Actually, we can only read data if it was saved in       *
 * hexadecimal or mixed form (so be careful when choosing   *
 * a lcrzo_printtype or lcrzo_printprofile for saving).     *
 ************************************************************/
/*FR*********************************************************
 * Note sur les records :                                   *
 * Un record est un fichier dans lequel on va sauvegarder   *
 * et lire des donnees.                                     *
 *                                                          *
 * Actuellement, on ne peut relire des donnees que si elles *
 * ont ete sauvees dans un format hexadecimal ou mixte      *
 * (donc, choisissez attentivement le lcrzo_printtype ou le *
 * lcrzo_printprofile pour la sauvegarde).                  *
 ************************************************************/

/*en Name : lcrzo_printtype_init_stdin_record
   Description :
     Ask user to choose a lcrzo_printtype.
     The choices contains only printtypes which can be read
     be a record.
   Input parameter(s) :
     message : message to print before
     defaultprinttype : default lcrzo_printtype to use if
                        user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     *pprinttype : lcrzo_printype chosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printtype_init_stdin_record
   Description :
     Demande a l'utilisateur de choisir un lcrzo_printtype.
     Les choix proposes ne contiennent que des types pouvant 
     etre relus dans un record.
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultprinttype : lcrzo_printtype par defaut a utiliser
                        si l'utilisateur n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprinttype : lcrzo_printype choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printtype_init_stdin_record(lcrzo_conststring message,
				 lcrzo_printtype defaultprinttype,
				 lcrzo_printtype *pprinttype);

/*en Name : lcrzo_printprofile_init_stdin_record
   Description :
     Ask user to choose a lcrzo_printprofile
     The choices contains only printprofiles which can be read
     be a record.
   Input parameter(s) :
     message : message to print before
     defaultprintprofile : default lcrzo_printprofile to use if
                        user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     *pprintprofile : lcrzo_printype chosen
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_printprofile_init_stdin_record
   Description :
     Demande a l'utilisateur de choisir un lcrzo_printprofile
     Les choix proposes ne contiennent que des profils pouvant 
     etre relus dans un record.
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultprintprofile : lcrzo_printprofile par defaut a utiliser
                        si l'utilisateur n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pprintprofile : lcrzo_printype choisi
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_printprofile_init_stdin_record(lcrzo_conststring message,
				    lcrzo_printprofile defaultprintprofile,
				    lcrzo_printprofile *pprintprofile);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_record_write_data
   Description :
     Save data in a record.
   Input parameter(s) :
     data : data to save
     datasize : size of data
     printtype : lcrzo_printtype to use
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_write_data
   Description :
     Sauvegarde des donnees dans un record.
   Parametre(s) d'entree :
     data : donnees a sauver
     datasize : taille de data
     printtype : lcrzo_printtype a employer
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_write_data(lcrzo_record *precord, 
			    lcrzo_constdata data,
			    lcrzo_int32 datasize,
			    lcrzo_printtype printtype);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_record_write_eth
   Description :
     Save an Ethernet packet in a record.
   Input parameter(s) :
     data : data to print
     datasize : size of data
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_write_eth
   Description :
     Sauvegarde un paquet Ethernet dans un record.
   Parametre(s) d'entree :
     data : donnees a ecrire
     datasize : taille de data
     printprofile : lcrzo_printprofile a employer
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_write_eth(lcrzo_record *precord, 
			   lcrzo_constdata data,
			   lcrzo_int32 datasize,
			   lcrzo_printprofile printprofile);

/*en Name : lcrzo_record_write_ip
   Description :
     Save an IP packet in a record.
   Input parameter(s) :
     data : data to print
     datasize : size of data
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_write_ip
   Description :
     Sauvegarde un paquet IP dans un record.
   Parametre(s) d'entree :
     data : donnees a ecrire
     datasize : taille de data
     printprofile : lcrzo_printprofile a employer
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_write_ip(lcrzo_record *precord, 
			  lcrzo_constdata data,
			  lcrzo_int32 datasize,
			  lcrzo_printprofile printprofile);

/*en Name : lcrzo_record_write_udp
   Description :
     Save an UDP packet in a record.
   Input parameter(s) :
     data : data to print
     datasize : size of data
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_write_udp
   Description :
     Sauvegarde un paquet UDP dans un record.
   Parametre(s) d'entree :
     data : donnees a ecrire
     datasize : taille de data
     printprofile : lcrzo_printprofile a employer
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_write_udp(lcrzo_record *precord, 
			   lcrzo_constdata data,
			   lcrzo_int32 datasize,
			   lcrzo_printprofile printprofile);

/*en Name : lcrzo_record_write_tcp
   Description :
     Save an TCP packet in a record.
   Input parameter(s) :
     data : data to print
     datasize : size of data
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_write_tcp
   Description :
     Sauvegarde un paquet TCP dans un record.
   Parametre(s) d'entree :
     data : donnees a ecrire
     datasize : taille de data
     printprofile : lcrzo_printprofile a employer
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_write_tcp(lcrzo_record *precord, 
			   lcrzo_constdata data,
			   lcrzo_int32 datasize,
			   lcrzo_printprofile printprofile);

/*en Name : lcrzo_record_write_icmp
   Description :
     Save an ICMP packet in a record.
   Input parameter(s) :
     data : data to print
     datasize : size of data
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_write_icmp
   Description :
     Sauvegarde un paquet ICMP dans un record.
   Parametre(s) d'entree :
     data : donnees a ecrire
     datasize : taille de data
     printprofile : lcrzo_printprofile a employer
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_write_icmp(lcrzo_record *precord, 
			    lcrzo_constdata data,
			    lcrzo_int32 datasize,
			    lcrzo_printprofile printprofile);

/*en Name : lcrzo_record_write_arp
   Description :
     Save an ARP packet in a record.
   Input parameter(s) :
     data : data to print
     datasize : size of data
     printprofile : lcrzo_printprofile to use
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_write_arp
   Description :
     Sauvegarde un paquet ARP dans un record.
   Parametre(s) d'entree :
     data : donnees a ecrire
     datasize : taille de data
     printprofile : lcrzo_printprofile a employer
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_write_arp(lcrzo_record *precord, 
			   lcrzo_constdata data,
			   lcrzo_int32 datasize,
			   lcrzo_printprofile printprofile);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_record_count
   Description :
     Count the number of items in a record
   Input parameter(s) :
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
     *pcount : number of items
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_count
   Description :
     Compte le nombre d'elements dans un record
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
     *pcount : nombre d'elements
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_count(lcrzo_record *precord, 
		       lcrzo_int32 *pcount);

/*EN*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/

/*en Name : lcrzo_record_valuem_pos
   Description :
     Get an item in a record
   Input parameter(s) :
     pos : position of the wanted item (cf. the above frame to
           know possible values)
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the 
		 item number 'pos'
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_valuem_pos
   Description :
     Obtient un element d'un record
   Parametre(s) d'entree :
     pos : position de l'element cherche (cf. le cadre
           ci-dessus pour connaitre les valeurs valides)
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra l'element
		 numero 'pos'
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_valuem_pos(lcrzo_record *precord, 
			    lcrzo_int32 pos,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*en Name : lcrzo_record_valuem_first
   Description :
     Get the first item of a record
   Input parameter(s) :
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the 
		 first item
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_valuem_first
   Description :
     Obtient le premier element d'un record
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra le premier element
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_valuem_first(lcrzo_record *precord, 
			      lcrzo_data *pdataout,
			      lcrzo_int32 *pdataoutsize);

/*en Name : lcrzo_record_valuem_last
   Description :
     Get the last item of a record
   Input parameter(s) :
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the 
		 last item
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_valuem_last
   Description :
     Obtient le dernier element d'un record
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra le dernier element
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_valuem_last(lcrzo_record *precord, 
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize);

/*en Name : lcrzo_record_loop_range
   Description :
     Call a function for each item in a
     given position range.
   Input parameter(s) :
     startpos : start position for the loop (cf. the above frame to
                know possible values)
     endpos : end position for the loop (cf. the above frame to
              know possible values)
     pfunc : memory address of the function which will be called
             for each item in the range [startpos->endpos].
             For each call, the first/second parameter 
	     ('data','datasize') will be set with the item, and
	     the third parameter
             ('pinfos') will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_loop_range
   Description :
     Appelle une fonction pour les elements
     situes dans un intervalle donne.
   Parametre(s) d'entree :
     startpos : position de debut de la boucle (cf. le cadre
                ci-dessus pour connaitre les valeurs valides)
     endpos : position de fin de la boucle (cf. le cadre
              ci-dessus pour connaitre les valeurs valides)
     pfunc : adresse memoire de la fonction appelee pour chaque
             element de l'intervalle [startpos->endpos]. Lors de
             chaque appel, le premier/deuxieme parametre ('data',
	     'datasize') est affecte avec l'element courant, et 
	     le troisieme parametre ('pinfos') est affecte
	     avec la valeur de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_loop_range(lcrzo_record *precord, 
			    lcrzo_int32 startpos,
			    lcrzo_int32 endpos,
			    int (*pfunc)(void *data,
					 lcrzo_int32 datasize,
					 void *pinfos),
			    void *pinfos);

/*en Name : lcrzo_record_loop_all
   Description :
     Call a function for all items.
   Input parameter(s) :
     pfunc : memory address of the function which will be called
             for each item.
             For each call, the first/second parameter 
	     ('data','datasize') will be set with the item, and
	     the third parameter
             ('pinfos') will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
     *precord : lcrzo_record to use
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_loop_all
   Description :
     Appelle une fonction pour tous les elements.
   Parametre(s) d'entree :
     pfunc : adresse memoire de la fonction appelee pour chaque
             element. Lors de
             chaque appel, le premier/deuxieme parametre ('data',
	     'datasize') est affecte avec l'element courant, et 
	     le troisieme parametre ('pinfos') est affecte
	     avec la valeur de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
     *precord : lcrzo_record a employer
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_loop_all(lcrzo_record *precord, 
			  int (*pfunc)(void *data,
				       lcrzo_int32 datasize,
				       void *pinfos),
			  void *pinfos);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_record_recfile_init_pcapfile
   Description :
     Convert a pcapfile (for example generated by tcpdump)
     to a lcrzo_record file.
   Input parameter(s) :
     pcapfile : pcap input file
     recfile : generated record file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_recfile_init_pcapfile
   Description :
     Convertit un fichier pcap (genere par exemple par tcpdump)
     en un fichier lcrzo_record.
   Parametre(s) d'entree :
     pcapfile : fichier pcap d'entree
     recfile : fichier record genere
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_recfile_init_pcapfile(lcrzo_conststring pcapfile,
				       lcrzo_conststring recfile);

/*en Name : lcrzo_record_pcapfile_init_recfile
   Description :
     Convert a lcrzo_record file to a pcapfile.
   Input parameter(s) :
     recfile : input record file
     pcapfile : pcap output file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_record_pcapfile_init_recfile
   Description :
     Convertit un fichier lcrzo_record en un fichier pcap.
   Parametre(s) d'entree :
     recfile : fichier record d'entree
     pcapfile : fichier pcap genere
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_record_pcapfile_init_recfile(lcrzo_conststring recfile,
				       lcrzo_conststring pcapfile);

