/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_sock module

  Functions herein allow to create clients and servers.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_sock

  Les fonctions presentes dans ce module permettent de creer des
  clients ou des serveurs reels, ainsi que virtuels.
*/

/*-------------------------------------------------------------*/
/*en definition of a lcrzo_sock. You should not use its
   fields directly. */
/*fr definition d'un lcrzo_sock. Vous ne devriez pas utiliser
   directement ses champs. */
typedef struct
{ lcrzo_int16 type;
  int socknum;
  lcrzo_process sniffprocess;
  lcrzo_process spoofprocess;
  lcrzo_bool sniffprocesshasended;
  lcrzo_bool spoofprocesshasended;
  int sniffprocessreturnedvalue;
  int spoofprocessreturnedvalue;
  lcrzo_ipc ipcsniff;
  lcrzo_ipc ipcspoof;
  lcrzo_ipc ipcsniffend;
  lcrzo_ipc ipcspoofend;
  lcrzo_ipc ipcoptions;
  lcrzo_ipc ipccontrol1;
  lcrzo_ipc ipccontrol2;
  lcrzo_ipopt ipopt;
  lcrzo_uint8 ipoptsize;
  lcrzo_device here_device;
  lcrzo_etha here_etha;
  lcrzo_ipl here_ipl;
  lcrzo_uint16 here_port;
  lcrzo_etha there_etha;
  lcrzo_ipl there_ipl;
  lcrzo_uint16 there_port;
  lcrzo_data storeddata;
  lcrzo_int32 storeddatasize;
  lcrzo_bool dataend;
  lcrzo_bool udpcanwrite;
  lcrzo_int16 tcpstate;
} lcrzo_sock;

/*-------------------------------------------------------------*/
/*EN************************************************************
 * Note about "real" and "virtual" :                           *
 *                                                             *
 * A real client/server uses IP address and Ethernet address   *
 * of the current computer (they are classical sockets).       *
 *                                                             *
 * A virtual client/server uses spoofed IP address and         *
 * Ethernet address. For example, a virtual tcp client behaves *
 * like this :                                                 *
 *  - spoof a syn packet                                       *
 *  - sniff the syn-ack from the server                        *
 *  - spoof a ack packet to terminate the handshake            *
 *  - then, it behaves exactly like a real client : we can use *
 *    the lcrzo_sock to read or write data on the network      *
 * This virtual IP stack doesn't exactly implement RFC 768 and *
 * RFC 793 : it's working, but all specials cases are not      *
 * implemented.                                                *
 ***************************************************************/
/*FR************************************************************
 * Note sur "reel" et "virtuel" :                              *
 *                                                             *
 * Les clients/serveurs reels emploient les adresses ip et     *
 * ethernet de la machine courante (ce sont les sockets        *
 * classiques telles que tout le monde les connait).           *
 *                                                             *
 * Les clients/serveurs virtuels utilisent des adresses        *
 * ip/ethernet spoofees. En quelques mots, prenons l'exemple   *
 * d'un client virtuel tcp :                                   *
 *  - on spoofe un packet syn                                  *
 *  - on sniffe le reseau a la recherche du syn-ack            *
 *  - on spoofe un packet ack pour terminer le handshake       *
 *  - ensuite, on peut lire et ecrire des donnees, comme on le *
 *    ferait avec un client tcp reel (en utilisant le          *
 *    lcrzo_sock)                                              *
 * Cette pile IP virtuelle n'implemente pas de maniere         *
 * rigoureuse les RFC 768 et RFC 793 : ca marche, mais tous    *
 * les cas particuliers ne sont pas traites.                   * 
 ***************************************************************/


/*-------------------------------------------------------------*/
/*EN                        UDP clients                          */
/*FR                        clients UDP                          */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_sock_udpcli_real
   Description :
     Initialize a real udp client
   Input parameter(s) :
     serveripl : IP address of the server where to connect
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_real
   Description :
     Initialise un client udp reel
   Parametre(s) d'entree :
     serveripl : adresse IP du serveur ou se connecter
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_real(lcrzo_ipl    serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock   *psock);

/*en Name : lcrzo_sock_udpcli_virt
   Description :
     Initialize a virtual udp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_virt
   Description :
     Initialise un client udp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresse Ethernet du serveur (ou routeur)
     clientipl : adresse IP a spoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_virt(lcrzo_constdevice device,
			   lcrzo_constetha clientetha,
			   lcrzo_constetha serveretha,
			   lcrzo_ipl clientipl,
			   lcrzo_ipl serveripl, 
			   lcrzo_uint16 clientport,
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*en Name : lcrzo_sock_udpcli_real2
   Description :
     Initialize a real udp client
   Input parameter(s) :
     clientipl : local IP address to select (must be a configured
                 IP address of the current computer).
		 If 0, use the default value
     serveripl : IP address of the server where to connect
     clientport : local source port number to use.
                  If 0, the computer choose for us (a value
		  greater than 1024).
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_real2
   Description :
     Initialise un client udp reel
   Parametre(s) d'entree :
     clientipl : adresse IP locale a employer (cette adresse IP
                 doit etre configuree sur la machine).
		 Si 0, la fonction utilise l'adresse par defaut
     serveripl : adresse IP du serveur ou se connecter
     client port : port source a utiliser
                   si 0, la machine choisit un numero disponible
		   et superieur a 1024
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_real2(lcrzo_ipl    clientipl,
			    lcrzo_ipl    serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock   *psock);

/*en Name : lcrzo_sock_udpcli_virt2
   Description :
     Initialize a virtual udp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_virt2
   Description :
     Initialise un client udp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresse Ethernet du serveur (ou routeur)
     clientipl : adresse IP a spoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_virt2(lcrzo_constdevice device,
			    lcrzo_constetha clientetha,
			    lcrzo_constetha serveretha,
			    lcrzo_ipl clientipl,
			    lcrzo_ipl serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*-------------------------------------------------------------*/
/*EN                        TCP clients                          */
/*FR                        clients TCP                          */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_sock_tcpcli_real
   Description :
     Initialize a real tcp client
   Input parameter(s) :
     serveripl : IP address of the server where to connect
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_real
   Description :
     Initialise un client tcp reel
   Parametre(s) d'entree :
     serveripl : adresse IP du serveur ou se connecter
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_real(lcrzo_ipl    serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock   *psock);

/*en Name : lcrzo_sock_tcpcli_virt
   Description :
     Initialize a virtual tcp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_virt
   Description :
     Initialise un client tcp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresse Ethernet du serveur (ou routeur)
     clientipl : adresse IP a spoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_virt(lcrzo_constdevice device,
			   lcrzo_constetha clientetha,
			   lcrzo_constetha serveretha,
			   lcrzo_ipl clientipl,
			   lcrzo_ipl serveripl, 
			   lcrzo_uint16 clientport,
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*en Name : lcrzo_sock_tcpcli_real2
   Description :
     Initialize a real tcp client
   Input parameter(s) :
     clientipl : local IP address to select (must be a configured
                 IP address of the current computer).
		 If 0, use the default value
     serveripl : IP address of the server where to connect
     clientport : local source port number to use.
                  If 0, the computer choose for us (a value
		  greater than 1024).
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_real2
   Description :
     Initialise un client tcp reel
   Parametre(s) d'entree :
     clientipl : adresse IP locale a employer (cette adresse IP
                 doit etre configuree sur la machine).
		 Si 0, la fonction utilise l'adresse par defaut
     serveripl : adresse IP du serveur ou se connecter
     clientport : port source a utiliser
                  si 0, la machine choisit un numero disponible
		  et superieur a 1024
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_real2(lcrzo_ipl    clientipl,
			    lcrzo_ipl    serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock   *psock);

/*en Name : lcrzo_sock_tcpcli_virt2
   Description :
     Initialize a virtual tcp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_virt2
   Description :
     Initialise un client tcp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresse Ethernet du serveur (ou routeur)
     clientipl : adresse IP a spoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_virt2(lcrzo_constdevice device,
			    lcrzo_constetha clientetha,
			    lcrzo_constetha serveretha,
			    lcrzo_ipl clientipl,
			    lcrzo_ipl serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*-------------------------------------------------------------*/
/*EN                        UDP servers                          */
/*FR                        serveurs UDP                         */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_sock_udpser_real
   Description :
     Initialize a real udp server
   Input parameter(s) :
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_real
   Description :
     Initialise un serveur udp reel
   Parametre(s) d'entree :
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_real(lcrzo_uint16 port,
			   lcrzo_sock *psock);

/*en Name : lcrzo_sock_udpser_virt
   Description :
     Initialize a virtual udp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP address on which we listen
     serverport : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_virt
   Description :
     Initialise un serveur udp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_virt(lcrzo_constdevice device,
			   lcrzo_constetha serveretha,
			   lcrzo_ipl serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*en Name : lcrzo_sock_udpser_real2
   Description :
     Initialize a real udp server
   Input parameter(s) :
     ipl : listening IP address 
     port : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_real2
   Description :
     Initialise un serveur udp reel
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_real2(lcrzo_ipl ipl,
			    lcrzo_uint16 port,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);

/*en Name : lcrzo_sock_udpser_virt2
   Description :
     Initialize a virtual udp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP address on which we listen
     serverport : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_virt2
   Description :
     Initialise un serveur udp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_virt2(lcrzo_constdevice device,
			    lcrzo_constetha serveretha,
			    lcrzo_ipl serveripl, 
			    lcrzo_uint16 serverport,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*-------------------------------------------------------------*/
/*EN                  UDP multiclients servers                   */
/*FR                  serveurs UDP multiclients                  */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_sock_udpmulser_real
   Description :
     Initialize a real multiclient udp server
   Input parameter(s) :
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpmulser_real
   Description :
     Initialise un serveur udp reel multiclient
   Parametre(s) d'entree :
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpmulser_real(lcrzo_uint16 port,
			      lcrzo_sock *psock);

/*en Name : lcrzo_sock_udpmulser_virt
   Description :
     Initialize a virtual udp multiclient server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP address on which we listen
     serverport : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpmulser_virt
   Description :
     Initialise un serveur udp virtuel multiclients
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpmulser_virt(lcrzo_constdevice device,
			      lcrzo_constetha serveretha,
			      lcrzo_ipl serveripl, 
			      lcrzo_uint16 serverport,
			      lcrzo_sock *psock);

/*en Name : lcrzo_sock_udpmulser_real2
   Description :
     Initialize a real multiclient udp server
   Input parameter(s) :
     ipl : listening IP address
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpmulser_real2
   Description :
     Initialise un serveur udp reel multiclient
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpmulser_real2(lcrzo_ipl ipl,
			       lcrzo_uint16 port,
			       lcrzo_sock *psock);


/*-------------------------------------------------------------*/
/*EN                        TCP servers                          */
/*FR                        serveurs TCP                         */
/*-------------------------------------------------------------*/

/*EN************************************************************
 * Note about TCP servers :                                    *
 * We return from the function only after a client has         *
 * connected.                                                  *
 ***************************************************************/
/*FR************************************************************
 * Note sur les serveurs TCP :                                 *
 * On ne quitte la fonction que lorsqu'un client s'est         *
 * connecte.                                                   *
 ***************************************************************/

/*en Name : lcrzo_sock_tcpser_real
   Description :
     Initialize a real tcp server
   Input parameter(s) :
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_real
   Description :
     Initialise un serveur tcp reel
   Parametre(s) d'entree :
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_real(lcrzo_uint16 port,
			   lcrzo_sock *psock);

/*en Name : lcrzo_sock_tcpser_virt
   Description :
     Initialize a virtual tcp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP address on which we listen
     serverport : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_virt
   Description :
     Initialise un serveur tcp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_virt(lcrzo_constdevice device,
			   lcrzo_constetha serveretha,
			   lcrzo_ipl serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*en Name : lcrzo_sock_tcpser_real2
   Description :
     Initialize a real tcp server
   Input parameter(s) :
     ipl : listening IP address
     port : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_real2
   Description :
     Initialise un serveur tcp reel
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_real2(lcrzo_ipl ipl,
			    lcrzo_uint16 port,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);

/*en Name : lcrzo_sock_tcpser_virt2
   Description :
     Initialize a virtual tcp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP address on which we listen
     serverport : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_virt2
   Description :
     Initialise un serveur tcp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_virt2(lcrzo_constdevice device,
			    lcrzo_constetha serveretha,
			    lcrzo_ipl serveripl, 
			    lcrzo_uint16 serverport,
			    lcrzo_constipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*-------------------------------------------------------------*/
/*EN                  TCP multiclients servers                   */
/*FR                  serveusr TCP multiclients                  */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_sock_tcpmulser_real
   Description :
     Deal with a real tcp multiclient server
   Input parameter(s) :
     port : listening port
     pfunc : memory address of the function which will be called
             for each client. For
             each call, the first parameter ('sock') will be set
             with socket, and the second parameter ('pinfos')
             will be set with the optional parameter below.
   Input/output parameter(s) :
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpmulser_real
   Description :
     Gere un serveur tcp reel multiclient
   Parametre(s) d'entree :
     port : port sur lequel ecouter
     pfunc : adresse memoire de la fonction appelee pour chaque
             client. Lors de
             chaque appel, le premier parametre ('sock') est 
             affecte avec le socket courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
   Parametre(s) d'entree-sortie :
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpmulser_real(lcrzo_uint16 port,
			      int (*pfunc)(lcrzo_sock *psock,
					   void *pinfos),
			      void *pinfos);

/*en Name : lcrzo_sock_tcpmulser_real2
   Description :
     Deal with a real tcp multiclient server
   Input parameter(s) :
     ipl : listening IP address
     port : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
     pfunc : memory address of the function which will be called
             for each client. For
             each call, the first parameter ('sock') will be set
             with socket, and the second parameter ('pinfos')
             will be set with the optional parameter below.
   Input/output parameter(s) :
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpmulser_real2
   Description :
     Gere un serveur tcp reel multiclient
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
     pfunc : adresse memoire de la fonction appelee pour chaque
             client. Lors de
             chaque appel, le premier parametre ('sock') est 
             affecte avec le socket courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
   Parametre(s) d'entree-sortie :
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpmulser_real2(lcrzo_ipl ipl,
			       lcrzo_uint16 port,
			       lcrzo_constipopt ipopt,
			       lcrzo_uint8 ipoptsize,
			       int (*pfunc)(lcrzo_sock *psock,
					    void *pinfos),
			       void *pinfos);


/*-------------------------------------------------------------*/
/*EN                        IP options                           */
/*FR                        options IP                           */
/*-------------------------------------------------------------*/

/*en the following functions add IP options to a connection.
   Read help of the lcrzo_ipopt module for an explanation of
   each parameter */
/*fr les fonctions suivantes ajoutent des options IP a une
   connexion. Consultez l'aide du module lcrzo_ipopt pour 
   connaitre la signification des parametres*/

/* set/get raw options */
int lcrzo_sock_ipopt_set(lcrzo_sock *psock,
			 lcrzo_constipopt ipopt,
			 lcrzo_uint8 ipoptsize);
int lcrzo_sock_ipopt_get(lcrzo_sock *psock,
			 lcrzo_ipopt ipopt,
			 lcrzo_uint8 *pipoptsize);

/* no operation */
int lcrzo_sock_ipopt_noop(lcrzo_sock *psock);

/* loose source routing */
int lcrzo_sock_ipopt_lsrr(lcrzo_sock *psock,
			  lcrzo_ipl ipl1,
			  lcrzo_ipl ipl2,
			  lcrzo_ipl ipl3,
			  lcrzo_ipl ipl4,
			  lcrzo_ipl ipl5,
			  lcrzo_ipl ipl6,
			  lcrzo_ipl ipl7,
			  lcrzo_ipl ipl8,
			  lcrzo_ipl ipl9,
			  lcrzo_int8 numberofsetipl);

/* strict source routing */
int lcrzo_sock_ipopt_ssrr(lcrzo_sock *psock,
			  lcrzo_ipl ipl1,
			  lcrzo_ipl ipl2,
			  lcrzo_ipl ipl3,
			  lcrzo_ipl ipl4,
			  lcrzo_ipl ipl5,
			  lcrzo_ipl ipl6,
			  lcrzo_ipl ipl7,
			  lcrzo_ipl ipl8,
			  lcrzo_ipl ipl9,
			  lcrzo_int8 numberofsetipl);

/* record route */
int lcrzo_sock_ipopt_rr(lcrzo_sock *psock,
			lcrzo_int8 numberofroutestosave);

/* timestamp */
int lcrzo_sock_ipopt_time(lcrzo_sock *psock,
			  lcrzo_int8 numberoftimestamptosave);
int lcrzo_sock_ipopt_timeip(lcrzo_sock *psock,
			    lcrzo_int8 numberoftimestamptosave);
int lcrzo_sock_ipopt_timeipp(lcrzo_sock *psock,
			     lcrzo_ipl ipl1,
			     lcrzo_ipl ipl2,
			     lcrzo_ipl ipl3,
			     lcrzo_ipl ipl4,
			     lcrzo_int8 numberofsetipl);


/*-------------------------------------------------------------*/
/*EN                     reading and writing                     */
/*FR                     lecture et ecriture                     */
/*-------------------------------------------------------------*/
/*en Name : lcrzo_sock_write
   Description :
     Write data to a lcrzo_sock.
     If datasize is too long, data will be written in several
     chunks (use lcrzo_sock_write_msg if you want data to be written
     in one step).
   Input parameter(s) :
     sock : lcrzo_sock where the data is written
     data : data to write
     datasize : size of data
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_write
   Description :
     Ecrit des donnees dans un lcrzo_sock.
     Si datasize est trop long, les donnees sont ecrites en plusieurs
     fois (employez lcrzo_sock_write_msg si vous desirez que les
     donnees soient ecrites en une seule fois).
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont ecrites
     data : donnees a ecrire
     datasize : taille de data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_write(lcrzo_sock *psock,
		     lcrzo_constdata data, 
		     lcrzo_int32 datasize);
int lcrzo_sock_write_msg(lcrzo_sock *psock,
			 lcrzo_constdata data, 
			 lcrzo_int32 datasize);

/*en Name : lcrzo_sock_readm
   Description :
     Read data from a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
                               exited without reading data
*/
/*fr Nom : lcrzo_sock_readm
   Description :
     Lit des donnees depuis un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_sock_readm(lcrzo_sock *psock,
  		     lcrzo_time *pmaxwaitingtime, 
		     lcrzo_data *pdataout,
		     lcrzo_int32 *pdataoutsize);

/*en Name : lcrzo_sock_readm_fixed
   Description :
     Read a fixed size data from a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
     datasizetoread : size to read
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
                               exited without reading data
*/
/*fr Nom : lcrzo_sock_readm_fixed
   Description :
     Lit des donnees de taille constante depuis un lcrzo_sock.
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE)
     datasizetoread : taille des donnees a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_sock_readm_fixed(lcrzo_sock *psock,
    		           lcrzo_time *pmaxwaitingtime, 
			   lcrzo_int32 datasizetoread,
			   lcrzo_data *pdataout);

/*en Name : lcrzo_sock_readm_line
   Description :
     Read data from a lcrzo_sock. We stop at first '\r' or '\n' (which
     is not stored in data).
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : the function received nothing
                               before pmaxwaitingtime, so it 
                               exited without reading data
*/
/*fr Nom : lcrzo_sock_readm_line
   Description :
     Lit des donnees depuis un lcrzo_sock. On s'arrete de lire
     au premier '\r' ou '\n' (qui n'est pas stocke dans
     *pdataout).
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : la fonction n'a pas recu de donnees
                               avant pmaxwaitingtime, donc on 
                               retourne sans donnees.
*/
int lcrzo_sock_readm_line(lcrzo_sock *psock,
			  lcrzo_time *pmaxwaitingtime, 
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*EN                          closing                            */
/*FR                         fermeture                           */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_sock_close
   Description :
     Close a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock to close
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_close
   Description :
     Ferme un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock a fermer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_close(lcrzo_sock *psock);

/*-------------------------------------------------------------*/
/*                          information                        */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_sock_get_infos
   Description :
     Get information about a lcrzo_sock
   Input parameter(s) :
     sock : lcrzo_sock where to get info
   Input/output parameter(s) :
   Output parameter(s) :
     here_device : local device
     here_etha : local Ethernet address
     *phere_ipl : local IP address
     *phere_port : local port
     there_etha : remote Ethernet address
     *pthere_ipl : remote IP address
     *pthere_port : remote port
   Normal return values :
     LCRZO_ERR_OK : ok
   Note : here_device, here_etha and there_etha are only set 
          for virtual clients/servers.
*/
/*fr Nom : lcrzo_sock_get_infos
   Description :
     Obtient les informations d'un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock sur lequel obtenir des informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     here_device : device local
     here_etha : adresse Ethernet locale
     *phere_ipl : adresse IP locale
     *phere_port : port local
     there_etha : adresse Ethernet distante
     *pthere_ipl : adresse IP distante
     *pthere_port : port distant
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
   Note : here_device, here_etha et there_etha sont uniquement
          affectes dans les clients/serveurs virtuels.
*/
int lcrzo_sock_get_infos(lcrzo_sock *psock,
			 lcrzo_device here_device,
			 lcrzo_etha here_etha,
			 lcrzo_ipl *phere_ipl,
			 lcrzo_uint16 *phere_port,
			 lcrzo_etha there_etha,
			 lcrzo_ipl *pthere_ipl,
			 lcrzo_uint16 *pthere_port);

/*en print those information */
/*fr ecrit ces informations */
int lcrzo_sock_fprint_infos(LCRZOFILE *pf,
			    lcrzo_sock *psock);
/*en idem, but on stdout */
/*fr idem, mais sur stdout */
#define lcrzo_sock_print_infos(a) lcrzo_sock_fprint_infos(stdout, a)


/*-------------------------------------------------------------*/
/*EN                     virtual answering                       */
/*FR                    reponses virtuelles                      */
/*-------------------------------------------------------------*/

/*en Name : lcrzo_virtual_answer_arp
   Description :
     Answer to an ARP request.
     If ethpacket contains an ARP request for ipl, this
     function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address of ipl
     ipl : IP address searched
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_arp
   Description :
     Repond a une requete ARP
     Si ethpacket contient une requete ARP pour ipl, cette 
     fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet de ipl
     ipl : adresse IP recherchee
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_arp(lcrzo_constdata ethpacket,
			     lcrzo_uint32 ethpacketsize,
			     lcrzo_constdevice device,
			     lcrzo_constetha etha,
			     lcrzo_ipl ipl);

/*en Name : lcrzo_virtual_answer_rarp
   Description :
     Answer to a RARP request.
     If ethpacket contains a RARP request for etha, this
     function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address searched
     ipl : IP address of etha
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_rarp
   Description :
     Repond a une requete RARP
     Si ethpacket contient une requete RARP pour etha, cette 
     fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet recherchee
     ipl : adresse IP de etha
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_rarp(lcrzo_constdata ethpacket,
			      lcrzo_uint32 ethpacketsize,
			      lcrzo_constdevice device,
			      lcrzo_constetha etha,
			      lcrzo_ipl ipl);

/*en Name : lcrzo_virtual_answer_ping
   Description :
     Answer to a ping request (icmp echo).
     If ethpacket contains a ping request for ipl, this
     function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address of ipl
     ipl : IP address searched
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_ping
   Description :
     Repond a une requete ping (icmp echo).
     Si ethpacket contient une requete ping pour ipl, cette 
     fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet de ipl
     ipl : adresse IP recherchee
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_ping(lcrzo_constdata ethpacket,
			      lcrzo_uint32 ethpacketsize,
			      lcrzo_constdevice device,
			      lcrzo_constetha etha,
			      lcrzo_ipl ipl);

/*en Name : lcrzo_virtual_answer_alive
   Description :
     Answer to a arp/rarp/ping requests
     If ethpacket contains an arp/rarp/ping request
     for ipl, this function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address of ipl
     ipl : IP address searched
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_alive
   Description :
     Repond a une requete arp/rarpping.
     Si ethpacket contient une requete arp/rarp/ping
     pour ipl, cette fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet de ipl
     ipl : adresse IP recherchee
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_alive(lcrzo_constdata ethpacket,
			       lcrzo_uint32 ethpacketsize,
			       lcrzo_constdevice device,
			       lcrzo_constetha etha,
			       lcrzo_ipl ipl);

