package Lire::AsciiDlf::Records;

use strict;

use vars qw/ $VERSION @ISA /;

use Lire::Program qw/tempfile lr_debug /;
use Lire::Records;
use Lire::DataTypes qw/ :basic :time format_numeric_type /;
use Lire::AsciiDlf::GroupOp;
use Lire::Report::Entry;
use Lire::Report::Group;

use POSIX qw/ strftime /;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.5 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw/ Lire::Records Lire::AsciiDlf::GroupOp /;
}

sub init_report {
    my ($self, $dlf_info) = @_;

    # Cache index and formatters
    my @slices = ();
    my @formatters = ();
    foreach my $f ( @{$self->fields()} ) {
	my $field = $self->{report_spec}->schema->field( $f );
	push @slices, $field->pos;
	my $type = $field->type;
	if ( is_time_type( $type ) ) {
	    push @formatters, 
	      sub { strftime( '%Y-%m-%d %H:%M:%S', localtime $_[0] ) };
	} elsif ( is_quantity_type( $type ) ) {
	    push @formatters,
	      sub { format_numeric_type( $_[0], $type ) };
	} else {
	    push @formatters, sub { $_[0] };
	}
    }
    $self->{slices}	= \@slices;
    $self->{formatters} = \@formatters;

    # We use a temporary file to sort the DLF records according to
    # the group id we are using.
    $self->{fh}		= tempfile();
    $self->{group_no}	= 0;

    $self;
}

sub init_group_data {
    my ( $self ) = @_;

    my $value = $self->{group_no}++;

    return $value;
}

sub update_group_data {
    my ( $self, $dlf, $data ) = @_;

    my $fh = $self->{fh};
    print $fh join( " ", $data, @$dlf[@{$self->{slices}}] ), "\n";
}

sub end_group_data {
    my ( $self, $data ) = @_;

    my $fh = $self->{fh};
    print $fh $data, " LIRE_RECORDS_GROUP_DATA_END\n";
}

sub add_entry_value {
    my ( $self, $entry, $data ) = @_;

    my $group = new Lire::Report::Group;
    $self->create_entries( $group, $data );
    $entry->add_group( $group );
}

sub create_entries {
    my ( $self, $group, $data ) = @_;

    $data ||= 0;

    seek $self->{fh}, 0, 0
      or die "can't seek at the beginning of the temporary DLF file: $!\n";

    my $count = 0;

    my $dlf_fh = $self->{fh};
    my $rx = qr/^$data /;
    while (<$dlf_fh>) {
	# Only keep the records matching the group id
	next unless /$rx/;

	chomp;
	my ( $id, @fields) = split / /, $_;

	# Check for end of group data marker
	last if $fields[0] eq 'LIRE_RECORDS_GROUP_DATA_END';

	my $entry = new Lire::Report::Entry;

	# FIXME: This needs to be revised once be get to 
	# the overhaul of the formatting engine. i.e. Multiple
	# names/values formatting.
	my $i = 0;
	foreach my $f ( @{$self->{formatters}} ) {
	    my $value = $fields[$i++];
	    my $content = $f->( $value );
	    if ( $i == $#fields ) {
		$entry->add_value( $content, $value );
	    } else {
		$entry->add_name( $content, $value );
	    }
	}

	$group->add_entry( $entry );

	$count++;
    }
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::AsciiDlf::Records -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Records.pm,v 1.5 2002/08/16 22:48:54 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
