package Lire::GroupField;

use strict;

use vars qw( $VERSION );

use Lire::DataTypes qw( :xml is_quantity_type is_time_type );
use Lire::ReportSpec;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.4 $' =~ m!Revision: ([.\d]+)!;
}

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;


    my %params = @_;
    my $name	= $params{name};
    croak "invalid name for a field: $name"
      unless check_xml_name( $name );

    my $report_spec = $params{report_spec};
    croak "report_spec argument must be of type Lire::ReportSpec"
      unless UNIVERSAL::isa( $report_spec, "Lire::ReportSpec" );

    croak "field $name isn't defined in the ", $report_spec->schema->id,
      " schema"
	unless $report_spec->schema->has_field( $name );

    my $type = $report_spec->schema->field( $name )->type;
    croak "field ", $name, "  is of type ", $type, 
      ". Use Lire::Rangegroup for this type\n"
	if ( is_quantity_type( $type ) );

    croak "field ", $name, "  is of type ", $type, 
      ". Use Lire::Timeslot or Lire::Timegroup for this type\n"
	if ( is_time_type( $type ) );

    bless { name	=> $name,
	    report_spec => $report_spec,
	  }, $class;
}

sub name {
    $_[0]{name};
}

sub field {
    $_[0]{report_spec}->schema()->field( $_[0]{name} );
}

sub print {
    my ($self, $fh, $pfx ) = @_;
    $pfx ||= 0;
    $fh  ||= \*STDOUT;

    print $fh ' ' x $pfx;
    print $fh "<lire:field name=\"$self->{name}\"/>\n";
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::GroupField -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: GroupField.pm,v 1.4 2002/03/23 23:33:54 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
