package Lire::Report::Image;

use strict;

use vars qw/ $VERSION @ISA /;

use Lire::XMLUtils qw/xml_encode/;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.3 $' =~ m!Revision: ([.\d]+)!;
}

=pod

=head1 NAME

Lire::Report::Image - Interface to the image's element.

=head1 SYNOPSIS

    print $image->title, "\n";
    print $image->file, " is of type ", $image->format, "\n";

=head1 DESCRIPTION

This class offers an API to the image's elements of a Lire report.

=head1 CONSTRUCTOR

=head2 new( $file, $format ) 

Creates a new Lire::Report::Image object. The $file and $format
parameters are used to sets the attribute with the same name.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my ( $file, $format ) = @_;

    croak __PACKAGE__, '::new: missing $file argument'   
      unless defined $file;
    croak __PACKAGE__, '::new: missing $format argument' 
      unless defined $format;

    bless { 
	   format => $format,
	   file   => $file,
	  }, $class;

}

=pod

=head1 OBJECT METHODS

=head2 title([$title])

Returns the image's title, if it has one. If the $title parameter is
set, the image's title will be set to this new value.

=cut

sub title {
    $_[0]{title} = $_[1] if defined $_[1];
    $_[0]{title};
}

=pod

=head2 format([$format])

Returns the image's file format. If the $format parameter is set, the
image's format attribute will be set to this new value.

=cut

sub format {
    $_[0]{format} = $_[1] if defined $_[1];
    
    $_[0]{format};
}

=pod

=head2 file([$file])

Returns the path to the image's file. This is the path that was
present in the XML report file, it may or may not exists as a real
file on the filesystem. If the $format parameter is set, the image's
file attribute will be set to this new value.

=cut

sub file {
    $_[0]{file} = $_[1] if defined $_[1];
    $_[0]{file};
}

sub write_report {
    my ( $self, $fh, $indent ) = @_;

    $fh ||= *STDOUT;
    my $pfx = ' ' x $indent;

    print $fh qq"$pfx<lire:image>\n";

    print $fh "$pfx <lire:title>", xml_encode( $self->{title} ),
      "</lire:title>\n\n"
      if defined $self->{title};

    print $fh qq!$pfx <lire:file format="$self->{format}">!, 
      xml_encode( $self->{file}), "</lire:file>\n\n"
      if defined $self->{title};

    print $fh "$pfx</lire:image>\n";
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportParser::ReportBuilder(3pm) Lire::Report(3pm)
Lire::Report::Subreport(3pm) Lire::Report::Entry(3pm)
Lire::Report::Group(3pm)

=head1 VERSION

$Id: Image.pm,v 1.3 2002/06/23 19:23:40 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
