package Lire::ReportSpec;

use strict;

use vars qw( $VERSION @ISA );

use Lire::Config;
use Lire::DataTypes qw( :special );
use Lire::XMLSpecContainer;
use Lire::Report::Subreport;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.24 $' =~ m!Revision: ([.\d]+)!;

    @ISA = qw/Lire::XMLSpecContainer/;
}

=pod

=head1 NAME

Lire::ReportSpec - API to the XML report specifications.

=head1 SYNOPSIS

    use Lire::ReportSpec;

=head1 DESCRIPTION

FIXME: Write some docs!

=cut


########################################################################
#			 Lire::XMLSpecContainer METHODS
########################################################################

sub root_element {
    my ( $self ) = @_;

    return "report-spec";
}

sub root_xml_attrs {
    my ( $self ) = @_;
    if (defined $self->{charttype}) {
	return qq{ charttype="$self->{charttype}"}
    } else {
	return "";
    }
}

sub file_from_id {
    my ( $self, $super, $id ) = @_;

    my $file;
    # Find in REPORTS_PATH
    foreach my $dir ( split /:/, $Lire::Config::REPORTS_PATH ) {
	next unless -d $dir;
	if ( -e $dir . "/$super/$id.xml" ) {
	    $file = $dir . "/$super/$id.xml";
	    last;
	}
    }
    die "can't find XML report specification for $id in $Lire::Config::REPORTS_PATH\n"
      unless defined $file;

    return $file;
}

sub print_children {
    my ( $self, $fh, $indent ) = @_;

    my $pfx = ' ' x $indent;
    if ( defined $self->{filter_spec} ) {
	print $fh "$pfx<lire:filter-spec>\n";
	$self->{filter_spec}->print( $fh, $indent + 1);
	print $fh "\n$pfx</lire:filter-spec>\n\n";
    }

    if ( defined $self->{calc_spec} ) {
	print $fh "$pfx<lire:report-calc-spec>\n";
	$self->{calc_spec}->print( $fh, $indent + 1 );
    } else {
	$self->{extension}->print( $fh, $indent );
    }
}

=pod

=head1 OBJECT METHODS

=cut

sub charttype {
    my ( $self, $chart ) = @_;

    if ( @_ == 2 ) {
	if ( $chart) {
	    croak "invalid chart type : $chart"
	      unless ( check_chart( $chart ));

	    $self->{charttype} = $chart;
	} else {
	    delete $self->{charttype};
	}
    }

    $self->{charttype};
}

sub filter_spec {
    my ( $self, $filter_spec ) = @_;

    if ( @_ == 2 ) {
	if ( defined $filter_spec ) {
	    croak "filter specification must be of type Lire::FilterExpr"
	      unless UNIVERSAL::isa( $filter_spec, "Lire::FilterExpr" );
	    $self->{filter_spec} = $filter_spec;
	} else {
	    delete $self->{filter_spec};
	}
    }

    $self->{filter_spec};
}

sub calc_spec {
    my ( $self, $calc_spec ) = @_;

    if ( @_ == 2 ) {
	croak "calc specification must be of type Lire::Aggregator"
	  unless UNIVERSAL::isa( $calc_spec, "Lire::Aggregator" );
	$self->{calc_spec} = $calc_spec;
	delete $self->{extension};
    }

    $self->{calc_spec};
}

sub extension {
    my ( $self, $extension ) = @_;
    if ( @_ == 2 ) {
	croak "extension must be of type Lire::Extension"
	  unless UNIVERSAL::isa( $extension, "Lire::Extension" );
	$self->{extension} = $extension;
	delete $self->{calc_spec};
    }

    $self->{extension};
}

sub needed_fields {
    my ( $self ) = @_;

    my @fields = ();
    my $filter_spec = $self->filter_spec;
    push @fields, $filter_spec->needed_fields
      if defined $filter_spec;

    my $calc_spec = $self->calc_spec;
    push @fields, $calc_spec->needed_fields
      if defined $calc_spec;

    my $extension = $self->extension;
    push @fields, $calc_spec->needed_fields
      if defined $extension;

    @fields;
}

=pod

=head2 mark_missing( $reason )

Mark that the report that should be generated by this report
specification should be marked as missing. The $reason parameter gives
the reason why the subreport will be missing.

=cut

sub mark_missing {
    my ( $self, $reason ) = @_;

    $self->{missing} = $reason;
}

=pod

=head2 is_missing()

Returns true if the subreport generated by this report specification
should be marked missing.

=cut

sub is_missing {
    return defined $_[0]{missing};
}

=pod

=head2 create_subreport( $report )

Creates a Lire::Report::Subreport object based on this report
specification. This will be a missing subreport if the mark_missing()
method was called. The $report parameter should be of type
Lire::Report and is needed by the Lire::Report::Subreport constructor.

=cut

sub create_subreport {
    my ( $self, $report ) = @_;

    croak "report parameter should be of type Lire::Report, not $report"
      unless UNIVERSAL::isa( $report, "Lire::Report" );

    if ( $self->{missing} ) {
	return new_missing Lire::Report::Subreport( $report, $self->id,
						    $self->{missing}) ;
    }
    
    my $subreport = new Lire::Report::Subreport( $report, $self->id );
    $subreport->title( $self->expanded_display_title );
    $subreport->charttype( $self->charttype );
    foreach my $u ( $self->target_users ) {
	$subreport->description( $u, 
				 $self->expanded_display_description( $u ) );
    }

    $self->create_entries( $subreport );

    $subreport;
}

=pod

=head1 METHODS THAT SHOULD BE OVERRIDEN BY IMPLEMENTATION

=head2 create_entries($subreport) 

This method is used by the create_subreport() implementation to
populate the subreport's data. The default implementation does nothing
which will result in empty subreports. The entries should be added to
the $subreport object.

=cut

sub create_entries {}

# keep perl happy
1;

__END__

=pod

=head1 VERSION

$Id: ReportSpec.pm,v 1.24 2002/07/12 22:51:30 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
