package Lire::Timeslot;

use strict;

use vars qw( $VERSION @ISA );

use Lire::NestableAggregator;
use Lire::DataTypes qw( :special :time );

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.5 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw( Lire::NestableAggregator );
}

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my %params = @_;

    my $self = $class->SUPER::new( %params,
				   element_name => "timeslot",
				 );

    $self->field( $params{field} );

    $self->unit( $params{unit} )
      if exists $params{unit};

    $self;
}

sub field {
    my ( $self, $field ) = @_;

    if (@_ == 2 ) {
	if ( defined $field ) {
	    croak "$field isn't a valid field for the ",
	      $self->{report_spec}->schema->id, " schema"
		unless $self->{report_spec}->schema->has_field( $field );

	    croak "$field isn't a timestamp field"
	      unless $self->{report_spec}->schema()->field( $field )->type()
		eq "timestamp";

	} else {
	    $field = $self->{report_spec}->schema()->timestamp_field()->name();
	}
	$self->{field} = $field;
    }

    return $self->{field};
}

sub unit {
    my ( $self, $unit ) = @_;

    if (@_ == 2 ) {
	my $unit_sec;
	if ( $unit =~ /^\$/ ) {
	    my $name = substr $unit, 1;
	    croak "$name isn't a defined parameter"
	      unless $self->{report_spec}->has_param( $name );

	    croak "$name parameter isn't of type duration"
	      unless $self->{report_spec}->param( $name )->type()
		eq "duration";
	} else {
	    croak "$unit isn't a valid duration"
	      unless check_duration( $unit );
	}

	$self->{unit} = $unit;
    }

    return $self->{unit};
}

sub needed_fields {
    my ( $self ) = @_;

    my @fields = ( $self->{report_spec}->schema->field( $self->field ) );
    push @fields, $self->SUPER::needed_fields;

    return @fields;
}

sub xml_attrs {
    my ( $self ) = @_;

    return qq{field="$self->{field}" unit="$self->{unit}"};
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Timeslot -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Timeslot.pm,v 1.5 2002/03/23 23:33:54 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
