package Lire::Aggregator;

use strict;

use vars qw( $VERSION );

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.3 $' =~ m!Revision: ([.\d]+)!;
}

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;


    my %params = @_;

    my $report_spec = $params{report_spec};
    croak "report_spec parameter isn't of type Lire::ReportSpec"
      unless UNIVERSAL::isa( $report_spec, "Lire::ReportSpec" );

    croak "missing element_name parameter"
      unless defined $params{element_name};

    my $self = bless {
		      report_spec   => $report_spec,
		      element_name  => $params{element_name},
		      ops	    => [],
		     }, $class;

    $self;
}

sub ops {
    my ( $self, $ops ) = @_;

    if ( @_ == 2 ) {
	croak "$ops isn't an array reference"
	  unless UNIVERSAL::isa( $ops, "ARRAY" );

	croak "fields array is empty"
	  if @$ops == 0;

	foreach my $op ( @$ops ) {
	    croak "$op isn't of type Lire::GroupOp"
	      unless UNIVERSAL::isa( $op, "Lire::GroupOp" );
	}

	$self->{ops} = $ops;
    }

    return $self->{ops};
}

sub op_by_name {
    my ( $self, $name ) = @_;

    # Check in ops
    foreach my $op ( @{$self->{ops}} ) {
	return $op if $op->name eq $name;
    }

    die "No operation named $name\n";
}

sub is_name_defined {
    my ( $self, $name ) = @_;

    # Check in ops
    foreach my $p ( @{$self->{ops}} ) {
	return 1 if $p->name eq $name;
    }
}

sub needed_fields {
    my ( $self ) = @_;

    my @fields = ();

    foreach my $op ( @{$self->ops} ) {
	push @fields, $op->needed_fields;
    }

    return @fields;
}

sub print {
    my ( $self, $fh, $pfx ) = @_;
    $fh  ||= \*STDOUT;
    $pfx ||= 0;

    my $prefix = " " x $pfx;
    my $attrs = $self->xml_attrs;

    print $fh $prefix, qq{<lire:$self->{element_name}};
    print $fh ' ', $attrs if length $attrs;
    print $fh ">\n";

    $self->print_content( $fh, $prefix + 1);

    print $fh $prefix, "</lire:$self->{element_name}>\n";
}

sub xml_attrs {
    return "";
}

sub print_content {
    my ( $self, $fh, $pfx ) = @_;

    $self->print_ops( $fh, $pfx );
}

sub print_ops {
    my ( $self, $fh, $pfx ) = @_;

    foreach my $o ( @{$self->{ops}} ) {
	$o->print( $fh, $pfx + 1);
    }
}


# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Aggregator - Base class for all lire:aggregator element

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Aggregator.pm,v 1.3 2002/06/08 21:49:43 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
