package Lire::Group;

use strict;

use vars qw( $VERSION @ISA $GROUP_ID );

use Lire::NestableAggregator;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.11 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw( Lire::NestableAggregator );
    $GROUP_ID = 0;
}

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;
    my %args = @_;

    my $self = $class->SUPER::new( %args, element_name => 'group' );
    $self->{fields} = [];

    $self->limit( $args{limit} )
      if exists $args{limit};

    $self;
}

sub limit {
    my ( $self, $limit ) = @_;

    if ( @_ == 2 ) {
	if ( defined $limit && $limit =~ /^\$/ ) {
	    my $name = substr $limit, 1;
	    croak "$limit isn't a defined parameter"
	      unless $self->{report_spec}->has_param( $name );
	    my $type = $self->{report_spec}->param( $name )->type();
	    croak "$limit parameter isn't of type int"
	      unless $type eq "int";
	    $self->{limit} = $limit;
	} elsif ( defined $limit ) {
	    croak "limit must be a positive integer"
	      unless check_int( $limit );
	    $self->{limit} = $limit;
	} else {
	    delete $self->{limit};
	}
    }
    return $self->{limit};
}

sub sort_fields {
    my ( $self, $fields ) = @_;

    if (@_ == 2 && defined $fields ) {
	croak "$fields isn't an array reference"
	  unless UNIVERSAL::isa( $fields, "ARRAY" );

	foreach my $f ( @$fields ) {
	    croak "$f isn't a defined sort field name"
	      unless $self->is_valid_sort_field( $f );
	}
	if ( @$fields ) {
	    $self->{sort_fields} = $fields;
	} else {
	    delete $self->{sort_fields};
	}
    } elsif ( @_ == 2 ) {
	delete $self->{sort_fields};
    }

    return $self->{sort_fields};
}

sub group_fields {
    my ( $self, $fields ) = @_;

    if ( @_ == 2 ) {
	croak "$fields isn't an array reference"
	  unless UNIVERSAL::isa( $fields, "ARRAY" );

	croak "group fields array is empty"
	  unless @$fields;

	foreach my $f ( @$fields ) {
	    croak "$f isn't of type Lire::GroupField"
	      unless UNIVERSAL::isa( $f, "Lire::GroupField" );
	}
	$self->{fields} = $fields;
    }
    return $self->{fields};
}

sub is_name_defined {
    my ( $self, $name ) = @_;

    # Check in ops
    return 1 if $self->SUPER::is_name_defined( $name );

    # Check in fields
    foreach my $f ( @{$self->{fields}} ) {
	return 1 if $f->name eq $name;
    }

    return 0;
}

sub is_valid_sort_field {
    my ( $self, $name ) = @_;

    # Remove descending control char
    $name =~ s/^-//g;

    $self->is_name_defined( $name );
}

sub needed_fields {
    my ( $self ) = @_;

    my @fields = $self->SUPER::needed_fields;

    foreach my $f ( @{$self->group_fields} ) {
	push @fields, $f->field;
    }

    return @fields;
}

sub xml_attrs {
    my ( $self ) = @_;

    my $attr = "";
    if ( defined $self->{sort_fields} &&
	 @{$self->{sort_fields}})
    {
	my $fields = join " ", @{$self->{sort_fields}};
	$attr .= qq{ sort="$fields"};
    }
    $attr .= qq{ limit="$self->{limit}"}
      if exists $self->{limit};

    $attr;
}

sub print_content {
    my ( $self, $fh, $pfx ) = @_;

    foreach my $f ( @{$self->{fields}} ) {
	$f->print( $fh, $pfx + 1);
    }

    $self->print_ops( $fh, $pfx );
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Group -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Group.pm,v 1.11 2002/03/23 23:33:54 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
