package Lire::Merge::GroupOp;

use strict;

use Lire::GroupOp;

use Lire::DataTypes qw/ format_numeric_type /;
use vars qw( $VERSION @ISA );

BEGIN {
    ($VERSION)	= '$Revision: 1.3 $' =~ m!Revision: ([.\d]+)!;
}

sub init_merge {
    my ( $self, $period_start, $period_end ) = @_;
}

sub end_merge {}

sub init_group_data {
    my $scalar = 0;
    return \$scalar;
}

sub end_group_data {}

sub add_entry_value {
    my ( $self, $entry, $data ) = @_;

    $self->add_content_value( $entry, $$data );
}

sub add_content_value {
    my ( $self, $entry, $value ) = @_;

    my $type	    = $self->{field_type} || "number";
    my $content   = format_numeric_type( $value, $type );

    $entry->add_value( $content, $value );
}

package Lire::Merge::GroupOp::Count;

use vars qw/ @ISA /;

BEGIN {
    @ISA = qw/ Lire::GroupOp::Count Lire::Merge::GroupOp /;
}

sub merge_entry_value {
    my ( $self, $value, $data ) = @_;

    # To merge two counts, we simply add them
    $$data += $value->{value};
}

package Lire::Merge::GroupOp::SimpleStat;

use vars qw/ @ISA /;

BEGIN {
    @ISA = qw/ Lire::GroupOp::SimpleStat Lire::Merge::GroupOp /;
}

sub init_merge {
    my ($self, $period_start, $period_end ) = @_;

    my $field = $self->{report_spec}->schema()->field( $self->field );
    $self->{field_type} = $field->type;

    $self;
}

package Lire::Merge::GroupOp::Sum;

use vars qw/ @ISA /;

BEGIN {
    @ISA = qw/ Lire::GroupOp::Sum Lire::Merge::GroupOp::SimpleStat /;
}

sub merge_entry_value {
    my ( $self, $value, $data ) = @_;

    # To merge two sums, we simply add them
    $$data += $value->{value};

}

package Lire::Merge::GroupOp::Avg;

use Lire::DataTypes qw/format_numeric_type/;
use Carp;

use vars qw/ @ISA /;

use constant AVG_VALUE => 0;
use constant AVG_SUM   => 1;
use constant AVG_N     => 2;

BEGIN {
    @ISA = qw/ Lire::GroupOp::Avg Lire::Merge::GroupOp /;
}

sub init_merge {
    my ( $self, $period_start, $period_end ) = @_;

    if ( $self->field ) {
	my $f = $self->{report_spec}->schema()->field( $self->field );
	$self->{field_type} = $f->type;
    }
}

sub init_group_data {
    my ( $self ) = @_;

    return [ 0, 0, 0 ];
}

sub merge_entry_value {
    my ( $self, $value, $data ) = @_;

    # Adds the total and n
    $data->[AVG_SUM] += $value->{total};
    $data->[AVG_N] += $value->{n};

}

sub end_group_data {
    my ( $self, $data ) = @_;

    if ( $data->[AVG_N]) {
	$data->[AVG_VALUE] = $data->[AVG_SUM] / $data->[AVG_N];
	$data->[AVG_VALUE] = sprintf "%.2f", $data->[AVG_VALUE];
    } else {
	$data->[AVG_VALUE] = "n/a";
    }
}

sub add_entry_value {
    my ( $self, $entry, $data ) = @_;

    my $type	 = $self->{field_type} || "number";
    my $content  = format_numeric_type( $data->[AVG_VALUE], $type );

    $entry->add_value( $content, $data->[AVG_VALUE], $data->[AVG_SUM], 
		       $data->[AVG_N] );
}

package Lire::Merge::GroupOp::Min;

use vars qw/ @ISA /;

BEGIN {
    @ISA = qw/ Lire::GroupOp::Min Lire::Merge::GroupOp::SimpleStat /;
}

sub init_group_data {
    my $scalar = undef;
    return \$scalar;
}

sub merge_entry_value {
    my ( $self, $value, $data ) = @_;

    # To merge two mins, we keep the lowest
    $$data = $value->{value}
      unless ( defined $$data );

    $$data = $value->{value}
      if $value->{value} < $$data;
}

sub end_group_data {
    my ( $self, $data ) = @_;

    $$data = "n/a" unless defined $$data;
}

package Lire::Merge::GroupOp::Max;

use vars qw/ @ISA /;

BEGIN {
    @ISA = qw/ Lire::GroupOp::Max Lire::Merge::GroupOp::SimpleStat /;
}

sub init_group_data {
    my $scalar = undef;
    return \$scalar;
}

sub merge_entry_value {
    my ( $self, $value, $data ) = @_;

    $$data = $value->{value}
      unless ( defined $$data );

    $$data = $value->{value}
      if $value->{value} > $$data;
}

sub end_group_data {
    my ( $self, $data ) = @_;

    $$data = "n/a" unless defined $$data;
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Merge::GroupOp -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: GroupOp.pm,v 1.3 2002/06/18 14:40:44 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
