package Lire::Rangegroup;

use strict;

use vars qw( $VERSION @ISA );

use Lire::NestableAggregator;
use Lire::DataTypes qw( :time :misc :basic );

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.4 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw( Lire::NestableAggregator );
}

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my %params = @_;

    my $self = $class->SUPER::new( %params,
				   element_name => "rangegroup",
				 );

    croak "missing 'field' attribute"
      unless exists $params{field};
    $self->field( $params{field} );

    croak "missing 'range-size' attribute"
      unless exists $params{'range-size'};
    $self->range_size( $params{'range-size'} );

    $self->range_start( $params{'range-start'} || 0 );

    $self->min_value( $params{'min-value'} )
      if exists $params{'min-value'};
    $self->max_value( $params{'max-value'} )
      if exists $params{'max-value'};

    $self->size_scale( $params{'size-scale'} || 1 );

    $self;
}

sub field {
    my ( $self, $field ) = @_;

    if (defined $field) {
	my $schema = $self->{report_spec}->schema;
	croak "$field isn't a valid field for the ", $schema->id, " schema"
	  unless $schema->has_field( $field );

	croak "$field isn't a bytes, duration, int or number field"
	  unless is_quantity_type( $schema->field( $field )->type() );

	$self->{field} = $field;
    }

    return $self->{field};
}

sub set_attr_value {
    my ( $self, $attr_name, $value ) = @_;

    my $field = $self->field;
    my $type  = $self->{report_spec}->schema->field( $field )->type;
    if ( $value =~ /^\$/ ) {
	my $name = substr $value, 1;
	    croak "$value isn't a defined parameter"
	      unless $self->{report_spec}->has_param( $name );

	croak "'$attr_name' attribute: '$value' should be of type $type"
	  if $self->{report_spec}->param( $name )->type() ne $type;

    } elsif ( $type eq 'duration' ) {
	croak "invalid duration value for '$attr_name' attribute: $value"
	  unless check_duration( $value );
    } elsif ($type eq 'bytes' ) {
	croak "invalid bytes value for '$attr_name' attribute: $value"
	  unless check_bytes( $value );
    } else {
	croak "invalid numeric value for '$attr_name' attribute: $value"
	  unless check_number( $value );
    }
    $attr_name =~ s/-/_/g;

    $self->{$attr_name} = $value;
}


sub range_start {
    my ( $self, $start ) = @_;

    $self->set_attr_value( 'range-start', $start )
      if (defined $start );

    return $self->{range_start};
}

sub range_size {
    my ( $self, $size ) = @_;

    $self->set_attr_value( 'range-size', $size )
      if (defined $size );

    return $self->{range_size};
}

sub min_value {
    my ( $self, $min ) = @_;

    $self->set_attr_value( 'min-value', $min )
      if (defined $min );

    return $self->{min_value};
}

sub max_value {
    my ( $self, $max ) = @_;

    $self->set_attr_value( 'max-value', $max )
      if (defined $max );

    return $self->{max_value};
}

sub size_scale {
    my ( $self, $size_scale ) = @_;

    if ( defined $size_scale ) {
	if ( $size_scale =~ /^\$/ ) {
	    my $name = substr $size_scale, 1;
	    croak "$size_scale isn't a defined parameter"
	      unless $self->{report_spec}->has_param( $name );

	    croak "'size-scale' attribute must be a number type"
	      unless $self->{report_spec}->param( $name )->type() ne 'number';

	} else {
	    croak "invalid 'size-scale' value. It should be a number"
	      unless check_number( $size_scale );
	}
	$self->{size_scale} = $size_scale;
    }

    return $self->{size_scale};
}

sub needed_fields {
    my ( $self ) = @_;

    my @fields = ( $self->{report_spec}->schema->field( $self->field ) );
    push @fields, $self->SUPER::needed_fields;

    return @fields;
}

sub xml_attrs {
    my ( $self ) = @_;

    return qq{field="$self->{field}" range-start="$self->{range_start}" range-size="$self->{range_size}"};
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Rangegroup -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Rangegroup.pm,v 1.4 2002/03/23 23:33:54 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
