package Lire::Records;

use strict;

use vars qw( $VERSION @ISA $RECORDS_ID );

use Lire::GroupOp;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.2 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw( Lire::GroupOp );
    $RECORDS_ID = 0;
}

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my %args = @_;

    my $report_spec = $args{report_spec};
    croak "report_spec argument ($report_spec) must be of type Lire::ReportSpec"
      unless UNIVERSAL::isa( $report_spec, "Lire::ReportSpec" );

    croak "missing parent parameter"
      unless defined $args{parent};
    croak "parent argument ($args{parent}) must be of type Lire::Aggregator"
      unless UNIVERSAL::isa( $args{parent}, "Lire::Aggregator" );
    croak "missing fields parameter"
      unless defined $args{fields};

    my $self = bless { report_spec  => $report_spec,
		       op	    => "records", # For print
		       parent	    => undef,
		       name	    => "*records" . $RECORDS_ID ++ . "*",
		     }, $class;

    $self->fields( $args{fields} );

    $self;
}

sub name {
    $_[0]->{name};
}

sub print {
    my ($self,$fh, $prefix) = @_;
    $fh	    ||= \*STDOUT;
    $prefix ||= 0;

    my $pfx = " " x $prefix;

    print $fh $pfx, '<lire:records fields="', join( " ", @{$self->{fields}}),
      qq{"/>\n};
}

sub fields {
    my ($self, $fields) = @_;

    if ( @_ == 2 ) {
	if ( defined $fields ) {
	    croak "$fields isn't an array reference"
	      unless UNIVERSAL::isa( $fields, 'ARRAY' );
	    die "fields cannot be empty\n"
	      unless @$fields;

	    foreach my $f ( @$fields ) {
		croak "$f isn't a defined field in the ",
		  $self->{report_spec}->schema->id, " schema"
		    unless $self->{report_spec}->schema()->has_field( $f );
	    }
	} else {
	    croak "undefined fields\n";
	}
	$self->{fields} = $fields;
    }
    $self->{fields};
}

sub needed_fields {
    my ( $self ) = @_;

    return map { $self->{report_spec}->schema->field($_) } @{$self->{fields}};
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::Summary -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Records.pm,v 1.2 2002/08/06 21:00:43 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
