package Lire::Report::Group;

use strict;

use vars qw/ $VERSION @ISA /;

BEGIN {
    ($VERSION)	= '$Revision: 1.4 $' =~ m!Revision: ([.\d]+)!;
}

=pod

=head1 NAME

Lire::Report::Group - Object that include subgroups and other entries.

=head1 SYNOPSIS

    foreach my $e ( $group->entries ) {
	# Process entries' data.
    }

=head1 DESCRIPTION

The Lire::Report::Group object contains entries grouped together in a
subreport.

=head1 CONSTRUCTOR

=head2 new()

Creates a new empty Lire::Report::Group object. Use the add_entry()
method to fill this group element.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    bless {
	   entries	 => [],
	  }, $class;
}

=pod


=head1 OBJECT METHODS

=pod

=head2 show( [$show] )

Returns the number of entries that should be displayed in the
formatted report. There may be more entries than that in the group.
When this parameter is zero or undefined, all available entries should
be displayed.

If the $show is parameter is set, the group's show parameter will
be set to this new value.

=cut

sub show {
    $_[0]{show} = $_[1] if @_ == 2;
    $_[0]{show};
}

=head2 entries()

Returns the entries in the group.

=cut

sub entries {
    @{$_[0]{entries}};
}

=pod

=head2 add_entry( $entry )

Adds a data entry to this group. The $entry parameter should be a
Lire::Report::Entry object.

=cut

sub add_entry {
    my ( $self, $entry ) = @_;

    croak ( "entry should be of type Lire::Report::Entry (not $entry)" )
      unless UNIVERSAL::isa( $entry, "Lire::Report::Entry" );

    push @{$self->{entries}}, $entry; 
}

sub write_report {
    my ( $self, $fh, $indent ) = @_;

    return unless $self->entries;

    $fh ||= *STDOUT;
    my $pfx = ' ' x $indent;

    print $fh "$pfx <lire:group";
    print $fh qq! show="$self->{show}"! 
      if defined $self->{show};
    print $fh ">\n";

    foreach my $e ( $self->entries ) {
	$e->write_report( $fh, $indent + 1 );
    }

    print $fh "$pfx</lire:group>\n";
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportParser::ReportBuilder(3pm) Lire::Report(3pm)
Lire::Report::Subreport(3pm) Lire::Report::Entry(3pm)
Lire::Report::Image(3pm) Lire::Report::Section(3pm)

=head1 VERSION

$Id: Group.pm,v 1.4 2002/06/23 19:23:40 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
