package org.jibble.pircbot;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * A Thread which reads lines from the IRC server.  It then
 * passes these lines to the PircBot without changing them.
 * This running Thread also detects disconnection from the server
 * and is thus used by the OutputThread to send lines to the server.
 *
 * @copyright Paul James Mutton, 2001-2002.
 * @author    Paul James Mutton, http://www.jibble.org/
 * @version 0.9.5 (Build time: Fri Jun 14 21:16:31 2002)
 */
public class InputThread extends Thread {
    
    /**
     * The InputThread reads lines from the IRC server and allows the
     * PircBot to handle them.
     *
     * @param bot An instance of the underlying PircBot.
     * @param breader The BufferedReader that reads lines from the server.
     */
    protected InputThread(PircBot bot, BufferedReader breader, BufferedWriter bwriter) {
        _bot = bot;
        _breader = breader;
        _bwriter = bwriter;
    }
    
    
    /**
     * Sends a raw line to the IRC server as soon as possible, bypassing the
     * outgoing message queue.
     *
     * @param line The raw line to send to the IRC server.
     */
    public void sendRawLine(String line) {
        if (line.length() > _bot.getMaxLineLength() - 2) {
            line = line.substring(0, _bot.getMaxLineLength() - 2);
        }
        synchronized(_bwriter) {
            try {
                _bwriter.write(line + "\r\n");
                _bwriter.flush();
                _bot.log(">>>" + line);
            }
            catch (Exception e) {
                // Silent response - just lose the line.
            }
        }
    }
    
    
    /**
     * Called to start this Thread reading lines from the IRC server.
     * When a line is read, this method calls the handleLine method
     * in the PircBot, which may subsequently call an 'onXxx' method
     * in the PircBot subclass.  If any subclass of Throwable (i.e.
     * any Exception or Error) is thrown by your method, then this
     * method will print the stack trace to the standard output.  It
     * is probable that the PircBot may still be functioning normally
     * after such a problem, but the existance of any uncaught exceptions
     * in your code is something you should really fix.
     *
     */
    public void run() {
        try {
            String line = null;
            while ((line = _breader.readLine()) != null) {
                try {
                    _bot.handleLine(line);
                }
                catch (Throwable t) {
                    // Stick the whole stack trace into a String so we can output it nicely.
                    StringWriter sw = new StringWriter();
                    PrintWriter pw = new PrintWriter(sw);
                    t.printStackTrace(pw);
                    pw.flush();
                    StringTokenizer tokenizer = new StringTokenizer(sw.toString(), "\r\n");
                    synchronized (_bot) {
                        _bot.log("### Your implementation of PircBot is faulty and you have");
                        _bot.log("### allowed an uncaught Exception or Error to propagate in your");
                        _bot.log("### code. It may be possible for PircBot to continue operating");
                        _bot.log("### normally. Here is the stack trace that was produced: -");
                        _bot.log("### ");
                        while (tokenizer.hasMoreTokens()) {
                            _bot.log("### " + tokenizer.nextToken());
                        }
                    }
                }
            }
        }
        catch (Exception e) {
            // Do nothing.
        }
        
        // If we reach this point, then we must have disconnected.
        _bot.disconnect();
    }
    
    private PircBot _bot = null;
    private BufferedReader _breader = null;
    private BufferedWriter _bwriter = null;
    
    public static final int MAX_LINE_LENGTH = 510;
    
}