/* 
Copyright Paul James Mutton, 2001-2003, http://www.jibble.org/

This file is part of PircBot.

You may freely use PircBot to make your own IRC bots and clients.

Any software that makes use of the PircBot package must clearly
state that it makes use of the PircBot Java IRC Bot Framework (or
some similar wording) found at http://www.jibble.org/

This source code, or parts thereof, may not be distributed in any
modified form without permission from the author.

PircBot is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

package org.jibble.pircbot;

import java.net.*;
import java.io.*;

/**
 * This class is used to allow the bot to interact with a DCC Chat session.
 *
 * @since   0.9c
 * @author  Paul James Mutton,
 *          <a href="http://www.jibble.org/">http://www.jibble.org/</a>
 * @version 1.0.3 (Build time: Sat Jan 18 23:29:54 2003)
 */
public class DccChat {
    
    
    /**
     * This constructor is used when we are accepting a DCC CHAT request
     * from somebody. It attempts to connect to the client that issued the
     * request.
     * 
     * @param bot An instance of the underlying PircBot.
     * @param sourceNick The nick of the sender.
     * @param address The address to connect to.
     * @param port The port number to connect to.
     * 
     * @throws IOException If the connection cannot be made.
     */
    protected DccChat(PircBot bot, String sourceNick, String address, int port) throws IOException {
        _bot = bot;
        _sourceNick = sourceNick;
        _socket = new Socket(address, port);
        _reader = new BufferedReader(new InputStreamReader(_socket.getInputStream()));
        _writer = new BufferedWriter(new OutputStreamWriter(_socket.getOutputStream()));
    }
    
    
    /**
     * This constructor is used after we have issued a DCC CHAT request to
     * somebody. If the client accepts the chat request, then the socket we
     * obtain is passed to this constructor.
     * 
     * @param bot An instance of the underlying PircBot.
     * @param sourceNick The nick of the user we are sending the request to.
     * @param socket The socket which will be used for the DCC CHAT session.
     * 
     * @throws IOException If the socket cannot be read from.
     */
    protected DccChat(PircBot bot, String sourceNick, Socket socket) throws IOException {
        _bot = bot;
        _sourceNick = sourceNick;
        _socket = socket;
        _reader = new BufferedReader(new InputStreamReader(_socket.getInputStream()));
        _writer = new BufferedWriter(new OutputStreamWriter(_socket.getOutputStream()));        
    }
    
    
    /**
     * Reads the next line of text from the client at the other end of our DCC Chat
     * connection.  This method blocks until something can be returned.
     * If the connection has closed, null is returned.
     *
     * @return The next line of text from the client.  Returns null if the
     *          connection has closed normally.
     * 
     * @throws IOException If an I/O error occurs.
     */
    public String readLine() throws IOException {
        String line = _reader.readLine();
        if (line != null) {
            _bot.log("+++ DCC CHAT " + _sourceNick + " " + line);
        }
        return line;
    }
    
    
    /**
     * Sends a line of text to the client at the other end of our DCC Chat
     * connection.
     * 
     * @param line The line of text to be sent.  This should not include
     *             linefeed characters.
     * 
     * @throws IOException If an I/O error occurs.
     */
    public void sendLine(String line) throws IOException {
        // No need for synchronization here really...
        _bot.log("+++ DCC CHAT " + _sourceNick + " >>>" + line);
        _writer.write(line + "\r\n");
        _writer.flush();
    }
    
    
    /**
     * Closes the DCC Chat connection.
     * 
     * @throws IOException If an I/O error occurs.
     */
    public void close() throws IOException {
        _bot.log("+++ DCC CHAT with " + _sourceNick + " ended by us.");
        _socket.close();
    }
    
    private PircBot _bot;
    private String _sourceNick;
    private BufferedReader _reader;
    private BufferedWriter _writer;
    private Socket _socket;
    
}