/* 
 * PVM++ - A C++ library for easy use of PVM
 * Copyright (C) 1997-2001 Sebastian Wilhelmi; University of Karlsruhe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __PVM_HOST_HH__
#define __PVM_HOST_HH__

#include <pvm++/types.hh>

#include <string>

namespace Pvm
{

  /** Class representing a host in the PVM.
      
      This class represents a host, that is part of the PVM. Internally
      it just stores a pointer to a data structure, so it is save and
      fast to use it as a value parameter and as a function result as
      well. Requests to the PVM, like Name() are cached, so you
      don't need to cache yourself.
      
      There is an ostream& operator<<( ostream& Stream, Host What )
      defined as well. It outputs the host id as used by the PVM
      prepended by a "h".  */
  
  class Host
  {
  public:
    Host ();
    Host (const Host &What);
    /** constructs a Host for the host with the host id What as
	used by PVM. */
    Host (unsigned int What);
    
    /** returns the host id as used by PVM. */
    operator unsigned int () const;
    
    /** returns the name of the host. */
    std::string Name () const;
    
    /** returns the name of the architecture of this host. */
    std::string Arch () const;
    
    /** returns the speed of this host. It's the same as in PVM. */
    unsigned int Speed () const;
    
    /** returns, whether the host is up and running and still part of
	the PVM. */
    bool Running () const;
    
    /** starts Num instances of the task with name Task on the current
	host and returns the set of started tasks in Result. The PVM
	rules regarding the default search path apply. */
    void Spawn (const std::string &Task, int Num, TaskSet &Result) const;
    
    /** starts the task with name Task on the host and returns the
	corresponding instance of Task.*/
    Task Spawn (const std::string &Task) const;
    
    /** returns a list of all tasks, currently running on this host, in
	the reference parameter Result. */
    void Tasks (TaskSet &Result) const;
  private:
    HostEntry *Where;
    HostEntry *Get (unsigned int Id);
  };

} // namespace Pvm

#endif /* __PVM_HOST_HH__ */
