/* 
 * PVM++ - A C++ library for easy use of PVM
 * Copyright (C) 1997-2001 Sebastian Wilhelmi; University of Karlsruhe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __PVM_TASK_HH__
#define __PVM_TASK_HH__

#include <pvm++/types.hh>

#include <string>

namespace Pvm
{

  /** Class representing a task on the PVM.
      
      This class represents a task, running (or preveously running) on
      the PVM. Internally it just stores a pointer to a data structure,
      so it is save and fast to use it as a value parameter and as a
      function result as well. Requests to the PVM, like Name() are
      cached, so you don't need to cache yourself.
      
      There is an ostream& operator<< (ostream &Stream, Task What)
      defined as well. It outputs the task id as used by the PVM
      prepended by a "t".  */ 
  class Task
  {
  public:
    Task ();
    Task (const Task &What);
    
    /** constructs a Task for the host with the task id What as
	used by PVM. */
    Task (unsigned int What);
    
    /** returns the host id as used by PVM. */
    operator unsigned int () const;
    
    /** returns, whether the task has a parent. */
    bool HasParent () const;
    
    /** returns the parent of the task. It is an error to call this
	function, if no parent exists. Use HasParent() to check. */
    Task Parent () const;
    
    /** returns the host, the task is running on. */
    ::Pvm::Host Host () const;
    
    /** returns the name of the task. The name is only available for
	programs started by PVM++ via a Spawn() call or started by PVM
	using the pvm_spawn() call. This is a limitation inherited
	from PVM. */
    std::string Name () const;
    
    /** returns, whether the task is running. */
    bool Running () const;
    
    /** kills the tasks. */
    void Kill () const;
  private:
    friend class HostSet;
    TaskEntry *Where;
    static TaskEntry *Add (unsigned int TaskId);
    void Refresh () const;
  };
  
} // namespace Pvm

#endif /* __PVM_TASK_HH__ */
