/* 
 * PVM++ - A C++ library for easy use of PVM
 * Copyright (C) 1997-2001 Sebastian Wilhelmi; University of Karlsruhe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __PVM_HOSTSET_HH__
#define __PVM_HOSTSET_HH__

#include <pvm++/types.hh>

#include <set>
#include <string>

namespace Pvm
{

  /** A set of hosts in the PVM.
      
      This class represents a set of hosts in the PVM. As it is
      publically derived from the STL-set, all STL-features can be
      used. Here is a short example:
      
      \begin{verbatim}
      // get all hosts except the one, I'm on.
      Pvm::HostSet AllHosts;
      Pvm::Pvm ().Hosts (AllHosts);
      AllHosts.erase (Pvm::Pvm ().I ().Host ());
      // start one task on all those hosts
      Pvm::TaskSet AllTasks;
      Pvm::HostSet::iterator Current; 
      for (Current = AllHosts.begin (); Current != AllHosts.end (); ++Current)
        { 
          // save all started tasks to AllTasks.
          AllTasks.insert (Current->Spawn (PROGNAME)); 
        }
      \end{verbatim} 
  */
  class HostSet : public std::set< Host >
  {
  public:
    /** starts Num instances of the task with name Task on the hosts
	in the set and returns the set of started tasks in Result.
	The PVM rules regarding the default search path apply. The tasks
	are distributed evenly, taking into account the speed (as set by
	PVM) of the hosts. */
    void Spawn (const std::string &Task, int Num, TaskSet &Result) const;
    
    /** starts the task with name Task on one of the hosts in the set
	and returns the corresponding instance of Task. This function
	is not very useful actually, but only here because of
	consistency. */
    Task Spawn (const std::string &Task) const;
    
  };
  
} // namespace Pvm

#endif /* __PVM_HOSTSET_HH__ */
