/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef QWT_MARKER_H
#define QWT_MARKER_H

#include <qglobal.h>
#include "qwt_symbol.h"
#include "qwt_dimap.h"
#include <qpen.h>
#include <qfont.h>
#include <qrect.h>
#include <qstring.h>

/*!
  \brief A class for drawing markers

  A marker can be a horizontal line, a vertical line,
  a symbol, a label or any combination of them, which can
  be drawn around a center point inside a bounding rectangle.

  The setSymbol member assigns a symbol to the marker.
  The symbol is drawn at the specified point.

  With setLabel, a label can be assigned to the marker.
  The setAlignment member specifies where the label is
  drawn. All the Align*-constants in qwindefs.h (see Qt documentation)
  are valid. The interpretation of the alignment depends on the marker's
  line style. The alignment refers to the center point of
  the marker, which means, for example, that the label would be printed
  left above the center point if the alignment was set to AlignLeft|AlignTop.
*/
class QwtMarker
{
public:

    enum LineStyle {NoLine, HLine, VLine, Cross};
    
private:
    
    QString d_label;
    QPen d_pen;
    QPen d_tPen;
    QFont d_font;
    QwtSymbol d_sym;
    QwtDiMap d_xMap;
    QwtDiMap d_yMap;
    int d_align;
    LineStyle d_style;
    
public:

    QwtMarker();
    virtual ~QwtMarker();
    const QwtMarker & operator=(const QwtMarker&);
    
    void setLineStyle(LineStyle st);
    void setSymbol(const QwtSymbol &s);
    void setLabel(const char *txt);
    void setLabelAlignment(int align);
    void setLabelPen(const QPen &p);
    void setLinePen(const QPen &p);
    void setFont(const QFont &f);
    
    void draw(QPainter *p, int x, int y, const QRect &r);

    /*!
      \return the line style
      \sa For a description of line styles, see QwtMarker::setLineStyle
    */
    LineStyle lineStyle() const { return d_style; }
    /*!
      \return the symbol
      \sa QwtMarker::setSymbol, QwtSymbol
    */
    const QwtSymbol &symbol() const { return d_sym; }
    /*!
      \return the label's pen
      \sa QwtMarker::setLabelPen
    */
    const QPen &labelPen() const { return d_tPen; }
    /*!
      \return the line pen
      \sa QwtMarker::setLinePen
    */
    const QPen &linePen() const { return d_pen; }
    /*!
      \return the font used for the label
      \sa QwtMarker::setFont
    */
    const QFont &font() const { return d_font; }
    /*! 
      \return the label alignment
      \sa QwtMarker::setLabelAlignment
    */
    int labelAlignment() const { return d_align; }
    /*!
      \return the label
      \sa QwtMarker::setLabel
    */
    const QString& label() const { return d_label; }
    
protected:
    /*!
      \brief Notify a change of parameters
      
      This virtual function is called when a marker property
      has been changed and redrawing is advisable. It can be
      reimplemented by derived classes. The default implementation
      does nothing.
      \sa void QwtMarker::markerChanged()
    */
    virtual void markerChanged() {}
};


#endif
