/*=============================================================================
    Iterators

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_ITERATORS_HPP
#define SPIRIT_ITERATORS_HPP

///////////////////////////////////////////////////////////////////////////////

#include <iterator>

///////////////////////////////////////////////////////////////////////////////

namespace spirit {

//////////////////////////////////
namespace impl {

    template <typename IteratorTag> struct iterator_kind
    { /* At least a forward iterator is required */ };

    template <> struct iterator_kind<std::forward_iterator_tag>
    { typedef std::forward_iterator_tag T; };

    template <> struct iterator_kind<std::bidirectional_iterator_tag>
    { typedef std::forward_iterator_tag T; };

    template <> struct iterator_kind<std::random_access_iterator_tag>
    { typedef std::forward_iterator_tag T; };

} // namespace impl

///////////////////////////////////////////////////////////////////////////////
//
//  scanner class
//
//      The scanner wraps an iterator and a skip-parser. The skipper
//      handles the skipping of characters in between words that form
//      sentences in a language. The scanner extracts data from the
//      input stream and filters this data to exclude skip- characters
//      as recognized by the skipper. The scanner invokes the skipper
//      filter when tasked to scan the next character from the input.
//      The scanner conforms to a standard STL forward iterator.
//
//      The scanner is not a full-blown lexical analyzer. It does not
//      extract tokens such as reserved words and operators. Nor does it
//      extract numbers and literal strings
//
///////////////////////////////////////////////////////////////////////////////
template <typename IteratorT, typename SkipT>
class scanner {

public:

    typedef IteratorT   iterator_type;
    typedef SkipT       skip_type;

    typedef std::iterator_traits<IteratorT> iterator_traits;
    typedef typename impl::iterator_kind
        <typename iterator_traits::iterator_category>::T iterator_category;

    typedef typename iterator_traits::difference_type   difference_type;
    typedef typename iterator_traits::value_type        value_type;
    typedef typename iterator_traits::pointer           pointer;
    typedef typename iterator_traits::reference         reference;

                        scanner(iterator_type const& iter, SkipT const* skip);
    scanner&            operator=(iterator_type const& iter_);
    IteratorT           iterator() const;

    value_type          operator*() const;
    pointer             operator->() const;
    scanner&            operator++();
    scanner             operator++(int);

    bool                operator==(scanner const& other) const;
    bool                operator!=(scanner const& other) const;

private:

    void                next() const;
    IteratorT           get() const;

    SkipT const*        skip;
    mutable IteratorT   iter;
    mutable bool        dirty;
};

///////////////////////////////////////////////////////////////////////////////
//
//  nocase_iterator class
//
//      The nocase_iterator class adapts an iterator and converts
//      all upper-case characters from the input to lower-case. The
//      nocase_iterator conforms to a standard STL forward iterator.
//
///////////////////////////////////////////////////////////////////////////////
template <typename IteratorT>
class nocase_iterator {

public:

    typedef IteratorT   iterator_type;
    typedef std::iterator_traits<IteratorT> iterator_traits;
    typedef typename impl::iterator_kind
        <typename iterator_traits::iterator_category>::T iterator_category;

    typedef typename iterator_traits::difference_type   difference_type;
    typedef typename iterator_traits::value_type        value_type;
    typedef typename iterator_traits::pointer           pointer;
    typedef typename iterator_traits::reference         reference;

                        nocase_iterator(IteratorT const& iter_);

    template <typename IteratorTB>
    nocase_iterator&    operator=(nocase_iterator<IteratorTB> const& iter_);
    nocase_iterator&    operator=(IteratorT const& iter_);
    IteratorT           iterator() const;

    reference           operator*();
    pointer             operator->() const;
    nocase_iterator&    operator++();
    nocase_iterator     operator++(int);

    bool                operator==(nocase_iterator const& other) const;
    bool                operator!=(nocase_iterator const& other) const;

private:

    iterator_type       iter;
    value_type          next;
};

///////////////////////////////////////////////////////////////////////////////
//
//  strip_scanner class and specializations
//
//      Helper class to strip the scanner wrapper off an iterator.
//      This utility is used when we want to perform character
//      level parsing (see scanner above).
//
///////////////////////////////////////////////////////////////////////////////
namespace impl {

    template <typename T>
    struct strip_scanner {

        typedef T iterator_type;

        static T
        get(T ptr);
    };

    //////////////////////////////////
    template <typename IteratorT, typename SkipT>
    struct strip_scanner<scanner<IteratorT, SkipT> > {

        typedef IteratorT iterator_type;

        static IteratorT
        get(scanner<IteratorT, SkipT> const& scanner);
    };

    //////////////////////////////////
    template <typename IteratorT, typename SkipT>
    struct strip_scanner<nocase_iterator<scanner<IteratorT, SkipT> > > {

        typedef nocase_iterator<IteratorT> iterator_type;

        static iterator_type
        get(nocase_iterator<scanner<IteratorT, SkipT> > const& nocase_iter);
    };

} // namespace impl

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
