/*=============================================================================
   Character set

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_CHSET_HPP
#define SPIRIT_CHSET_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/parser.hpp"
#include "boost/spirit/primitives.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  chset class
//
//      Character set class. Matches a set of characters over a finite
//      range bounded by the limits of its template parameter CharT.
//      This class is an optimization of a parser that acts on on a set
//      of single characters. The template class is parametized by the
//      character type (CharT) and can work efficiently with 8, 16 and
//      32 bit characters.
//
//      chsets are constructed from literals (e.g. 'x'), chlits, ranges,
//      anychar and nothing (see primitives.hpp) or copy-constructed
//      from another chset. The chset class uses a copy-on-write scheme
//      that enables instances to be passed along easily by value.
//      Examples:
//
//          chset<>     s1('x');
//          chset<>     s2(anychar - s1);
//
//      Optionally, character sets may also be constructed using a
//      definition string following a syntax that resembles posix style
//      regular expression character sets, except that double quotes
//      delimit the set elements instead of square brackets and there is
//      no special negation '^' character.
//
//      Character set definition strings follow the meta-syntax:
//
//          range   = anychar >> '-' >> anychar;
//          set     = *(range | anychar);
//
//      Since we are defining the set using a string, the usual C/C++
//      literal string syntax rules apply. Examples:
//
//          chset<>     s1("a-zA-Z");       //  alphabetic characters
//          chset<>     s2("0-9a-fA-F");    //  hexadecimal characters
//          chset<>     s3("actgACTG");     //  DNA identifiers
//          chset<>     s4("\xff\xfe");     //  Hexadecimal 0xFF and 0xFE
//
//      The standard Spirit set operators apply (see operators.hpp) plus
//      an additional character-set-specific inverse (negation) operator:
//
//          ~a      //  Set inverse
//          a | b   //  Set union
//          a & b   //  Set intersection
//          a - b   //  Set difference
//          a ^ b   //  Set xor
//
//      where operands a and b are both chsets or one of the operand is
//      either a literal character, a chlit, a range, anychar or
//      nothing. Special optimized overloads are provided for anychar or
//      nothing operands. A nothing operand is converted to an empty
//      set, while an anychar operand is converted to a set having
//      elements of the full range of the character type used (e.g. 0-
//      255 for unsigned 8 bit chars).
//
//      A special case is ~anychar which yields nothing, but ~nothing
//      is illegal. Inversion of anychar is a one-way trip not unlike
//      converting a T* to a void*.
//
//      An assignment and a copy construtor are provided to allow mixed
//      conversion from a chset<A> to a chset<B>. This is possible when
//      type A is convertible to type B. For example if type A is a char
//      and type B is a wchar_t.
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT = char>
class chset: public char_parser<chset<CharT> > {

public:
                    chset();
                    chset(chset const& arg);
    explicit        chset(CharT arg);
    explicit        chset(anychar_ arg);
    explicit        chset(nothing_ arg);
    explicit        chset(chlit<CharT> const& arg);
    explicit        chset(range<CharT> const& arg);
                    ~chset();

    chset&          operator=(chset const& rhs);
    chset&          operator=(CharT rhs);
    chset&          operator=(anychar_ rhs);
    chset&          operator=(nothing_ rhs);
    chset&          operator=(chlit<CharT> const& rhs);
    chset&          operator=(range<CharT> const& rhs);

    void            set(range<CharT> const& arg);
    void            clear(range<CharT> const& arg);
    bool            test(CharT ch) const;

    template <typename CharTB>
    explicit
    chset(CharTB const* definition);

    template <typename CharTB>
    chset(chset<CharTB> const& arg);

    template <typename CharTB>
    chset&
    operator=(chset<CharTB> const& rhs);

    class rep;
    private: rep* ptr;
    friend class rep;
};

///////////////////////////////////////////////////////////////////////////////
//
//  chset free operators
//
//      Where a and b are both chsets, implements:
//
//          a | b, a & b, a - b, a ^ b
//
//      Where a is a chset, implements:
//
//          ~a
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
chset<CharT>
operator~(chset<CharT> const& a);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(chset<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(chset<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(chset<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(chset<CharT> const& a, chset<CharT> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  range <--> chset free operators
//
//      Where a is a chset and b is a range, and vice-versa, implements:
//
//          a | b, a & b, a - b, a ^ b
//
//      Where a is a range, implements:
//
//          ~a
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
chset<CharT>
operator~(range<CharT> const& a);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(chset<CharT> const& a, range<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(chset<CharT> const& a, range<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(chset<CharT> const& a, range<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(chset<CharT> const& a, range<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(range<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(range<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(range<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(range<CharT> const& a, chset<CharT> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  chlit <--> chset free operators
//
//      Where a is a chset and b is a chlit, and vice-versa, implements:
//
//          a | b, a & b, a - b, a ^ b
//
//      Where a is a chlit, implements:
//
//          ~a
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
chset<CharT>
operator~(chlit<CharT> const& a);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(chset<CharT> const& a, chlit<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(chset<CharT> const& a, chlit<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(chset<CharT> const& a, chlit<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(chset<CharT> const& a, chlit<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(chlit<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(chlit<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(chlit<CharT> const& a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(chlit<CharT> const& a, chset<CharT> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  literal primitives <--> chset free operators
//
//      Where a is a chset and b is a literal primitive,
//      and vice-versa, implements:
//
//          a | b, a & b, a - b, a ^ b
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(chset<CharT> const& a, CharT b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(chset<CharT> const& a, CharT b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(chset<CharT> const& a, CharT b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(chset<CharT> const& a, CharT b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(CharT a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(CharT a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(CharT a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(CharT a, chset<CharT> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  anychar_ <--> chset free operators
//
//      Where a is chset and b is a anychar_, and vice-versa, implements:
//
//          a | b, a & b, a - b, a ^ b
//
//      Where a is anychar, implements:
//
//          ~a
//
///////////////////////////////////////////////////////////////////////////////
nothing_
operator~(anychar_ a);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(chset<CharT> const& a, anychar_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(chset<CharT> const& a, anychar_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(chset<CharT> const& a, anychar_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(chset<CharT> const& a, anychar_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(anychar_ a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(anychar_ a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(anychar_ a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(anychar_ a, chset<CharT> const& b);

///////////////////////////////////////////////////////////////////////////////
//
//  nothing_ <--> chset free operators
//
//      Where a is chset and b is nothing_, and vice-versa, implements:
//
//          a | b, a & b, a - b, a ^ b
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(chset<CharT> const& a, nothing_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(chset<CharT> const& a, nothing_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(chset<CharT> const& a, nothing_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(chset<CharT> const& a, nothing_ b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator|(nothing_ a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator&(nothing_ a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator-(nothing_ a, chset<CharT> const& b);

//////////////////////////////////
template <typename CharT>
chset<CharT>
operator^(nothing_ a, chset<CharT> const& b);

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
