/*=============================================================================
    Parser directives

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

======================================================================================*/
#ifndef SPIRIT_DIRECTIVES_IPP
#define SPIRIT_DIRECTIVES_IPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/directives.hpp"
#include "boost/spirit/operators.hpp"
#include "boost/spirit/iterators.hpp"
#include "boost/spirit/primitives.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  contiguous class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename S>
inline contiguous<S>::contiguous(S const& a)
:   unary<S>(a) {}

//////////////////////////////////
template <typename S>
template <typename IteratorT>
inline match
contiguous<S>::parse(IteratorT& first, IteratorT const& last) const
{
    typedef typename impl::strip_scanner<IteratorT>
        ::iterator_type plain_iter;

    plain_iter  i1 = impl::strip_scanner<IteratorT>::get(first);
    plain_iter  i2 = impl::strip_scanner<IteratorT>::get(last);
    match       hit = this->subject().parse(i1, i2);

    if (hit)
        first = i1;
    return hit;
}

///////////////////////////////////////////////////////////////////////////////
//
//  inhibit_case class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename S>
inline inhibit_case<S>::inhibit_case(S const& a)
:   unary<S>(a) {}

//////////////////////////////////
template <typename S>
template <typename IteratorT>
inline match
inhibit_case<S>::parse(IteratorT& first, IteratorT const& last) const
{
    nocase_iterator<IteratorT>  i1(first);
    nocase_iterator<IteratorT>  i2(last);
    match hit = this->subject().parse(i1, i2);

    if (hit)
        first = i1.iterator();
    return hit;
}

//////////////////////////////////
inline inhibit_case<chlit<char> >
inhibit_case_parser_gen::operator[](char ch) const
{
    return inhibit_case<chlit<char> >(chlit<char>(ch));
}

//////////////////////////////////
inline inhibit_case<chlit<wchar_t> >
inhibit_case_parser_gen::operator[](wchar_t ch) const
{
    return inhibit_case<chlit<wchar_t> >(chlit<wchar_t>(ch));
}

//////////////////////////////////
inline inhibit_case<strlit<cstring<char> > >
inhibit_case_parser_gen::operator[](char const* str) const
{
    return inhibit_case<strlit<cstring<char> > >
        (strlit<cstring<char> >(str));
}

//////////////////////////////////
inline inhibit_case<strlit<cstring<wchar_t> > >
inhibit_case_parser_gen::operator[](wchar_t const* str) const
{
    return inhibit_case<strlit<cstring<wchar_t> > >
        (strlit<cstring<wchar_t> >(str));
}

///////////////////////////////////////////////////////////////////////////////
//
//  longest_alternative class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
inline longest_alternative<A, B>::longest_alternative(A const& a, B const& b)
:   binary<A, B>(a, b) {}

//////////////////////////////////
template <typename A, typename B>
template <typename IteratorT>
inline match
longest_alternative<A, B>::parse(IteratorT& first, IteratorT const& last) const
{
    IteratorT   ls = first;
    IteratorT   rs = first;
    match       l = this->left().parse(ls, last);
    match       r = this->right().parse(rs, last);

    if (l || r)
    {
        bool less = l.length() < r.length();
        first = less ? rs : ls;
        return less ? r : l;
    }

    return match();
}

//////////////////////////////////
namespace impl {

    //////////////////////////////////
    template <typename T>
    struct to_longest_alternative {

        typedef T return_t;

        static return_t const&
        convert(T const& a)  //  Special (end) case
        {
            return a;
        }
    };

    //////////////////////////////////
    template <typename A, typename B>
    struct to_longest_alternative<alternative<A, B> > {

        typedef typename to_longest_alternative<A>::return_t    a_t;
        typedef typename to_longest_alternative<B>::return_t    b_t;
        typedef longest_alternative<a_t, b_t>                   return_t;

        static return_t
        convert(alternative<A, B> const& alt) // Recursive case
        {
            return return_t(
                to_longest_alternative<A>::convert(alt.left()),
                to_longest_alternative<B>::convert(alt.right()));
        }
    };

} // namespace impl;

///////////////////////////////////////////////////////////////////////////////
//
//  shortest_alternative class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
inline shortest_alternative<A, B>::shortest_alternative(A const& a, B const& b)
:   binary<A, B>(a, b) {}

//////////////////////////////////
template <typename A, typename B>
template <typename IteratorT>
inline match
shortest_alternative<A, B>::parse(IteratorT& first, IteratorT const& last) const
{
    IteratorT   ls = first;
    IteratorT   rs = first;
    match       l = this->left().parse(ls, last);
    match       r = this->right().parse(rs, last);

    if (l || r)
    {
        bool less = l.length() < r.length();
        first = less ? ls : rs;
        return less ? l : r;
    }

    return match();
}

//////////////////////////////////
namespace impl {

    //////////////////////////////////
    template <typename T>
    struct to_shortest_alternative {

        typedef T return_t;

        static return_t const&
        convert(T const& a) //  Special (end) case
        {
            return a;
        }
    };

    //////////////////////////////////
    template <typename A, typename B>
    struct to_shortest_alternative<alternative<A, B> > {

        typedef typename to_shortest_alternative<A>::return_t   a_t;
        typedef typename to_shortest_alternative<B>::return_t   b_t;
        typedef shortest_alternative<a_t, b_t>                  return_t;

        static return_t
        convert(alternative<A, B> const& alt) //  Recursive case
        {
            return return_t(
                to_shortest_alternative<A>::convert(alt.left()),
                to_shortest_alternative<B>::convert(alt.right()));
        }
    };

} // namespace impl

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
