/*=============================================================================
    Parser Loops

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_LOOPS_HPP
#define SPIRIT_LOOPS_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/parser.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  fixed_loop class
//
//      This class takes care of the constructs:
//
//          p.repeat(exact)
//          p(exact)
//
//      where p is a parser and exact is the number of times to repeat.
//      The parser iterates over the input exactly 'exact' times. The
//      parse function fails if the parser does not match the input
//      exactly 'exact' times.
//
//      This class is parametizable and can accept constant arguments
//      (e.g. p.repeat(5)) as well as references to variables (e.g.
//      p.repeat(ref(n))).
//
///////////////////////////////////////////////////////////////////////////////
template <typename S, typename ExactT>
class fixed_loop
:   public unary<S>,
    public parser<fixed_loop<S, ExactT> > {

public:

    fixed_loop(S const& subject, ExactT const& exact);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    ExactT    exact;
};

///////////////////////////////////////////////////////////////////////////////
//
//  finite_loop class
//
//      This class takes care of the constructs:
//
//          p.repeat(min, max)
//          p(min, max)
//
//      where p is a parser, min and max specifies the minimum and maximum
//      iterations over p. The parser iterates over the input at least
//      'min' times and at most 'max' times. The parse function fails if
//      the parser does not match the input at least 'min' times and at
//      most 'max' times.
//
//      This class is parametizable and can accept constant arguments
//      (e.g. p.repeat(5, 10)) as well as references to variables (e.g.
//      p.repeat(ref(n1), ref(n2))).
//
///////////////////////////////////////////////////////////////////////////////
template <typename S, typename MinT, typename MaxT>
class finite_loop
:   public unary<S>,
    public parser<finite_loop<S, MinT, MaxT> > {

public:

    finite_loop(S const& a, MinT const& min, MaxT const& max);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    MinT    min;
    MaxT    max;
};

///////////////////////////////////////////////////////////////////////////////
//
//  infinite_loop class
//
//      This class takes care of the constructs:
//
//          p.repeat(min, more)
//          p(min, more)
//
//      where p is a parser, min is the minimum iteration over p and more
//      specifies the iteration to proceed indefinitely. The parser iterates
//      over the input at least 'min' times and continues indefinitely until
//      p fails. The parse function fails if the parser does not match the
//      input at least 'min' times.
//
//      This class is parametizable and can accept constant arguments
//      (e.g. p.repeat(5, more)) as well as references to variables (e.g.
//      p.repeat(ref(n), more)).
//
///////////////////////////////////////////////////////////////////////////////
template <typename S, typename MinT>
class infinite_loop
:   public unary<S>,
    public parser<infinite_loop<S, MinT> > {

public:

    infinite_loop(S const& subject, MinT const& min, more_t);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    MinT    min;
};

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif

