/*=============================================================================
    Parser primitives

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_PRIMITIVES_HPP
#define SPIRIT_PRIMITIVES_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/parser.hpp"
#include "boost/spirit/composite.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  char_action class
//
//      Links a character parser with a user defined semantic action.
//      The semantic action may be a function or a functor. A function
//      should be compatible with the interface:
//
//          void f(CharT ch);
//
//      A functor should have a member operator() with a compatible signature
//      as above. The matching character is passed into the function/functor.
//      This is the default class that character parsers use when dealing with
//      the construct:
//
//          p[f]
//
//      where p is a parser and f is a function or functor.
//
///////////////////////////////////////////////////////////////////////////////
template <typename ParserT, typename ActionT>
class char_action
:   public unary<ParserT>,
    public parser<char_action<ParserT, ActionT> > {

public:

    char_action(ParserT const& subject, ActionT const& actor_);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    ActionT  actor;
};

///////////////////////////////////////////////////////////////////////////////
//
//  char_parser class
//
///////////////////////////////////////////////////////////////////////////////
template <typename DerivedT>
struct char_parser : public parser<DerivedT> {

    template <typename ActionT>
    char_action<DerivedT, ActionT>
    operator[](ActionT const& actor) const;

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;
};

///////////////////////////////////////////////////////////////////////////////
//
//  chlit class
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT = char>
struct chlit : public char_parser<chlit<CharT> > {

    chlit(CharT ch);

    template <typename T>
    bool
    test(T ch) const;

    CharT  ch;
};

///////////////////////////////////////////////////////////////////////////////
//
//  range class
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharAT = char, typename CharBT = CharAT>
struct range : public char_parser<range<CharAT, CharBT> > {

    range(CharAT first, CharBT last);

    template <typename T>
    bool
    test(T ch) const;

    CharAT  first;
    CharBT  last;
};

///////////////////////////////////////////////////////////////////////////////
//
//  cstring class
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT = char>
class cstring {

public:

    typedef CharT const* const_iterator;

    cstring(const_iterator str, unsigned len);
    cstring(const_iterator str);

    const_iterator      begin() const;
    const_iterator      end() const;
    std::size_t         length() const;

private:

    const_iterator first;
    const_iterator last;
};

///////////////////////////////////////////////////////////////////////////////
//
//  user_string class
//
///////////////////////////////////////////////////////////////////////////////
template <typename StringRefT>
class user_string {

public:

    typedef typename StringRefT::element_type::const_iterator const_iterator;
    typedef typename StringRefT::element_type element_type;

    user_string(StringRefT const& ref);

    const_iterator      begin() const;
    const_iterator      end() const;
    std::size_t         length() const;

private:

    StringRefT    ref;
};

///////////////////////////////////////////////////////////////////////////////
template <typename T>   struct remove_ref           { typedef T type; };
template <typename T>   struct remove_ref<T&>       { typedef T type; };
template <typename T>   struct remove_ref<T const&> { typedef T const type; };

///////////////////////////////////////////////////////////////////////////////
//
//  strlit class
//
///////////////////////////////////////////////////////////////////////////////
template <typename StringT = cstring<char> >
struct strlit : public parser<strlit<StringT> > {

    typedef typename remove_ref<StringT>::type      string_type;
    typedef typename string_type::const_iterator    raw_string_type;

    strlit(StringT str);
    strlit(raw_string_type str);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

    StringT str;
};

///////////////////////////////////////////////////////////////////////////////
//
//  Predefined parser primitives
//
///////////////////////////////////////////////////////////////////////////////
struct anychar_ : public char_parser<anychar_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

///////////////////////////////////////////////////////////////////////////////
struct nothing_ : public parser<nothing_> {

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;
};

///////////////////////////////////////////////////////////////////////////////
struct epsilon_ : public parser<epsilon_> {

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;
};

///////////////////////////////////////////////////////////////////////////////
struct alnum_ : public char_parser<alnum_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

struct alpha_ : public char_parser<alpha_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct cntrl_ : public char_parser<cntrl_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct digit_ : public char_parser<digit_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct graph_ : public char_parser<graph_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct lower_ : public char_parser<lower_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct print_ : public char_parser<print_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct punct_ : public char_parser<punct_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct space_ : public char_parser<space_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct upper_ : public char_parser<upper_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

//////////////////////////////////
struct xdigit_ : public char_parser<xdigit_> {

    template <typename CharT>
    bool
    test(CharT ch) const;
};

///////////////////////////////////////////////////////////////////////////////
//
//  Generator functions
//
///////////////////////////////////////////////////////////////////////////////

//  Forward declarations
template <typename T> class reference_wrapper;
template <int N, typename ContextT> class context_wrapper;

//////////////////////////////////
template <typename ParamT>
chlit<ParamT>
ch_p(ParamT param);

//////////////////////////////////
template <typename ParamAT, typename ParamBT>
range<ParamAT, ParamBT>
range_p(ParamAT first, ParamBT last);

//////////////////////////////////
template <typename CharT>
strlit<cstring<CharT> >
str_p(CharT const* str);

//////////////////////////////////
template <typename StringT>
strlit<StringT&>
str_p(reference_wrapper<StringT> const& ref);

//////////////////////////////////
template <int N, typename ContextT>
strlit<user_string<context_wrapper<N, ContextT> > >
str_p(context_wrapper<N, ContextT> const& ref);

///////////////////////////////////////////////////////////////////////////////
//
//  Generator objects
//
///////////////////////////////////////////////////////////////////////////////
const anychar_  anychar = anychar_();
const nothing_  nothing = nothing_();
const epsilon_  epsilon = epsilon_();

const alnum_    alnum   = alnum_();
const alpha_    alpha   = alpha_();
const cntrl_    cntrl   = cntrl_();
const digit_    digit   = digit_();
const graph_    graph   = graph_();
const lower_    lower   = lower_();
const print_    print   = print_();
const punct_    punct   = punct_();
const space_    space   = space_();
const upper_    upper   = upper_();
const xdigit_   xdigit  = xdigit_();

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
